/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.file;

import static com.google.common.collect.Lists.transform;
import static org.kuali.common.util.base.Exceptions.illegalState;
import static org.kuali.common.util.base.Precondition.checkIsDir;
import static org.kuali.common.util.base.Precondition.checkNotBlank;
import static org.kuali.common.util.base.Precondition.checkNotNull;

import java.io.File;
import java.io.IOException;
import java.util.List;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;

/**
 * @deprecated
 */
@Deprecated
public final class Files {

	public static long sumFileLengths(Iterable<File> files) {
		long sum = 0;
		for (File file : files) {
			sum += file.length();
		}
		return sum;
	}

	public static List<File> canonicalList(File dir) {
		return transform(listFiles(checkIsDir(dir, "dir")), canonical());
	}

	public static File currentWorkingDirectory() {
		return getCanonicalFile(".");
	}

	public static Function<File, File> canonical() {
		return FileFunction.CANONICAL;
	}

	public static Function<File, Long> length() {
		return LongFunction.LENGTH;
	}

	public static Function<File, Long> lastModified() {
		return LongFunction.LAST_MODIFIED;
	}

	public static File getCanonicalFile(File file) {
		return canonical().apply(file);
	}

	public static File getCanonicalFile(String path) {
		return getCanonicalFile(new File(checkNotBlank(path, "path")));
	}

	public static File getCanonicalFile(File parent, String child) {
		return getCanonicalFile(new File(checkNotNull(parent, "parent"), checkNotBlank(child, "child")));
	}

	/**
	 * Return an immutable list of files from <code>dir</code>, sorted naturally
	 * 
	 * @throws IllegalArgumentException
	 *             if dir is null, does not exist, or exists but is not a directory
	 */
	public static List<File> listFiles(File dir) {
		return listFiles(dir, Ordering.<File> natural());
	}

	/**
	 * Return an immutable list of files from <code>dir</code>, sorted by size
	 * 
	 * @throws IllegalArgumentException
	 *             if dir is null, does not exist, or exists but is not a directory
	 */
	public static List<File> listFilesBySize(File dir) {
		return listFiles(dir, Ordering.natural().onResultOf(length()));
	}

	/**
	 * Return an immutable list of files from <code>dir</code>, sorted by last modified timestamp
	 * 
	 * @throws IllegalArgumentException
	 *             if dir is null, does not exist, or exists but is not a directory
	 */
	public static List<File> listFilesByLastModified(File dir) {
		return listFiles(dir, Ordering.natural().onResultOf(lastModified()));
	}

	/**
	 * Return an immutable list of files from <code>dir</code>, sorted by <code>ordering</code>
	 * 
	 * @throws IllegalArgumentException
	 *             if dir is null, does not exist, or exists but is not a directory
	 */
	public static List<File> listFiles(File dir, Ordering<File> ordering) {
		checkIsDir(dir, "dir");
		return ordering.immutableSortedCopy(ImmutableList.copyOf(dir.listFiles()));
	}

	private enum FileFunction implements Function<File, File> {
		CANONICAL {
			@Override
			public File apply(File file) {
				try {
					return file.getCanonicalFile();
				} catch (IOException e) {
					throw illegalState(e);
				}
			}
		},
		PARENT {
			@Override
			public File apply(File file) {
				return file.getParentFile();
			}
		},
		ABSOLUTE {
			@Override
			public File apply(File file) {
				return file.getAbsoluteFile();
			}
		}
	}

	private enum LongFunction implements Function<File, Long> {
		LENGTH {
			@Override
			public Long apply(File file) {
				return file.length();
			}
		},
		LAST_MODIFIED {
			@Override
			public Long apply(File file) {
				return file.lastModified();
			}
		},
		TOTAL_SPACE {
			@Override
			public Long apply(File file) {
				return file.getTotalSpace();
			}
		},
		USABLE_SPACE {
			@Override
			public Long apply(File file) {
				return file.getUsableSpace();
			}
		},
		FREE_SPACE {
			@Override
			public Long apply(File file) {
				return file.getFreeSpace();
			}
		}
	}

}
