/**
 * Copyright 2010-2014 The Kuali Foundation
 *
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.opensource.org/licenses/ecl2.php
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.common.util.xml.jaxb.adapter;

import javax.xml.bind.annotation.adapters.XmlAdapter;

import org.kuali.common.util.Assert;
import org.kuali.common.util.Str;

import com.google.common.base.Optional;

public final class FlattenOptionalStringAdapter extends XmlAdapter<String, Optional<String>> {

	public FlattenOptionalStringAdapter() {
		this(FlattenConstants.DEFAULT_CR_REPLACEMENT, FlattenConstants.DEFAULT_LF_REPLACEMENT);
	}

	public FlattenOptionalStringAdapter(String carriageReturnReplacement, String linefeedReplacement) {
		// No blanks because this needs to work in both directions (flatten + inflate)
		Assert.noBlanks(carriageReturnReplacement, linefeedReplacement);
		this.carriageReturnReplacement = carriageReturnReplacement;
		this.linefeedReplacement = linefeedReplacement;
	}

	private final String carriageReturnReplacement;
	private final String linefeedReplacement;

	@Override
	public String marshal(Optional<String> optional) {
		if (optional.isPresent()) {
			return Str.flatten(optional.get(), carriageReturnReplacement, linefeedReplacement);
		} else {
			return null;
		}
	}

	@Override
	public Optional<String> unmarshal(String value) {
		if (value == null) {
			return Optional.<String> absent();
		} else {
			return Optional.<String> of(Str.inflate(value, carriageReturnReplacement, linefeedReplacement));
		}
	}

	public String getCarriageReturnReplacement() {
		return carriageReturnReplacement;
	}

	public String getLinefeedReplacement() {
		return linefeedReplacement;
	}

}
