/*
 * Licensed under MIT (https://github.com/ligoj/ligoj/blob/master/LICENSE)
 */
define(function () {
	const current = {

		/**
		 * Flag objects
		 */
		table: null,
		search: false,
		containerType: 'group',

		/**
		 * Edited users' identifier
		 */
		currentId: 0,

		initialize: function (parameters) {
			current.onHashChange(parameters);
		},

		/**
		 * Manage internal navigation.
		 */
		onHashChange: function () {
			// Search mode
			current.currentId = null;
			current.initializeSearch();
			$(function() {
				_('search').trigger('focus');
			});
		},

		/**
		 * Initialize the search UI components
		 */
		initializeSearch: function () {
			if (current.search) {
				return;
			}
			current.search = true;
			_('confirmCreate').click(current.save);

			// User edition pop-up
			_('popup').on('shown.bs.modal', function () {
				_('name').trigger('focus');
			}).on('show.bs.modal', function (event) {
				validationManager.reset($(this));
				const $source = $(event.relatedTarget);
				const $tr = $source.closest('tr');
				let uc = ($tr.length && current.table.fnGetData($tr[0])) || undefined;
				uc = uc?.id ? uc : {};
				current.currentId = uc.id;
				_('modal-title').html(Handlebars.compile(current.$messages['container-scope-title'])(current.$messages[current.containerType]));
				_('name').val(uc.name || null);
				_('dn').val(uc.dn || null);
			});
			_('type').on('change', 'input[type="radio"]', function () {
				current.containerType = $(this).attr('id');
				current.table.api().ajax.reload();
			});

			// Global datatables filter
			_('search').on('keyup', function () {
                if (current.table?.fnSettings()?.oPreviousSearch?.sSearch || '' !== $(this).val()) {
				    current.table?.fnFilter($(this).val());
				}
            });

			// Also initialize the datatables component
			current.initializeDataTable();
		},

		/**
		 * Initialize the users datatables (server AJAX)
		 */
		initializeDataTable: function () {
			current.table = _('table').dataTable({
				dom: 'rt<"row"<"col-xs-6"i><"col-xs-6"p>>',
				serverSide: true,
				searching: true,
				ajax: function () {
					return REST_PATH + 'service/id/container-scope/' + current.containerType;
				},
				createdRow: function (nRow) {
					$(nRow).find('.delete').on('click', current.deleteEntry);
				},
				columns: [{
					data: 'name',
					width: '150px'
				}, {
					data: 'dn',
					className: 'truncate'
				}, {
					data: null,
					width: '48px',
					orderable: false,
					render: function (_i, _j, data) {
						if (!data.locked) {
							const editLink = `<a class="update" data-toggle="modal" data-target="#popup"><i class="fas fa-pencil-alt" data-toggle="tooltip" title="${current.$messages.update}"></i></a>`;
							return editLink + `<a class="delete"><i class="fas fa-trash-alt" data-toggle="tooltip" title="${current.$messages.delete}"></i></a>`;
						}
						return '&nbsp;';
					}
				}]
			});
		},

		formToObject: function () {
			return {
				id: current.currentId,
				name: _('name').val(),
				dn: _('dn').val(),
				type: current.containerType
			};
		},

		save: function () {
			const data = current.formToObject();
			$.ajax({
				type: current.currentId ? 'PUT' : 'POST',
				url: REST_PATH + 'service/id/container-scope',
				dataType: 'json',
				contentType: 'application/json',
				data: JSON.stringify(data),
				success: function (data) {
					notifyManager.notify(Handlebars.compile(
						current.$messages[current.currentId ? 'updated' : 'created'])(current.currentId || data));
					current.table?.api().ajax.reload();
					_('popup').modal('hide');
				}
			});
		},

		/**
		 * Delete the selected entry after popup confirmation, or directly from its identifier.
		 */
		deleteEntry: function (id, text) {
			if ((typeof id) === 'number') {
				// Delete without confirmation
				$.ajax({
					type: 'DELETE',
					url: REST_PATH + 'service/id/container-scope/' + id,
					success: function () {
						notifyManager.notify(Handlebars.compile(current.$messages.deleted)(text + '(' + id + ')'));
						current.table?.api().ajax.reload();
					}
				});
			} else {
				// Requires a confirmation
				const entity = current.table.fnGetData($(this).closest('tr')[0]);
				const display = entity.name + '/' + entity.type;
				bootbox.confirmDelete(function (confirmed) {
					confirmed && current.deleteEntry(entity.id, display);
				}, text);
			}
		}
	};
	return current;
});
