/*
 * Decompiled with CFR 0.152.
 */
package liquibase.sdk.github;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.DateFormat;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.kohsuke.github.GHArtifact;
import org.kohsuke.github.GHCommitState;
import org.kohsuke.github.GHFileNotFoundException;
import org.kohsuke.github.GHIssueState;
import org.kohsuke.github.GHPullRequest;
import org.kohsuke.github.GHRelease;
import org.kohsuke.github.GHRepository;
import org.kohsuke.github.GHWorkflow;
import org.kohsuke.github.GHWorkflowRun;
import org.kohsuke.github.GHWorkflowRunQueryBuilder;
import org.kohsuke.github.GitHub;
import org.kohsuke.github.PagedIterator;
import org.slf4j.Logger;

public class GitHubClient {
    private final GitHub github;
    private final Logger log;
    private final String githubToken;
    private static final String CORE_REPOSITORY = "liquibase";
    private static final String CORE_ARTIFACT = "liquibase-core";
    private static final String PRO_REPOSITORY = "liquibase-pro";
    private static final String PRO_ARTIFACT = "liquibase-commercial";

    GitHubClient(GitHub github, Logger log) {
        this.github = github;
        this.log = log;
        this.githubToken = null;
    }

    public GitHubClient(String githubToken, Logger log) throws IOException {
        this.log = log;
        this.githubToken = githubToken;
        if (StringUtils.trimToNull((String)githubToken) == null) {
            throw new IOException("Missing github token\nYour github token is not set in liquibase.sdk.github.token.\n\nIt can be set via any Maven property-setting mechanism, but the best is to add the following to your " + SystemUtils.getUserHome() + "/.m2/settings.xml file in the <profiles></profiles> section:\n\t<profile>\n\t\t<id>liquibase-sdk</id>\n\t\t<activation>\n\t\t\t<activeByDefault>true</activeByDefault>\n\t\t</activation>\n\t\t<properties>\n\t\t\t<liquibase.sdk.github.token>YOUR_TOKEN</liquibase.sdk.github.token>\n\t\t</properties>\n\t</profile>\n\nIf you do not have a GitHub personal access token, you can create one at https://github.com/settings/tokens. It needs to be assigned the 'repo' scope");
        }
        this.github = GitHub.connectUsingOAuth((String)githubToken);
        if (!this.github.isCredentialValid()) {
            throw new IOException("Invalid github credentials. Check your liquibase.sdk.token property");
        }
        log.debug("Successfully connected to github");
    }

    public GHRelease getRelease(String repo, String tagName) throws IOException {
        GHRepository repository = this.getRepository(repo);
        this.log.debug("Successfully found repository " + repository.getHtmlUrl());
        return repository.getReleaseByTagName(tagName);
    }

    public String findMatchingBranch(String repo, String ... branches) throws IOException {
        if (branches.length == 1 && branches[0].contains(",")) {
            branches = branches[0].split("\\s*,\\s*");
        }
        GHRepository repository = this.getRepository(repo);
        this.log.debug("Successfully found repository " + repository.getHtmlUrl());
        Map<String, GHPullRequest> pullRequests = this.getAllOpenPullRequests(repository);
        for (String branch : branches) {
            branch = GitHubClient.simplifyBranch(branch);
            LinkedHashSet<String> branchVariations = new LinkedHashSet<String>();
            branchVariations.add(branch);
            if (branch.contains(":") && (branch.endsWith(":master") || !branch.endsWith(":main"))) {
                branchVariations.add(branch.replace(":", "-"));
                branchVariations.add(branch.replace(".+:", ""));
            }
            for (String branchVariation : branchVariations) {
                if (this.useLocalBranch(branchVariation)) {
                    try {
                        return repository.getOwnerName() + ":" + repository.getBranch(branchVariation).getName();
                    }
                    catch (GHFileNotFoundException e) {
                        this.log.info("No branch '" + branchVariation + "' in " + repository.getHtmlUrl());
                        continue;
                    }
                }
                GHPullRequest pr = pullRequests.get(branchVariation);
                if (pr == null) {
                    for (GHPullRequest otherPr : pullRequests.values()) {
                        String otherBranchName = otherPr.getHead().getRef();
                        String otherLabel = otherPr.getHead().getLabel();
                        if (!otherBranchName.equals(branchVariation) && !otherLabel.replace(":", "-").equals(branchVariation)) continue;
                        return otherPr.getHead().getLabel();
                    }
                    this.log.info("No PR for branch '" + branchVariation + "' in " + repository.getHtmlUrl());
                    continue;
                }
                return pr.getHead().getLabel();
            }
        }
        return null;
    }

    public static String simplifyBranch(String branch) {
        if (branch == null) {
            return null;
        }
        return branch.replace("refs/heads/", "").replace("refs/heads/tags", "");
    }

    private GHRepository getRepository(String repo) throws IOException {
        if (!repo.contains("/")) {
            repo = "liquibase/" + repo;
        }
        return this.github.getRepository(repo);
    }

    protected Map<String, GHPullRequest> getAllOpenPullRequests(GHRepository repository) throws IOException {
        HashMap<String, GHPullRequest> pullRequests = new HashMap<String, GHPullRequest>();
        repository.queryPullRequests().state(GHIssueState.OPEN).list().withPageSize(50).toList().forEach(pr -> pullRequests.put(pr.getHead().getLabel(), (GHPullRequest)pr));
        return pullRequests;
    }

    private boolean useLocalBranch(String branchVariation) {
        return branchVariation.equals("master") || branchVariation.equals("main");
    }

    public GHWorkflowRun findLastBuild(String repo, BuildFilter buildFilter, String workflowId) throws IOException {
        GHRepository repository = this.getRepository(repo);
        this.log.debug("Successfully found repository " + repository.getHtmlUrl());
        GHWorkflow workflow = repository.getWorkflow(workflowId);
        this.log.debug("Successfully found workflow " + workflow.getHtmlUrl());
        this.log.debug("Workflow state: " + workflow.getState());
        return this.findRun(repository, workflow, buildFilter, true, null);
    }

    public GHWorkflowRun findBuild(String repo, long runId) throws IOException {
        GHRepository repository = this.getRepository(repo);
        this.log.debug("Successfully found repository " + repository.getHtmlUrl());
        GHWorkflowRun workflowRun = repository.getWorkflowRun(runId);
        this.log.debug("Successfully found run " + runId);
        return workflowRun;
    }

    private GHWorkflowRun findRun(GHRepository repository, GHWorkflow workflow, BuildFilter buildFilter, boolean recentOnly, GHWorkflowRun foundFailedRun) throws IOException {
        this.log.debug("Fetching recent workflow runs " + (recentOnly ? "(recent)" : "") + ".... ");
        GHWorkflowRunQueryBuilder queryBuilder = repository.queryWorkflowRuns();
        if (!recentOnly) {
            queryBuilder = queryBuilder.branch(buildFilter.getBranch());
        }
        PagedIterator runIterator = queryBuilder.list().withPageSize(25).iterator();
        this.log.debug("Fetching workflow runs....COMPLETE");
        this.log.debug("Finding most recent successful run...");
        GHWorkflowRun runToDownload = null;
        int page = 0;
        while (runIterator.hasNext()) {
            if (page++ > 1 && recentOnly) {
                return this.findRun(repository, workflow, buildFilter, false, foundFailedRun);
            }
            runToDownload = (GHWorkflowRun)runIterator.next();
            if (foundFailedRun != null && foundFailedRun.getId() == runToDownload.getId() || runToDownload.getWorkflowId() != workflow.getId() || !runToDownload.getHeadBranch().equals(buildFilter.getBranch())) continue;
            if (!runToDownload.getHeadRepository().getOwnerName().equals(buildFilter.fork)) {
                this.log.info("Skipping " + buildFilter.getBranch() + " from " + runToDownload.getHeadRepository().getOwnerName() + " because it's not from " + buildFilter.fork + "'s fork " + runToDownload.getHtmlUrl());
                continue;
            }
            if (runToDownload.getStatus() != GHWorkflowRun.Status.COMPLETED) {
                this.log.info("Skipping " + runToDownload.getStatus() + " build #" + runToDownload.getRunNumber() + " from " + DateFormat.getDateTimeInstance().format(runToDownload.getCreatedAt()) + " " + runToDownload.getHtmlUrl());
                continue;
            }
            if (runToDownload.getConclusion() == GHWorkflowRun.Conclusion.SUCCESS) {
                return runToDownload;
            }
            if (buildFilter.skipFailedBuilds) {
                this.log.debug("Found run " + runToDownload.getName() + ": " + runToDownload.getStatus() + " -- " + runToDownload.getConclusion() + "  build #" + runToDownload.getRunNumber() + " " + runToDownload.getHtmlUrl());
                this.log.info("Skipping unsuccessful " + runToDownload.getConclusion() + " build #" + runToDownload.getRunNumber() + " " + runToDownload.getHtmlUrl());
                continue;
            }
            if (foundFailedRun == null) {
                this.log.debug("Found failed run " + runToDownload.getId() + " but seeing if there is another build in the same run that passed...");
                foundFailedRun = runToDownload;
                continue;
            }
            if (foundFailedRun.getRunNumber() != runToDownload.getRunNumber()) continue;
            if (foundFailedRun.getHeadCommit().getId().equals(runToDownload.getHeadCommit().getId())) break;
            this.log.debug("Found another failed run for " + runToDownload.getRunNumber());
        }
        if (foundFailedRun == null) {
            return null;
        }
        if (buildFilter.skipFailedBuilds) {
            throw new IOException("Latest build #" + foundFailedRun + " " + foundFailedRun.getHtmlUrl() + " failed");
        }
        this.log.debug("Found run " + foundFailedRun.getName() + ": " + foundFailedRun.getStatus() + " -- " + foundFailedRun.getConclusion() + "  build #" + foundFailedRun.getRunNumber() + " " + foundFailedRun.getHtmlUrl());
        return foundFailedRun;
    }

    public File downloadArtifact(String repo, String branchLabel, String artifactName, String workflowId, boolean skipFailedBuilds) throws IOException {
        GHWorkflowRun runToDownload = this.findLastBuild(repo, new BuildFilter(repo, branchLabel, skipFailedBuilds), workflowId);
        if (runToDownload == null) {
            throw new IOException("Could not find successful build for branch " + branchLabel);
        }
        this.log.info("Downloading artifacts in build #" + runToDownload.getRunNumber() + " originally ran at " + DateFormat.getDateTimeInstance().format(runToDownload.getCreatedAt()) + " -- " + runToDownload.getHtmlUrl());
        for (GHArtifact artifact : runToDownload.listArtifacts()) {
            if (artifact.getName().equals(artifactName)) {
                this.log.info("Downloading " + artifact.getName() + "...");
                URL url = artifact.getArchiveDownloadUrl();
                return this.downloadArtifact(url);
            }
            this.log.debug("Not downloading " + artifact.getName());
        }
        return null;
    }

    public File downloadArtifact(URL url) throws IOException {
        String extension = url.getPath().replaceFirst(".*\\.", "");
        if (extension.equals(url.getPath())) {
            extension = url.getPath().endsWith("/zip") ? "zip" : "tmp";
        }
        File file = File.createTempFile("liquibase-sdk-" + url.getPath().replaceFirst(".*/", "").replaceAll("\\W", "_") + "-", "." + extension);
        try (CloseableHttpClient httpclient = HttpClients.createDefault();){
            HttpGet httpGet = new HttpGet(url.toURI());
            httpGet.addHeader("Authorization", (Object)("token " + this.githubToken));
            try (CloseableHttpResponse response = httpclient.execute((ClassicHttpRequest)httpGet);){
                if (response.getCode() != 200) {
                    throw new IOException("Non-200 response: " + response.getCode() + " " + response.getReasonPhrase());
                }
                try (FileOutputStream out = new FileOutputStream(file);){
                    response.getEntity().writeTo((OutputStream)out);
                }
            }
        }
        catch (URISyntaxException e) {
            throw new IOException(e);
        }
        return file;
    }

    public void setCommitStatus(String repo, String sha1, GHCommitState statusState, String statusContext, String statusDescription, String statusUrl) throws IOException {
        GHRepository repository = this.getRepository(repo);
        this.log.debug("Successfully found repository " + repository.getHtmlUrl());
        repository.createCommitStatus(sha1, statusState, statusUrl, statusDescription, statusContext);
    }

    public Properties getInstalledBuildProperties(String repo) throws IOException {
        GHRepository ghRepository = this.getRepository(repo);
        String artifactName = this.handleArtifactName(ghRepository.getName());
        String m2Location = String.format("/.m2/repository/org/%s/%s/0-SNAPSHOT/%s-0-SNAPSHOT.jar", ghRepository.getOwner().getName().toLowerCase(), artifactName, artifactName);
        File libraryJar = new File(System.getProperty("user.home") + m2Location);
        if (!libraryJar.exists()) {
            throw new IOException(String.format("Could not find jar for %s at %s", artifactName, libraryJar.getAbsolutePath()));
        }
        try (FileInputStream fileInputStream = new FileInputStream(libraryJar);
             JarInputStream jarInputStream = new JarInputStream(fileInputStream);){
            JarEntry entry = jarInputStream.getNextJarEntry();
            while (true) {
                if (entry != null) {
                    if (entry.getName().equals("liquibase.build.properties")) {
                        Properties properties = new Properties();
                        properties.load(jarInputStream);
                        for (Map.Entry<Object, Object> property : properties.entrySet()) {
                            this.log.debug("Found property " + property.getKey() + "=" + property.getValue());
                        }
                        Properties properties2 = properties;
                        return properties2;
                    }
                    entry = jarInputStream.getNextJarEntry();
                    continue;
                }
                break;
            }
        }
        return null;
    }

    public void setPullRequestComment(String repo, String newComment, String pullRef, Pattern replaceComment, String mojoVersion) throws IOException {
        GHPullRequest pullRequest;
        newComment = newComment.replace("\\n", "\n").replace("\\t", "\t");
        GHRepository repository = this.getRepository(repo);
        this.log.debug("Successfully found repository " + repository.getHtmlUrl());
        if (pullRef.startsWith("#")) {
            pullRequest = repository.getPullRequest(Integer.parseInt(pullRef.substring(1)));
        } else {
            List prs = repository.queryPullRequests().head(pullRef).state(GHIssueState.OPEN).list().toList();
            if (prs.size() == 0) {
                throw new RuntimeException("Cannot find open PR for branch " + pullRef);
            }
            if (prs.size() > 1) {
                throw new RuntimeException("Found " + prs.size() + " PRs for branch " + pullRef);
            }
            pullRequest = (GHPullRequest)prs.get(0);
        }
        if (newComment.equals("BUILD_TESTING")) {
            this.log.info("Generating 'BUILD_TESTING' comment...");
            GHWorkflowRun lastBuild = this.findLastBuild(repo, new BuildFilter(repo, pullRequest.getBase().getRef(), false), GitHubClient.getWorkflowId(repo, null));
            newComment = "### Testing These Changes\nTo test this PR, use the artifacts attached to the [latest CI build](" + lastBuild.getHtmlUrl() + "#artifacts)\n\n#### Artifacts Available:\n- __" + repository.getName() + "-artifacts:__ Zip containing the .jar file to test\n- __test-reports-*:__ Detailed automated test results\n\n#### Download with liquibase-sdk-maven-plugin\nAlternately, you can use the [Liquibase SDK Maven Plugin](https://mvnrepository.com/artifact/org.liquibase.ext/liquibase-sdk-maven-plugin)\n\n##### Download the artifacts\n```mvn org.liquibase.ext:liquibase-sdk-maven-plugin:" + mojoVersion + ":download-snapshot-artifacts -Dliquibase.sdk.repo=" + repository.getFullName() + " -Dliquibase.sdk.branchSearch=" + pullRequest.getHead().getLabel() + " -Dliquibase.sdk.downloadDirectory=download -Dliquibase.sdk.artifactPattern=*-artifacts -Dliquibase.sdk.unzipArtifacts=true```\n##### Install to your local maven cache\n```mvn  org.liquibase.ext:liquibase-sdk-maven-plugin:" + mojoVersion + ":install-snapshot -Dliquibase.sdk.repo=" + repository.getFullName() + " -Dliquibase.sdk.branchSearch=" + pullRequest.getHead().getLabel() + "```\n";
            replaceComment = Pattern.compile("^#+ Testing These Changes");
        }
        if (replaceComment == null) {
            this.log.info("Creating new comment on " + pullRequest.getHtmlUrl());
            pullRequest.comment(newComment);
        } else {
            AtomicBoolean updated = new AtomicBoolean(false);
            String finalNewComment = newComment;
            Pattern finalReplaceComment = replaceComment;
            pullRequest.listComments().toList().forEach(comment -> {
                if (finalReplaceComment.matcher(comment.getBody()).find()) {
                    try {
                        this.log.info("Updating comment on " + pullRequest.getHtmlUrl());
                        comment.update(finalNewComment);
                        updated.set(true);
                    }
                    catch (IOException e) {
                        throw new RuntimeException(e);
                    }
                }
            });
            if (!updated.get()) {
                this.log.info("No existing matching comment, creating new comment on " + pullRequest.getHtmlUrl());
                pullRequest.comment(newComment);
            }
        }
    }

    public static String getWorkflowId(String repo, String workflowId) {
        if (workflowId != null) {
            return workflowId;
        }
        if (repo.endsWith("/liquibase") || repo.endsWith("/liquibase-pro")) {
            return "build.yml";
        }
        return "ci.yml";
    }

    public String handleArtifactName(String repositoryName) {
        String artifact = null;
        if (repositoryName.equalsIgnoreCase(PRO_REPOSITORY)) {
            artifact = PRO_ARTIFACT;
        } else if (repositoryName.equalsIgnoreCase(CORE_REPOSITORY)) {
            artifact = CORE_ARTIFACT;
        }
        return artifact != null ? artifact : repositoryName;
    }

    public static enum BuildStatusFilter {
        SUCCESS;

    }

    public static class BuildFilter {
        private String fork;
        private String branch;
        private final boolean skipFailedBuilds;

        public BuildFilter(String repo, String branch, boolean skipFailedBuilds) {
            this.skipFailedBuilds = skipFailedBuilds;
            this.branch = GitHubClient.simplifyBranch(branch);
            this.fork = repo;
            if (this.fork.contains("/")) {
                this.fork = repo.split("/")[0];
            }
            if (this.branch.contains(":")) {
                String[] split = this.branch.split(":", 2);
                this.fork = split[0];
                this.branch = split[1];
            }
        }

        public String getBranch() {
            return this.branch;
        }

        public String getFork() {
            return this.fork;
        }
    }
}

