/*
 * Copyright 2016 Azavea
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package geotrellis.spark.io.hbase

import geotrellis.spark.LayerId
import geotrellis.spark.io._
import org.slf4j.LoggerFactory
import com.typesafe.scalalogging.Logger
import org.apache.hadoop.hbase.client._
import org.apache.hadoop.hbase.filter.PrefixFilter

import scala.collection.JavaConverters._

class HBaseLayerDeleter(val attributeStore: AttributeStore, instance: HBaseInstance) extends LayerDeleter[LayerId] {
  @transient protected lazy val logger = Logger(LoggerFactory.getLogger(getClass.getName))

  def delete(id: LayerId): Unit = {
    try{
      val header = attributeStore.readHeader[HBaseLayerHeader](id)

      // Deletion list should be mutable
      val list = new java.util.ArrayList[Delete]()
      val scan = new Scan()
      scan.addFamily(HBaseRDDWriter.tilesCF)
      scan.setFilter(new PrefixFilter(HBaseRDDWriter.layerIdString(id)))

      instance.withTableConnectionDo(header.tileTable) { table =>
        val scanner = table.getScanner(scan)
        try {
          scanner.iterator().asScala.foreach { kv =>
            val delete = new Delete(kv.getRow)
            delete.addFamily(HBaseRDDWriter.tilesCF)
            list.add(delete)
          }
        } finally scanner.close()

        table.delete(list)
      }
    } catch {
      case e: AttributeNotFoundError =>
        logger.info(s"Metadata for $id was not found. Any associated layer data (if any) will require manual deletion")
        throw new LayerDeleteError(id).initCause(e)
    } finally {
      attributeStore.delete(id)
    }
  }
}

object HBaseLayerDeleter {
  def apply(attributeStore: AttributeStore, instance: HBaseInstance): HBaseLayerDeleter =
    new HBaseLayerDeleter(attributeStore, instance)

  def apply(attributeStore: HBaseAttributeStore): HBaseLayerDeleter =
    new HBaseLayerDeleter(attributeStore, attributeStore.instance)

  def apply(instance: HBaseInstance): HBaseLayerDeleter =
    apply(HBaseAttributeStore(instance), instance)
}
