package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ResourceRequirements}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableResourceRequirements.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableResourceRequirements
    implements ResourceRequirements {
  private final @Nullable Resources limits;
  private final @Nullable Reservations reservations;

  private ImmutableResourceRequirements(
      @Nullable Resources limits,
      @Nullable Reservations reservations) {
    this.limits = limits;
    this.reservations = reservations;
  }

  /**
   * @return The value of the {@code limits} attribute
   */
  @JsonProperty("Limits")
  @Override
  public @Nullable Resources limits() {
    return limits;
  }

  /**
   * @return The value of the {@code reservations} attribute
   */
  @JsonProperty("Reservations")
  @Override
  public @Nullable Reservations reservations() {
    return reservations;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ResourceRequirements#limits() limits} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for limits (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableResourceRequirements withLimits(@Nullable Resources value) {
    if (this.limits == value) return this;
    return new ImmutableResourceRequirements(value, this.reservations);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ResourceRequirements#reservations() reservations} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for reservations (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableResourceRequirements withReservations(@Nullable Reservations value) {
    if (this.reservations == value) return this;
    return new ImmutableResourceRequirements(this.limits, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableResourceRequirements} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableResourceRequirements
        && equalTo((ImmutableResourceRequirements) another);
  }

  private boolean equalTo(ImmutableResourceRequirements another) {
    return Objects.equals(limits, another.limits)
        && Objects.equals(reservations, another.reservations);
  }

  /**
   * Computes a hash code from attributes: {@code limits}, {@code reservations}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(limits);
    h += (h << 5) + Objects.hashCode(reservations);
    return h;
  }

  /**
   * Prints the immutable value {@code ResourceRequirements} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ResourceRequirements{"
        + "limits=" + limits
        + ", reservations=" + reservations
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ResourceRequirements} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ResourceRequirements instance
   */
  public static ImmutableResourceRequirements copyOf(ResourceRequirements instance) {
    if (instance instanceof ImmutableResourceRequirements) {
      return (ImmutableResourceRequirements) instance;
    }
    return ImmutableResourceRequirements.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableResourceRequirements ImmutableResourceRequirements}.
   * <pre>
   * ImmutableResourceRequirements.builder()
   *    .limits(org.mandas.docker.client.messages.swarm.Resources | null) // nullable {@link ResourceRequirements#limits() limits}
   *    .reservations(org.mandas.docker.client.messages.swarm.Reservations | null) // nullable {@link ResourceRequirements#reservations() reservations}
   *    .build();
   * </pre>
   * @return A new ImmutableResourceRequirements builder
   */
  public static ImmutableResourceRequirements.Builder builder() {
    return new ImmutableResourceRequirements.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableResourceRequirements ImmutableResourceRequirements}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ResourceRequirements.Builder {
    private Resources limits;
    private Reservations reservations;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ResourceRequirements} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ResourceRequirements instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Resources limitsValue = instance.limits();
      if (limitsValue != null) {
        limits(limitsValue);
      }
      @Nullable Reservations reservationsValue = instance.reservations();
      if (reservationsValue != null) {
        reservations(reservationsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ResourceRequirements#limits() limits} attribute.
     * @param limits The value for limits (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Limits")
    public final Builder limits(@Nullable Resources limits) {
      this.limits = limits;
      return this;
    }

    /**
     * Initializes the value for the {@link ResourceRequirements#reservations() reservations} attribute.
     * @param reservations The value for reservations (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Reservations")
    public final Builder reservations(@Nullable Reservations reservations) {
      this.reservations = reservations;
      return this;
    }

    /**
     * Builds a new {@link ImmutableResourceRequirements ImmutableResourceRequirements}.
     * @return An immutable instance of ResourceRequirements
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableResourceRequirements build() {
      return new ImmutableResourceRequirements(limits, reservations);
    }
  }
}
