package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link NetworkAttachmentConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNetworkAttachmentConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNetworkAttachmentConfig
    implements NetworkAttachmentConfig {
  private final @Nullable String target;
  private final @Nullable List<String> aliases;

  private ImmutableNetworkAttachmentConfig(
      @Nullable String target,
      @Nullable List<String> aliases) {
    this.target = target;
    this.aliases = aliases;
  }

  /**
   * @return The value of the {@code target} attribute
   */
  @JsonProperty("Target")
  @Override
  public @Nullable String target() {
    return target;
  }

  /**
   * @return The value of the {@code aliases} attribute
   */
  @JsonProperty("Aliases")
  @Override
  public @Nullable List<String> aliases() {
    return aliases;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkAttachmentConfig#target() target} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for target (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableNetworkAttachmentConfig withTarget(@Nullable String value) {
    if (Objects.equals(this.target, value)) return this;
    return new ImmutableNetworkAttachmentConfig(value, this.aliases);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link NetworkAttachmentConfig#aliases() aliases}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableNetworkAttachmentConfig withAliases(@Nullable String... elements) {
    if (elements == null) {
      return new ImmutableNetworkAttachmentConfig(this.target, null);
    }
    @Nullable List<String> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableNetworkAttachmentConfig(this.target, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link NetworkAttachmentConfig#aliases() aliases}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of aliases elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableNetworkAttachmentConfig withAliases(@Nullable Iterable<String> elements) {
    if (this.aliases == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableNetworkAttachmentConfig(this.target, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNetworkAttachmentConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNetworkAttachmentConfig
        && equalTo(0, (ImmutableNetworkAttachmentConfig) another);
  }

  private boolean equalTo(int synthetic, ImmutableNetworkAttachmentConfig another) {
    return Objects.equals(target, another.target)
        && Objects.equals(aliases, another.aliases);
  }

  /**
   * Computes a hash code from attributes: {@code target}, {@code aliases}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(target);
    h += (h << 5) + Objects.hashCode(aliases);
    return h;
  }

  /**
   * Prints the immutable value {@code NetworkAttachmentConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NetworkAttachmentConfig{"
        + "target=" + target
        + ", aliases=" + aliases
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NetworkAttachmentConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NetworkAttachmentConfig instance
   */
  public static ImmutableNetworkAttachmentConfig copyOf(NetworkAttachmentConfig instance) {
    if (instance instanceof ImmutableNetworkAttachmentConfig) {
      return (ImmutableNetworkAttachmentConfig) instance;
    }
    return ImmutableNetworkAttachmentConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNetworkAttachmentConfig ImmutableNetworkAttachmentConfig}.
   * <pre>
   * ImmutableNetworkAttachmentConfig.builder()
   *    .target(String | null) // nullable {@link NetworkAttachmentConfig#target() target}
   *    .aliases(List&amp;lt;String&amp;gt; | null) // nullable {@link NetworkAttachmentConfig#aliases() aliases}
   *    .build();
   * </pre>
   * @return A new ImmutableNetworkAttachmentConfig builder
   */
  public static ImmutableNetworkAttachmentConfig.Builder builder() {
    return new ImmutableNetworkAttachmentConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNetworkAttachmentConfig ImmutableNetworkAttachmentConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements NetworkAttachmentConfig.Builder {
    private String target;
    private List<String> aliases = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NetworkAttachmentConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NetworkAttachmentConfig instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String targetValue = instance.target();
      if (targetValue != null) {
        target(targetValue);
      }
      @Nullable List<String> aliasesValue = instance.aliases();
      if (aliasesValue != null) {
        addAllAliases(aliasesValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkAttachmentConfig#target() target} attribute.
     * @param target The value for target (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Target")
    public final Builder target(@Nullable String target) {
      this.target = target;
      return this;
    }

    /**
     * Adds one element to {@link NetworkAttachmentConfig#aliases() aliases} list.
     * @param element A aliases element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder aliase(String element) {
      if (this.aliases == null) {
        this.aliases = new ArrayList<String>();
      }
      this.aliases.add(Objects.requireNonNull(element, "aliases element"));
      return this;
    }

    /**
     * Adds elements to {@link NetworkAttachmentConfig#aliases() aliases} list.
     * @param elements An array of aliases elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder aliases(String... elements) {
      if (this.aliases == null) {
        this.aliases = new ArrayList<String>();
      }
      for (String element : elements) {
        this.aliases.add(Objects.requireNonNull(element, "aliases element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link NetworkAttachmentConfig#aliases() aliases} list.
     * @param elements An iterable of aliases elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Aliases")
    public final Builder aliases(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.aliases = null;
        return this;
      }
      this.aliases = new ArrayList<String>();
      return addAllAliases(elements);
    }

    /**
     * Adds elements to {@link NetworkAttachmentConfig#aliases() aliases} list.
     * @param elements An iterable of aliases elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllAliases(Iterable<String> elements) {
      Objects.requireNonNull(elements, "aliases element");
      if (this.aliases == null) {
        this.aliases = new ArrayList<String>();
      }
      for (String element : elements) {
        this.aliases.add(Objects.requireNonNull(element, "aliases element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableNetworkAttachmentConfig ImmutableNetworkAttachmentConfig}.
     * @return An immutable instance of NetworkAttachmentConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNetworkAttachmentConfig build() {
      return new ImmutableNetworkAttachmentConfig(target, aliases == null ? null : createUnmodifiableList(true, aliases));
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
