package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ServiceSpec}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableServiceSpec.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableServiceSpec implements ServiceSpec {
  private final @Nullable String name;
  private final @Nullable Map<String, String> labels;
  private final TaskSpec taskTemplate;
  private final @Nullable ServiceMode mode;
  private final @Nullable UpdateConfig updateConfig;
  private final @Nullable List<NetworkAttachmentConfig> networks;
  private final @Nullable EndpointSpec endpointSpec;

  private ImmutableServiceSpec(
      @Nullable String name,
      @Nullable Map<String, String> labels,
      TaskSpec taskTemplate,
      @Nullable ServiceMode mode,
      @Nullable UpdateConfig updateConfig,
      @Nullable List<NetworkAttachmentConfig> networks,
      @Nullable EndpointSpec endpointSpec) {
    this.name = name;
    this.labels = labels;
    this.taskTemplate = taskTemplate;
    this.mode = mode;
    this.updateConfig = updateConfig;
    this.networks = networks;
    this.endpointSpec = endpointSpec;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public @Nullable String name() {
    return name;
  }

  /**
   * @return The value of the {@code labels} attribute
   */
  @JsonProperty("Labels")
  @Override
  public @Nullable Map<String, String> labels() {
    return labels;
  }

  /**
   * @return The value of the {@code taskTemplate} attribute
   */
  @JsonProperty("TaskTemplate")
  @Override
  public TaskSpec taskTemplate() {
    return taskTemplate;
  }

  /**
   * @return The value of the {@code mode} attribute
   */
  @JsonProperty("Mode")
  @Override
  public @Nullable ServiceMode mode() {
    return mode;
  }

  /**
   * @return The value of the {@code updateConfig} attribute
   */
  @JsonProperty("UpdateConfig")
  @Override
  public @Nullable UpdateConfig updateConfig() {
    return updateConfig;
  }

  /**
   * @return The value of the {@code networks} attribute
   */
  @JsonProperty("Networks")
  @Override
  public @Nullable List<NetworkAttachmentConfig> networks() {
    return networks;
  }

  /**
   * @return The value of the {@code endpointSpec} attribute
   */
  @JsonProperty("EndpointSpec")
  @Override
  public @Nullable EndpointSpec endpointSpec() {
    return endpointSpec;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ServiceSpec#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableServiceSpec withName(@Nullable String value) {
    if (Objects.equals(this.name, value)) return this;
    return new ImmutableServiceSpec(
        value,
        this.labels,
        this.taskTemplate,
        this.mode,
        this.updateConfig,
        this.networks,
        this.endpointSpec);
  }

  /**
   * Copy the current immutable object by replacing the {@link ServiceSpec#labels() labels} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the labels map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableServiceSpec withLabels(@Nullable Map<String, ? extends String> entries) {
    if (this.labels == entries) return this;
    @Nullable Map<String, String> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableServiceSpec(
        this.name,
        newValue,
        this.taskTemplate,
        this.mode,
        this.updateConfig,
        this.networks,
        this.endpointSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ServiceSpec#taskTemplate() taskTemplate} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for taskTemplate
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableServiceSpec withTaskTemplate(TaskSpec value) {
    if (this.taskTemplate == value) return this;
    TaskSpec newValue = Objects.requireNonNull(value, "taskTemplate");
    return new ImmutableServiceSpec(
        this.name,
        this.labels,
        newValue,
        this.mode,
        this.updateConfig,
        this.networks,
        this.endpointSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ServiceSpec#mode() mode} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for mode (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableServiceSpec withMode(@Nullable ServiceMode value) {
    if (this.mode == value) return this;
    return new ImmutableServiceSpec(
        this.name,
        this.labels,
        this.taskTemplate,
        value,
        this.updateConfig,
        this.networks,
        this.endpointSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ServiceSpec#updateConfig() updateConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updateConfig (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableServiceSpec withUpdateConfig(@Nullable UpdateConfig value) {
    if (this.updateConfig == value) return this;
    return new ImmutableServiceSpec(this.name, this.labels, this.taskTemplate, this.mode, value, this.networks, this.endpointSpec);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ServiceSpec#networks() networks}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableServiceSpec withNetworks(@Nullable NetworkAttachmentConfig... elements) {
    if (elements == null) {
      return new ImmutableServiceSpec(
          this.name,
          this.labels,
          this.taskTemplate,
          this.mode,
          this.updateConfig,
          null,
          this.endpointSpec);
    }
    @Nullable List<NetworkAttachmentConfig> newValue = Arrays.asList(elements) == null ? null : createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableServiceSpec(
        this.name,
        this.labels,
        this.taskTemplate,
        this.mode,
        this.updateConfig,
        newValue,
        this.endpointSpec);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ServiceSpec#networks() networks}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of networks elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableServiceSpec withNetworks(@Nullable Iterable<? extends NetworkAttachmentConfig> elements) {
    if (this.networks == elements) return this;
    @Nullable List<NetworkAttachmentConfig> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableServiceSpec(
        this.name,
        this.labels,
        this.taskTemplate,
        this.mode,
        this.updateConfig,
        newValue,
        this.endpointSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ServiceSpec#endpointSpec() endpointSpec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for endpointSpec (can be {@code null})
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableServiceSpec withEndpointSpec(@Nullable EndpointSpec value) {
    if (this.endpointSpec == value) return this;
    return new ImmutableServiceSpec(this.name, this.labels, this.taskTemplate, this.mode, this.updateConfig, this.networks, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableServiceSpec} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableServiceSpec
        && equalTo(0, (ImmutableServiceSpec) another);
  }

  private boolean equalTo(int synthetic, ImmutableServiceSpec another) {
    return Objects.equals(name, another.name)
        && Objects.equals(labels, another.labels)
        && taskTemplate.equals(another.taskTemplate)
        && Objects.equals(mode, another.mode)
        && Objects.equals(updateConfig, another.updateConfig)
        && Objects.equals(networks, another.networks)
        && Objects.equals(endpointSpec, another.endpointSpec);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code labels}, {@code taskTemplate}, {@code mode}, {@code updateConfig}, {@code networks}, {@code endpointSpec}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(name);
    h += (h << 5) + Objects.hashCode(labels);
    h += (h << 5) + taskTemplate.hashCode();
    h += (h << 5) + Objects.hashCode(mode);
    h += (h << 5) + Objects.hashCode(updateConfig);
    h += (h << 5) + Objects.hashCode(networks);
    h += (h << 5) + Objects.hashCode(endpointSpec);
    return h;
  }

  /**
   * Prints the immutable value {@code ServiceSpec} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ServiceSpec{"
        + "name=" + name
        + ", labels=" + labels
        + ", taskTemplate=" + taskTemplate
        + ", mode=" + mode
        + ", updateConfig=" + updateConfig
        + ", networks=" + networks
        + ", endpointSpec=" + endpointSpec
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ServiceSpec} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ServiceSpec instance
   */
  public static ImmutableServiceSpec copyOf(ServiceSpec instance) {
    if (instance instanceof ImmutableServiceSpec) {
      return (ImmutableServiceSpec) instance;
    }
    return ImmutableServiceSpec.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableServiceSpec ImmutableServiceSpec}.
   * <pre>
   * ImmutableServiceSpec.builder()
   *    .name(String | null) // nullable {@link ServiceSpec#name() name}
   *    .labels(Map&amp;lt;String, String&amp;gt; | null) // nullable {@link ServiceSpec#labels() labels}
   *    .taskTemplate(org.mandas.docker.client.messages.swarm.TaskSpec) // required {@link ServiceSpec#taskTemplate() taskTemplate}
   *    .mode(org.mandas.docker.client.messages.swarm.ServiceMode | null) // nullable {@link ServiceSpec#mode() mode}
   *    .updateConfig(org.mandas.docker.client.messages.swarm.UpdateConfig | null) // nullable {@link ServiceSpec#updateConfig() updateConfig}
   *    .networks(List&amp;lt;org.mandas.docker.client.messages.swarm.NetworkAttachmentConfig&amp;gt; | null) // nullable {@link ServiceSpec#networks() networks}
   *    .endpointSpec(org.mandas.docker.client.messages.swarm.EndpointSpec | null) // nullable {@link ServiceSpec#endpointSpec() endpointSpec}
   *    .build();
   * </pre>
   * @return A new ImmutableServiceSpec builder
   */
  public static ImmutableServiceSpec.Builder builder() {
    return new ImmutableServiceSpec.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableServiceSpec ImmutableServiceSpec}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ServiceSpec.Builder {
    private static final long INIT_BIT_TASK_TEMPLATE = 0x1L;
    private long initBits = 0x1L;

    private String name;
    private Map<String, String> labels = null;
    private TaskSpec taskTemplate;
    private ServiceMode mode;
    private UpdateConfig updateConfig;
    private List<NetworkAttachmentConfig> networks = null;
    private EndpointSpec endpointSpec;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ServiceSpec} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ServiceSpec instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String nameValue = instance.name();
      if (nameValue != null) {
        name(nameValue);
      }
      @Nullable Map<String, String> labelsValue = instance.labels();
      if (labelsValue != null) {
        putAllLabels(labelsValue);
      }
      taskTemplate(instance.taskTemplate());
      @Nullable ServiceMode modeValue = instance.mode();
      if (modeValue != null) {
        mode(modeValue);
      }
      @Nullable UpdateConfig updateConfigValue = instance.updateConfig();
      if (updateConfigValue != null) {
        updateConfig(updateConfigValue);
      }
      @Nullable List<NetworkAttachmentConfig> networksValue = instance.networks();
      if (networksValue != null) {
        addAllNetworks(networksValue);
      }
      @Nullable EndpointSpec endpointSpecValue = instance.endpointSpec();
      if (endpointSpecValue != null) {
        endpointSpec(endpointSpecValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ServiceSpec#name() name} attribute.
     * @param name The value for name (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(@Nullable String name) {
      this.name = name;
      return this;
    }

    /**
     * Put one entry to the {@link ServiceSpec#labels() labels} map.
     * @param key The key in the labels map
     * @param value The associated value in the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(String key, String value) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      this.labels.put(
          Objects.requireNonNull(key, "labels key"),
          value == null ? Objects.requireNonNull(value, "labels value for key: " + key) : value);
      return this;
    }

    /**
     * Put one entry to the {@link ServiceSpec#labels() labels} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addLabel(Map.Entry<String, ? extends String> entry) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      String k = entry.getKey();
      String v = entry.getValue();
      this.labels.put(
          Objects.requireNonNull(k, "labels key"),
          v == null ? Objects.requireNonNull(v, "labels value for key: " + k) : v);
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link ServiceSpec#labels() labels} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Labels")
    public final Builder labels(@Nullable Map<String, ? extends String> entries) {
      if (entries == null) {
        this.labels = null;
        return this;
      }
      this.labels = new LinkedHashMap<String, String>();
      return putAllLabels(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link ServiceSpec#labels() labels} map. Nulls are not permitted
     * @param entries The entries that will be added to the labels map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllLabels(Map<String, ? extends String> entries) {
      if (this.labels == null) {
        this.labels = new LinkedHashMap<String, String>();
      }
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.labels.put(
            Objects.requireNonNull(k, "labels key"),
            v == null ? Objects.requireNonNull(v, "labels value for key: " + k) : v);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ServiceSpec#taskTemplate() taskTemplate} attribute.
     * @param taskTemplate The value for taskTemplate 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("TaskTemplate")
    public final Builder taskTemplate(TaskSpec taskTemplate) {
      this.taskTemplate = Objects.requireNonNull(taskTemplate, "taskTemplate");
      initBits &= ~INIT_BIT_TASK_TEMPLATE;
      return this;
    }

    /**
     * Initializes the value for the {@link ServiceSpec#mode() mode} attribute.
     * @param mode The value for mode (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Mode")
    public final Builder mode(@Nullable ServiceMode mode) {
      this.mode = mode;
      return this;
    }

    /**
     * Initializes the value for the {@link ServiceSpec#updateConfig() updateConfig} attribute.
     * @param updateConfig The value for updateConfig (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UpdateConfig")
    public final Builder updateConfig(@Nullable UpdateConfig updateConfig) {
      this.updateConfig = updateConfig;
      return this;
    }

    /**
     * Adds one element to {@link ServiceSpec#networks() networks} list.
     * @param element A networks element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder network(NetworkAttachmentConfig element) {
      if (this.networks == null) {
        this.networks = new ArrayList<NetworkAttachmentConfig>();
      }
      this.networks.add(Objects.requireNonNull(element, "networks element"));
      return this;
    }

    /**
     * Adds elements to {@link ServiceSpec#networks() networks} list.
     * @param elements An array of networks elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder networks(NetworkAttachmentConfig... elements) {
      if (this.networks == null) {
        this.networks = new ArrayList<NetworkAttachmentConfig>();
      }
      for (NetworkAttachmentConfig element : elements) {
        this.networks.add(Objects.requireNonNull(element, "networks element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ServiceSpec#networks() networks} list.
     * @param elements An iterable of networks elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Networks")
    public final Builder networks(@Nullable Iterable<? extends NetworkAttachmentConfig> elements) {
      if (elements == null) {
        this.networks = null;
        return this;
      }
      this.networks = new ArrayList<NetworkAttachmentConfig>();
      return addAllNetworks(elements);
    }

    /**
     * Adds elements to {@link ServiceSpec#networks() networks} list.
     * @param elements An iterable of networks elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllNetworks(Iterable<? extends NetworkAttachmentConfig> elements) {
      Objects.requireNonNull(elements, "networks element");
      if (this.networks == null) {
        this.networks = new ArrayList<NetworkAttachmentConfig>();
      }
      for (NetworkAttachmentConfig element : elements) {
        this.networks.add(Objects.requireNonNull(element, "networks element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ServiceSpec#endpointSpec() endpointSpec} attribute.
     * @param endpointSpec The value for endpointSpec (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("EndpointSpec")
    public final Builder endpointSpec(@Nullable EndpointSpec endpointSpec) {
      this.endpointSpec = endpointSpec;
      return this;
    }

    /**
     * Builds a new {@link ImmutableServiceSpec ImmutableServiceSpec}.
     * @return An immutable instance of ServiceSpec
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableServiceSpec build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableServiceSpec(
          name,
          labels == null ? null : createUnmodifiableMap(false, false, labels),
          taskTemplate,
          mode,
          updateConfig,
          networks == null ? null : createUnmodifiableList(true, networks),
          endpointSpec);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_TASK_TEMPLATE) != 0) attributes.add("taskTemplate");
      return "Cannot build ServiceSpec, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>();
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        if (v == null) Objects.requireNonNull(v, "value for key: " + k);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size());
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            if (v == null) Objects.requireNonNull(v, "value for key: " + k);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
