package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link JoinTokens}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableJoinTokens.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableJoinTokens implements JoinTokens {
  private final String worker;
  private final String manager;

  private ImmutableJoinTokens(String worker, String manager) {
    this.worker = worker;
    this.manager = manager;
  }

  /**
   * @return The value of the {@code worker} attribute
   */
  @JsonProperty("Worker")
  @Override
  public String worker() {
    return worker;
  }

  /**
   * @return The value of the {@code manager} attribute
   */
  @JsonProperty("Manager")
  @Override
  public String manager() {
    return manager;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link JoinTokens#worker() worker} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for worker
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableJoinTokens withWorker(String value) {
    String newValue = Objects.requireNonNull(value, "worker");
    if (this.worker.equals(newValue)) return this;
    return new ImmutableJoinTokens(newValue, this.manager);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link JoinTokens#manager() manager} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for manager
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableJoinTokens withManager(String value) {
    String newValue = Objects.requireNonNull(value, "manager");
    if (this.manager.equals(newValue)) return this;
    return new ImmutableJoinTokens(this.worker, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableJoinTokens} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableJoinTokens
        && equalTo(0, (ImmutableJoinTokens) another);
  }

  private boolean equalTo(int synthetic, ImmutableJoinTokens another) {
    return worker.equals(another.worker)
        && manager.equals(another.manager);
  }

  /**
   * Computes a hash code from attributes: {@code worker}, {@code manager}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + worker.hashCode();
    h += (h << 5) + manager.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code JoinTokens} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "JoinTokens{"
        + "worker=" + worker
        + ", manager=" + manager
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link JoinTokens} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable JoinTokens instance
   */
  public static ImmutableJoinTokens copyOf(JoinTokens instance) {
    if (instance instanceof ImmutableJoinTokens) {
      return (ImmutableJoinTokens) instance;
    }
    return ImmutableJoinTokens.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableJoinTokens ImmutableJoinTokens}.
   * <pre>
   * ImmutableJoinTokens.builder()
   *    .worker(String) // required {@link JoinTokens#worker() worker}
   *    .manager(String) // required {@link JoinTokens#manager() manager}
   *    .build();
   * </pre>
   * @return A new ImmutableJoinTokens builder
   */
  public static ImmutableJoinTokens.Builder builder() {
    return new ImmutableJoinTokens.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableJoinTokens ImmutableJoinTokens}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_WORKER = 0x1L;
    private static final long INIT_BIT_MANAGER = 0x2L;
    private long initBits = 0x3L;

    private String worker;
    private String manager;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code JoinTokens} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(JoinTokens instance) {
      Objects.requireNonNull(instance, "instance");
      worker(instance.worker());
      manager(instance.manager());
      return this;
    }

    /**
     * Initializes the value for the {@link JoinTokens#worker() worker} attribute.
     * @param worker The value for worker 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Worker")
    public final Builder worker(String worker) {
      this.worker = Objects.requireNonNull(worker, "worker");
      initBits &= ~INIT_BIT_WORKER;
      return this;
    }

    /**
     * Initializes the value for the {@link JoinTokens#manager() manager} attribute.
     * @param manager The value for manager 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Manager")
    public final Builder manager(String manager) {
      this.manager = Objects.requireNonNull(manager, "manager");
      initBits &= ~INIT_BIT_MANAGER;
      return this;
    }

    /**
     * Builds a new {@link ImmutableJoinTokens ImmutableJoinTokens}.
     * @return An immutable instance of JoinTokens
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableJoinTokens build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableJoinTokens(worker, manager);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_WORKER) != 0) attributes.add("worker");
      if ((initBits & INIT_BIT_MANAGER) != 0) attributes.add("manager");
      return "Cannot build JoinTokens, some of required attributes are not set " + attributes;
    }
  }
}
