/*
 * Copyright (C) 2013  Camptocamp
 *
 * This file is part of MapFish Print
 *
 * MapFish Print is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * MapFish Print is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MapFish Print.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.mapfish.print.map.readers;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.mapfish.print.RenderingContext;
import org.mapfish.print.Transformer;
import org.mapfish.print.map.renderers.TileRenderer;
import org.mapfish.print.utils.DistanceUnit;
import org.mapfish.print.utils.PJsonArray;
import org.mapfish.print.utils.PJsonObject;

/**
 * Support for the protocol talking direclty to a web-accessible ka-Map cache
 * generated by the precache2.php script.
 */
public class KaMapCacheMapReader extends TileableMapReader {
    public static class Factory implements MapReaderFactory {

        @Override
        public List<MapReader> create(String type, RenderingContext context,
                PJsonObject params) {
            ArrayList<MapReader> target = new ArrayList<MapReader>();

            String map = params.getString("map");
            String group = params.getString("group");
            int metaTileWidth = params.getInt("metaTileWidth");
            int metaTileHeight = params.getInt("metaTileHeight");
            String units = context.getGlobalParams().getString("units");

            target.add(new KaMapCacheMapReader(map, group, units, metaTileWidth, metaTileHeight, context, params));

            return target;
        }
    }

    private final String map;
    private final String group;
    private final String units;

    private final int metaTileWidth;
    private final int metaTileHeight;

    private KaMapCacheMapReader(String map, String group, String units, int metaTileWidth, int metaTileHeight, RenderingContext context, PJsonObject params) {
        super(context, params);
        this.map = map;
        this.group = group;
        this.units = units;
        this.metaTileWidth = metaTileWidth;
        this.metaTileHeight = metaTileHeight;

        PJsonArray maxExtent = params.getJSONArray("maxExtent");
        PJsonArray tileSize = params.getJSONArray("tileSize");
        tileCacheLayerInfo = new TileCacheLayerInfo(params.getJSONArray("resolutions"), tileSize.getInt(0), tileSize.getInt(1), maxExtent.getFloat(0), maxExtent.getFloat(1), maxExtent.getFloat(2), maxExtent.getFloat(3), params.getString("extension"));
    }
    @Override
    protected TileRenderer.Format getFormat() {
        return TileRenderer.Format.BITMAP;
    }
    @Override
    protected void addCommonQueryParams(Map<String, List<String>> result, Transformer transformer, String srs, boolean first) {
        //not much query params for this protocol...
    }
    @Override
    protected URI getTileUri(URI commonUri, Transformer transformer, double minGeoX, double minGeoY, double maxGeoX, double maxGeoY, long w, long h) throws URISyntaxException, UnsupportedEncodingException {
        double targetResolution = (maxGeoX - minGeoX) / w;
        TileCacheLayerInfo.ResolutionInfo resolution = tileCacheLayerInfo.getNearestResolution(targetResolution);

        // scale, calculated from units used
        final DistanceUnit unitEnum = DistanceUnit.fromString(units);
        if (unitEnum == null) {
            throw new RuntimeException("Unknown unit: '" + units + "'");
        }
        final double scale = context.getConfig().getBestScale(Math.max(
            (maxGeoX - minGeoX) / (DistanceUnit.PT.convertTo(w, unitEnum)),
            (maxGeoY- minGeoY) / (DistanceUnit.PT.convertTo(h, unitEnum))));

        // top & left
        double pX = Math.round(minGeoX / resolution.value);
        double pY = Math.round(maxGeoY / resolution.value) * -1;
        pX = Math.floor(pX / w) * new Double(w);
        pY = Math.floor(pY / h) * new Double(h);
        int pX4Path = (int)pX;
        int pY4Path = (int)pY;

        // metaX and metaY
        long metaX = (long)Math.floor(pX / w / metaTileWidth) * w * metaTileWidth;
        long metaY = (long)Math.floor(pY / h / metaTileHeight) * h * metaTileHeight;

        // path building
        StringBuilder path = new StringBuilder();
        if (!commonUri.getPath().endsWith("/")) {
            path.append('/');
        }
        path.append(map);
        path.append('/').append(scale);
        path.append('/').append(group).append("");
        path.append("/def");
        path.append("/t").append(metaY);
        path.append("/l").append(metaX);
        path.append("/t").append(pY4Path).append('l').append(pX4Path).append('.').append(tileCacheLayerInfo.getExtension());

        return new URI(commonUri.getScheme(), commonUri.getUserInfo(), commonUri.getHost(), commonUri.getPort(), commonUri.getPath() + path.toString(), commonUri.getQuery(), commonUri.getFragment());
    }
    @Override
    public boolean testMerge(MapReader other) {
        return false;
    }
    @Override
    public boolean canMerge(MapReader other) {
        return false;
    }
    @Override
    public String toString() {
        return map;
    }
}