/*
MariaDB Client for Java

Copyright (c) 2012-2014 Monty Program Ab.

This library is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the Free
Software Foundation; either version 2.1 of the License, or (at your option)
any later version.

This library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
for more details.

You should have received a copy of the GNU Lesser General Public License along
with this library; if not, write to Monty Program Ab info@montyprogram.com.

This particular MariaDB Client for Java file is work
derived from a Drizzle-JDBC. Drizzle-JDBC file which is covered by subject to
the following copyright and notice provisions:

Copyright (c) 2009-2011, Marcus Eriksson

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:
Redistributions of source code must retain the above copyright notice, this list
of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice, this
list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

Neither the name of the driver nor the names of its contributors may not be
used to endorse or promote products derived from this software without specific
prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS  AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

package org.mariadb.jdbc.internal.util.dao;


import org.mariadb.jdbc.internal.util.ExceptionCode;
import org.mariadb.jdbc.internal.util.SqlStates;

import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

public class QueryException extends Exception {
    private static final long serialVersionUID = 974263994278018455L;
    /**
     * the internal code.
     */
    private int errorCode;
    /**
     * the sql state.
     */
    private String sqlState;
    private String message;

    /**
     * Creates a default query exception with errorCode -1 and sqlState HY0000.
     *
     * @param message the message to set
     */
    public QueryException(final String message) {
        super(message);
        this.message = message;
        this.errorCode = -1;
        this.sqlState = "HY0000";

    }

    /**
     * Creates a query exception with a message.
     *
     * @param message   the message
     * @param errorCode the error code
     * @param sqlState  the sqlstate
     */
    public QueryException(final String message,
                          final int errorCode,
                          final String sqlState) {
        super(message);
        this.message = message;
        this.errorCode = errorCode;
        this.sqlState = sqlState;
    }

    /**
     * Creates a query exception with a message.
     *
     * @param message   the message
     * @param errorCode the error code
     * @param sqlState  the sqlstate
     */
    public QueryException(final String message,
                          final int errorCode,
                          final SqlStates sqlState) {
        super(message);
        this.message = message;
        this.errorCode = errorCode;
        this.sqlState = sqlState.getSqlState();
    }

    /**
     * Creates a query exception with a message.
     *
     * @param message   the message
     * @param exceptionCode exceptionCode
     */
    public QueryException(final String message, ExceptionCode exceptionCode) {
        super(message);
        this.message = message;
        this.errorCode = -1;
        this.sqlState = exceptionCode.sqlState;
    }

    /**
     * Creates a query exception with a message and a cause.
     *
     * @param message   the exception message
     * @param errorCode the error code
     * @param sqlState  the sql state
     * @param cause     the cause of the exception
     */
    public QueryException(String message,
                          int errorCode,
                          String sqlState,
                          Throwable cause) {
        super(message, cause);
        this.message = message;
        this.errorCode = errorCode;
        this.sqlState = sqlState;
    }

    /**
     * Creates a query exception with a message and a cause.
     *
     * @param message   the exception message
     * @param errorCode the error code
     * @param sqlState  the sql state
     * @param cause     the cause of the exception
     */
    public QueryException(String message,
                          int errorCode,
                          SqlStates sqlState,
                          Throwable cause) {
        super(message, cause);
        this.message = message;
        this.errorCode = errorCode;
        this.sqlState = sqlState.getSqlState();
    }

    @Override
    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    /**
     * Indicate if Exception is due to query not preparable.
     * @return is excption due because query isn't preparable
     */
    public boolean isPrepareError() {
        return false;
    }

    /**
     * returns the error code.
     *
     * @return the error code
     */
    public final int getErrorCode() {
        return errorCode;
    }


    /**
     * Gets the sql state.
     *
     * @return the sql state
     */
    public final String getSqlState() {
        return sqlState;
    }

    /**
     * Sets the sql state.
     *
     * @param sqlState sqlState
     */
    public void setSqlState(String sqlState) {
        this.sqlState = sqlState;
    }


    /**
     * Adds an <code>QueryException</code> object to the end of the chain.
     *
     * @param ex the new exception that will be added to the end of
     *            the <code>QueryException</code> chain
     */
    public void setNextException(QueryException ex) {

        QueryException current = this;
        for (;;) {
            QueryException next = current.next;
            if (next != null) {
                current = next;
                continue;
            }

            if (nextUpdater.compareAndSet(current,null,ex)) {
                return;
            }
            current = current.next;
        }
    }

    public QueryException getNextException() {
        return next;
    }

    private volatile QueryException next;

    private static final AtomicReferenceFieldUpdater<QueryException,QueryException> nextUpdater =
            AtomicReferenceFieldUpdater.newUpdater(QueryException.class,QueryException.class,"next");
}
