/*
 * @(#)MathFunctions.java        6.1.0    2024-10-06
 *
 * MathParser.org-mXparser DUAL LICENSE AGREEMENT as of date 2024-05-19
 * The most up-to-date license is available at the below link:
 * - https://mathparser.org/mxparser-license
 *
 * AUTHOR: Copyright 2010 - 2024 Mariusz Gromada - All rights reserved
 * PUBLISHER: INFIMA - https://payhip.com/infima
 *
 * SOFTWARE means source code and/or binary form and/or documentation.
 * PRODUCT: MathParser.org-mXparser SOFTWARE
 * LICENSE: DUAL LICENSE AGREEMENT
 *
 * BY INSTALLING, COPYING, OR OTHERWISE USING THE PRODUCT, YOU AGREE TO BE
 * BOUND BY ALL OF THE TERMS AND CONDITIONS OF THE DUAL LICENSE AGREEMENT.
 *
 * The AUTHOR & PUBLISHER provide the PRODUCT under the DUAL LICENSE AGREEMENT
 * model designed to meet the needs of both non-commercial use and commercial
 * use.
 *
 * NON-COMMERCIAL USE means any use or activity where a fee is not charged
 * and the purpose is not the sale of a good or service, and the use or
 * activity is not intended to produce a profit. Examples of NON-COMMERCIAL USE
 * include:
 *
 * 1. Non-commercial open-source software.
 * 2. Non-commercial mobile applications.
 * 3. Non-commercial desktop software.
 * 4. Non-commercial web applications/solutions.
 * 5. Non-commercial use in research, scholarly and educational context.
 *
 * The above list is non-exhaustive and illustrative only.
 *
 * COMMERCIAL USE means any use or activity where a fee is charged or the
 * purpose is the sale of a good or service, or the use or activity is
 * intended to produce a profit. COMMERCIAL USE examples:
 *
 * 1. OEMs (Original Equipment Manufacturers).
 * 2. ISVs (Independent Software Vendors).
 * 3. VARs (Value Added Resellers).
 * 4. Other distributors that combine and distribute commercially licensed
 *    software.
 *
 * The above list is non-exhaustive and illustrative only.
 *
 * IN CASE YOU WANT TO USE THE PRODUCT COMMERCIALLY, YOU MUST PURCHASE THE
 * APPROPRIATE LICENSE FROM "INFIMA" ONLINE STORE, STORE ADDRESS:
 *
 * 1. https://mathparser.org/order-commercial-license
 * 2. https://payhip.com/infima
 *
 * NON-COMMERCIAL LICENSE
 *
 * Redistribution and use of the PRODUCT in source and/or binary forms,
 * with or without modification, are permitted provided that the following
 * conditions are met:
 *
 * 1. Redistributions of source code must retain the unmodified content of
 *    the entire MathParser.org-mXparser DUAL LICENSE AGREEMENT, including
 *    the definition of NON-COMMERCIAL USE, the definition of COMMERCIAL USE,
 *    the NON-COMMERCIAL LICENSE conditions, the COMMERCIAL LICENSE conditions,
 *    and the following DISCLAIMER.
 * 2. Redistributions in binary form must reproduce the entire content of
 *    MathParser.org-mXparser DUAL LICENSE AGREEMENT in the documentation
 *    and/or other materials provided with the distribution, including the
 *    definition of NON-COMMERCIAL USE, the definition of COMMERCIAL USE, the
 *    NON-COMMERCIAL LICENSE conditions, the COMMERCIAL LICENSE conditions,
 *    and the following DISCLAIMER.
 * 3. Any form of redistribution requires confirmation and signature of
 *    the NON-COMMERCIAL USE by successfully calling the method:
 *       License.iConfirmNonCommercialUse(...)
 *    The method call is used only internally for logging purposes, and
 *    there is no connection with other external services, and no data is
 *    sent or collected. The lack of a method call (or its successful call)
 *    does not affect the operation of the PRODUCT in any way. Please see
 *    the API documentation.
 *
 * COMMERCIAL LICENSE
 *
 *  1. Before purchasing a commercial license, the AUTHOR & PUBLISHER allow
 *     you to download, install, and use up to three copies of the PRODUCT to
 *     perform integration tests, confirm the quality of the PRODUCT, and
 *     its suitability. The testing period should be limited to fourteen
 *     days. Tests should be performed under the test environments conditions
 *     and not for profit generation.
 *  2. Provided that you purchased a license from "INFIMA" online store
 *     (store address: https://mathparser.org/order-commercial-license or
 *     https://payhip.com/infima), and you comply with all terms and
 *     conditions below, and you have acknowledged and understood the
 *     following DISCLAIMER, the AUTHOR & PUBLISHER grant you a nonexclusive
 *     license with the following rights:
 *  3. The license is granted only to you, the person or entity that made
 *     the purchase, identified and confirmed by the data provided during
 *     the purchase.
 *  4. If you purchased a license in the "ONE-TIME PURCHASE" model, the
 *     license is granted only for the PRODUCT version specified in the
 *     purchase. The upgrade policy gives you additional rights, described
 *     in the dedicated section below.
 *  5. If you purchased a license in the "SUBSCRIPTION" model, you may
 *     install and use any version of the PRODUCT during the subscription
 *     validity period.
 *  6. If you purchased a "SINGLE LICENSE" you may install and use the
 *     PRODUCT on/from one workstation that is located/accessible at/from
 *     any of your premises.
 *  7. Additional copies of the PRODUCT may be installed and used on/from
 *     more than one workstation, limited to the number of workstations
 *     purchased per order.
 *  8. If you purchased a "SITE LICENSE", the PRODUCT may be installed
 *     and used on/from all workstations located/accessible at/from any
 *     of your premises.
 *  9. You may incorporate the unmodified PRODUCT into your own products
 *     and software.
 * 10. If you purchased a license with the "SOURCE CODE" option, you may
 *     modify the PRODUCT's source code and incorporate the modified source
 *     code into your own products and/or software.
 * 11. Provided that the license validity period has not expired, you may
 *     distribute your product and/or software with the incorporated
 *     PRODUCT royalty-free.
 * 12. You may make copies of the PRODUCT for backup and archival purposes.
 * 13. Any form of redistribution requires confirmation and signature of
 *     the COMMERCIAL USE by successfully calling the method:
 *        License.iConfirmCommercialUse(...)
 *     The method call is used only internally for logging purposes, and
 *     there is no connection with other external services, and no data is
 *     sent or collected. The lack of a method call (or its successful call)
 *     does not affect the operation of the PRODUCT in any way. Please see
 *     the API documentation.
 * 14. The AUTHOR & PUBLISHER reserve all rights not expressly granted to
 *     you in this agreement.
 *
 * ADDITIONAL CLARIFICATION ON WORKSTATION
 *
 * A workstation is a device, a remote device, or a virtual device, used by
 * you, your employees, or other entities to whom you have commissioned
 * tasks. For example, the number of workstations may refer to the number
 * of software developers, engineers, architects, scientists, and other
 * professionals who use the PRODUCT on your behalf. The number of
 * workstations is not the number of copies of your end-product that you
 * distribute to your end-users.
 *
 * By purchasing the COMMERCIAL LICENSE, you only pay for the number of
 * workstations, while the number of copies/users of your final product
 * (delivered to your end-users) is not limited.
 *
 * Below are some examples to help you select the right license size:
 *
 * Example 1: Single Workstation License
 * Only one developer works on the development of your application. You do
 * not use separate environments for testing, meaning you design, create,
 * test, and compile your final application on one environment. In this
 * case, you need a license for a single workstation.
 *
 * Example 2: Up to 5 Workstations License
 * Two developers are working on the development of your application.
 * Additionally, one tester conducts tests in a separate environment.
 * You use three workstations in total, so you need a license for up to
 * five workstations.
 *
 * Example 3: Up to 20 Workstations License
 * Ten developers are working on the development of your application.
 * Additionally, five testers conduct tests in separate environments.
 * You use fifteen workstations in total, so you need a license for
 * up to twenty workstations.
 *
 * Example 4: Site License
 * Several dozen developers and testers work on the development of your
 * application using multiple environments. You have a large,
 * multi-disciplinary team involved in creating your solution. As your team
 * is growing and you want to avoid licensing limitations, the best choice
 * would be a site license.
 *
 * UPGRADE POLICY
 *
 * The PRODUCT is versioned according to the following convention:
 *
 *    [MAJOR].[MINOR].[PATCH]
 *
 * 1. COMMERCIAL LICENSE holders can install and use the updated version
 *    for bug fixes free of charge, i.e. if you have purchased a license
 *    for the [MAJOR].[MINOR] version (e.g., 5.0), you can freely install
 *    all releases specified in the [PATCH] version (e.g., 5.0.2).
 *    The license terms remain unchanged after the update.
 * 2. COMMERCIAL LICENSE holders for the [MAJOR].[MINOR] version (e.g., 5.0)
 *    can install and use the updated version [MAJOR].[MINOR + 1] free of
 *    charge, i.e., plus one release in the [MINOR] range (e.g., 5.1). The
 *    license terms remain unchanged after the update.
 * 3. COMMERCIAL LICENSE holders who wish to upgrade their version, but are
 *    not eligible for the free upgrade, can claim a discount when
 *    purchasing the upgrade. For this purpose, please contact us via e-mail.
 *
 * DISCLAIMER
 *
 * THIS PRODUCT IS PROVIDED BY THE AUTHOR & PUBLISHER "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL AUTHOR OR PUBLISHER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS PRODUCT, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE VIEWS AND CONCLUSIONS CONTAINED IN THE PRODUCT AND DOCUMENTATION ARE
 * THOSE OF THE AUTHORS AND SHOULD NOT BE INTERPRETED AS REPRESENTING
 * OFFICIAL POLICIES, EITHER EXPRESSED OR IMPLIED, OF THE AUTHOR OR PUBLISHER.
 *
 * CONTACT
 *
 * - e-mail: info@mathparser.org
 * - website: https://mathparser.org
 * - source code: https://github.com/mariuszgromada/MathParser.org-mXparser
 * - online store: https://mathparser.org/order-commercial-license
 * - online store: https://payhip.com/infima
 */
package org.mariuszgromada.math.mxparser.mathcollection;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

import org.mariuszgromada.math.mxparser.Argument;
import org.mariuszgromada.math.mxparser.Expression;
import org.mariuszgromada.math.mxparser.mXparser;

/**
 * MathFunctions - the most popular math functions. Many of function implemented by this class
 * could be found in java Math package (in fact functions from MathFunctions typically calls
 * original functions from the Math package). The reason why it was "re-implemented" is:
 * if you decide to implement your own function you do not need to change anything in the parser,
 * jut modify function implementation in this class.
 *
 * @author         <b>Mariusz Gromada</b><br>
 *                 <a href="https://mathparser.org" target="_blank">MathParser.org - mXparser project page</a><br>
 *                 <a href="https://github.com/mariuszgromada/MathParser.org-mXparser" target="_blank">mXparser on GitHub</a><br>
 *                 <a href="https://payhip.com/INFIMA" target="_blank">INFIMA place to purchase a commercial MathParser.org-mXparser software license</a><br>
 *                 <a href="mailto:info@mathparser.org">info@mathparser.org</a><br>
 *                 <a href="https://scalarmath.org/" target="_blank">ScalarMath.org - a powerful math engine and math scripting language</a><br>
 *                 <a href="https://play.google.com/store/apps/details?id=org.mathparser.scalar.lite" target="_blank">Scalar Lite</a><br>
 *                 <a href="https://play.google.com/store/apps/details?id=org.mathparser.scalar.pro" target="_blank">Scalar Pro</a><br>
 *                 <a href="https://mathspace.pl" target="_blank">MathSpace.pl</a><br>
 *
 * @version        6.1.0
 */
public final class MathFunctions {
	private static int MAX_RECURSION_CALLS = mXparser.getMaxAllowedRecursionDepth();
	private static final DecimalFormat DECIMAL_FORMAT = new DecimalFormat("0", DecimalFormatSymbols.getInstance(Locale.ENGLISH)) {{setMaximumFractionDigits(340);}};

	private static void refreshMaxAllowedRecursionDepth() {
		MAX_RECURSION_CALLS = mXparser.getMaxAllowedRecursionDepth();
	}
	/**
	 * Addition a + b applying canonical rounding if canonical
	 * rounding is enabled
	 *
	 * @param a  The a parameter
	 * @param b  The b parameter
	 * @return   The result of addition
	 */
	public static double plus(double a, double b) {
		if (Double.isNaN(a)) return Double.NaN;
		if (Double.isNaN(b)) return Double.NaN;
		if (!mXparser.checkIfCanonicalRounding()) return a + b;
		if (Double.isInfinite(a)) return a + b;
		if (Double.isInfinite(b)) return a + b;
		BigDecimal da = BigDecimal.valueOf(a);
		BigDecimal db = BigDecimal.valueOf(b);
		return da.add(db).doubleValue();
	}
	/**
	 * Subtraction a - b applying canonical rounding if canonical
	 * rounding is enabled
	 *
	 * @param a  The a parameter
	 * @param b  The b parameter
	 * @return   The result of subtraction
	 */
	public static double minus(double a, double b) {
		if (Double.isNaN(a)) return Double.NaN;
		if (Double.isNaN(b)) return Double.NaN;
		if (!mXparser.checkIfCanonicalRounding()) return a - b;
		if (Double.isInfinite(a)) return a - b;
		if (Double.isInfinite(b)) return a - b;
		BigDecimal da = BigDecimal.valueOf(a);
		BigDecimal db = BigDecimal.valueOf(b);
		return da.subtract(db).doubleValue();
	}
	/**
	 * Multiplication a * b applying canonical rounding if canonical
	 * rounding is enabled
	 *
	 * @param a  The a parameter
	 * @param b  The b parameter
	 * @return   The result of multiplication
	 */
	public static double multiply(double a, double b) {
		if (Double.isNaN(a)) return Double.NaN;
		if (Double.isNaN(b)) return Double.NaN;
		if (!mXparser.checkIfCanonicalRounding()) return a * b;
		if (Double.isInfinite(a)) return a * b;
		if (Double.isInfinite(b)) return a * b;
		BigDecimal da = BigDecimal.valueOf(a);
		BigDecimal db = BigDecimal.valueOf(b);
		return da.multiply(db).doubleValue();
	}
	/**
	 * Division a / b applying canonical rounding if canonical
	 * rounding is enabled
	 *
	 * @param a  The a parameter
	 * @param b  The b parameter
	 * @return   The result of division
	 */
	public static double div(double a, double b) {
		if (b == 0) return Double.NaN;
		if (Double.isNaN(a)) return Double.NaN;
		if (Double.isNaN(b)) return Double.NaN;
		if (!mXparser.checkIfCanonicalRounding()) return a / b;
		if (Double.isInfinite(a)) return a / b;
		if (Double.isInfinite(b)) return a / b;
		BigDecimal da = BigDecimal.valueOf(a);
		BigDecimal db = BigDecimal.valueOf(b);
		try {
			return da.divide(db, MathContext.DECIMAL128).doubleValue();
		} catch (Throwable e) {
			return a / b;
		}
	}
	/**
	 * Integer division (also called quotient) a / b applying canonical rounding if canonical
	 * rounding is enabled
	 *
	 * @param a  The a parameter
	 * @param b  The b parameter
	 * @return   The result of integer division
	 */
	public static double divQuotient(double a, double b) {
		double result = div(a, b);
		if (Double.isNaN(result) || Double.isInfinite(result)) return result;
		if (isAlmostInt(result)) return Math.round(result);
		return Math.floor(result);
	}
	/**
	 * Bell Numbers
	 *
	 * @param      n                   the n
	 *
	 * @return     if n &gt;= 0 returns Bell numbers,
	 *             otherwise returns Double.NaN.
	 */
	public static double bellNumber(int n) {
		double result = Double.NaN;
		if (n > 1) {
			n -= 1;
			if ( (n+1)*(n+1) >= Integer.MAX_VALUE ) return Double.NaN;
			long[][] bellTriangle = new long[n+1][n+1];
			bellTriangle[0][0] = 1;
			bellTriangle[1][0] = 1;
			for (int r = 1; r <= n; r++) {
				for (int k = 0; k < r; k++)
					bellTriangle[r][k+1] = bellTriangle[r-1][k] + bellTriangle[r][k];
				if (r < n)
					bellTriangle[r+1][0] = bellTriangle[r][r];
				if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
			}
			result = bellTriangle[n][n];
		} else if (n >= 0)
			result = 1;
		return result;
	}
	/**
	 * Bell number
	 * @param      n                   the n
	 *
	 * @return     if n &lt;&gt; Double.NaN return bellNumber( (int)Math.round(n) ),
	 *             otherwise return Double.NaN.
	 */
	public static double bellNumber(double n) {
		if (Double.isNaN(n))
			return Double.NaN;
		return bellNumber( (int)Math.round(n) );
	}
	/**
	 * Euler numbers
	 *
	 * @param      n                   the n function param
	 * @param      k                   the k function param
	 *
	 * @return     if n &gt;=0 returns Euler number,
	 *             otherwise return Double.NaN.
	 *             Returns also Double.NaN when MAX RECURSION CALLS
	 *             is exceeded.
	 *
	 * @see mXparser#getMaxAllowedRecursionDepth()
	 * @see mXparser#setMaxAllowedRecursionDepth(int)
	 */
	public static double eulerNumber(int n, int k) {
		refreshMaxAllowedRecursionDepth();
		return eulerNumber(n, k, 1);
	}
	private static double eulerNumber(int n, int k, int recursionCall) {
		if (recursionCall > MAX_RECURSION_CALLS)
			return Double.NaN;
		if (n < 0)
			return Double.NaN;
		if (k < 0)
			return 0;
		if (n == 0)
			if (k == 0)
				return 1;
			else
				return 0;
		if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		double e1 = eulerNumber(n - 1, k, recursionCall+1);
		if (Double.isNaN(e1))
			return Double.NaN;
		double e2 = eulerNumber(n - 1, k - 1, recursionCall + 1);
		if (Double.isNaN(e2))
			return Double.NaN;
		return (k + 1) * e1 + (n - k) * e2;
	}

	/**
	 * Euler numbers
	 *
	 * @param      n                   the n function param
	 * @param      k                   the k function param
	 *
	 * @return     if n, k &lt;&gt; Double.NaN returns eulerNumber( (int)Math.round(n), (int)Math.round(k) ),
	 *             otherwise return Double.NaN.
	 */
	public static double eulerNumber(double n, double k) {
		if (Double.isNaN(n) || Double.isNaN(k))
			return Double.NaN;
		return eulerNumber( (int)Math.round(n), (int)Math.round(k) );
	}
	/**
	 * Factorial
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     Factorial if n &gt;=0, otherwise returns Double.NaN.
	 */
	public static double factorial(int n) {
		double f = Double.NaN;
		if (n >= 0)
			if (n < 2) f = 1;
			else {
				f = 1;
				for (int i = 1; i <= n; i++) {
					f = f*i;
					if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
				}
			}
		return f;
	}
	/**
	 * Factorial
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &lt;&gt; Double.NaN return factorial( (int)Math.round(n) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double factorial(double n) {
		if (Double.isNaN(n))
			return Double.NaN;
		return factorial( (int)Math.round(n) );
	}
	/**
	 * Falling factorial polynomial
	 * @param x  Argument
	 * @param n  Polynomial degree
	 * @return   Falling factorial polynomial of degree n at point x
	 */
	public static double factorialFalling(double x, double n){
		if (Double.isNaN(x)) return Double.NaN;
		if (Double.isNaN(n)) return Double.NaN;
		if (n < 0) return Double.NaN;
		if (BinaryRelations.isEqualOrAlmost(n, 0)) return 1.0;
		double k, y;
		y = 1;
		for (k = 0; k <= n - 1; k = k + 1) {
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
			y = y * (x - k);
		}
		return y;
	}
	/**
	 * Rising factorial polynomial
	 * @param x  Argument
	 * @param n  Polynomial degree
	 * @return   Rising factorial polynomial of degree n at point x
	 */
	public static double factorialRising(double x, double n){
		if (Double.isNaN(x)) return Double.NaN;
		if (Double.isNaN(n)) return Double.NaN;
		if (n < 0) return Double.NaN;
		if (BinaryRelations.isEqualOrAlmost(n, 0)) return 1.0;
		double k, y;
		y = 1;
		for (k = 0; k <= n - 1; k = k + 1) {
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
			y = y * (x + k);
		}
		return y;
	}
	/**
	 * Generalized binomial coefficient
	 *
	 * @param      n                   the n function parameter
	 * @param k    k                   the k function parameter
	 *
	 * @return     Generalized binomial coefficient, if
	 *             n = Double.NaN or k &lt; 0 returns Double.NaN.
	 */
	public static double binomCoeff(double n, long k) {
		if (Double.isNaN(n))
			return Double.NaN;
		double result = Double.NaN;
		if ( k >= 0 ){
			double numerator = 1;
			if (k > 0 )
				for (long i = 0; i <= k-1; i++) {
					numerator*=(n-i);
					if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
				}
			double denominator = 1;
			if ( k > 1 )
				for (long i = 1; i <= k; i++) {
					denominator *= i;
					if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
				}
			result = numerator / denominator;
		}
		return result;
	}
	/**
	 * Generalized binomial coefficient
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return     if n, k &lt;&gt; Double.NaN returns binomCoeff(n, (int)Math.round(k) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double binomCoeff(double n, double k) {
		if (Double.isNaN(n) || Double.isNaN(k))
			return Double.NaN;
		return binomCoeff(n, Math.round(k));
	}
	/**
	 * Generalized coefficient returning number of k permutations
	 * that can be drawn for n elements set.
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return   For k greater than 0 return number of permutations, otherwise
	 *           returns Double.NaN
	 */
	public static double numberOfPermutations(double n, long k) {
		if (Double.isNaN(n))
			return Double.NaN;
		double result = Double.NaN;
		if ( k >= 0 ){
			double numerator = 1;
			if (k > 0 )
				for (long i = 0; i <= k-1; i++) {
					numerator*=(n-i);
					if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
				}
			result = numerator;
		}
		return result;
	}
	/**
	 * Generalized coefficient returning number of k permutations
	 * that can be drawn for n elements set.
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return   For k greater than 0 return number of permutations, otherwise
	 *           returns Double.NaN
	 */
	public static double numberOfPermutations(double n, double k) {
		if (Double.isNaN(n) || Double.isNaN(k))
			return Double.NaN;
		return numberOfPermutations(n, Math.round(k));
	}
	/**
	 * Bernoulli numbers
	 *
	 * @param      m                   the m function parameter
	 * @param      n                   the n function parameter
	 *
	 * @return     if n, m &gt;= 0 returns Bernoulli number,
	 *             otherwise returns Double.NaN.
	 */
	public static double bernoulliNumber(int m, int n) {
		double result = Double.NaN;
		if ( (m >= 0) && (n >= 0) ) {
			result = 0;
			for (int k = 0; k <= m; k++)
				for (int v = 0; v <= k; v++) {
					result += Math.pow(-1, v) * binomCoeff(k, v)
						* ( Math.pow(n + v, m) / (k + 1) );
					if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
				}
		}
		return result;
	}
	/**
	 * Bernoulli numbers
	 *
	 * @param      m                   the m function parameter
	 * @param      n                   the n function parameter
	 *
	 * @return     if n, m &lt;&gt; Double.NaN returns bernoulliNumber( (int)Math.round(m), (int)Math.round(n) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double bernoulliNumber(double m, double n) {
		if (Double.isNaN(m) || Double.isNaN(n))
			return Double.NaN;
		return bernoulliNumber( (int)Math.round(m), (int)Math.round(n) );
	}
	/**
	 * Stirling numbers of the first kind
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return     Stirling numbers of the first kind
	 *             Returns also Double.NaN when MAX RECURSION CALLS
	 *             is exceeded.
	 *
	 * @see mXparser#getMaxAllowedRecursionDepth()
	 * @see mXparser#setMaxAllowedRecursionDepth(int)
	 */
	public static double Stirling1Number(int n, int k) {
		refreshMaxAllowedRecursionDepth();
		return Stirling1Number(n, k, 1);
	}
	private static double Stirling1Number(int n, int k, int recursionCall) {
		if (recursionCall > MAX_RECURSION_CALLS)
			return Double.NaN;
		if (k > n) return 0;
		if (n == 0)
			if (k == 0) return 1;
			else return 0;
		if (k == 0) return 0;

		if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		double s1 = Stirling1Number(n - 1, k, recursionCall + 1);
		if (Double.isNaN(s1))
			return Double.NaN;
		double s2 = Stirling1Number(n - 1, k - 1, recursionCall + 1);
		if (Double.isNaN(s2))
			return Double.NaN;
		return (n - 1) * s1 + s2;
	}
	/**
	 * Stirling numbers of the first kind
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return     if n, k &lt;&gt; Doube.NaN returns Stirling1Number( (int)Math.round(n), (int)Math.round(k) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double Stirling1Number(double n, double k) {
		if (Double.isNaN(n) || Double.isNaN(k))
			return Double.NaN;
		return Stirling1Number( (int)Math.round(n), (int)Math.round(k) );
	}
	/**
	 * Stirling numbers of the second kind
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return     Stirling numbers of the second kind
	 *             Returns also Double.NaN when MAX RECURSION CALLS
	 *             is exceeded.
	 *
	 * @see mXparser#getMaxAllowedRecursionDepth()
	 * @see mXparser#setMaxAllowedRecursionDepth(int)
	 */
	public static double Stirling2Number(int n, int k) {
		refreshMaxAllowedRecursionDepth();
		return Stirling2Number(n, k, 1);
	}
	private static double Stirling2Number(int n, int k, int recursionCall) {
		if (recursionCall > MAX_RECURSION_CALLS)
			return Double.NaN;
		if (k > n) return 0;
		if (n == 0)
			if (k == 0) return 1;
			else return 0;
		if (k == 0) return 0;

		if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		double s1 = Stirling2Number(n - 1, k, recursionCall + 1);
		if (Double.isNaN(s1))
			return Double.NaN;
		double s2 = Stirling2Number(n - 1, k - 1, recursionCall + 1);
		if (Double.isNaN(s2))
			return Double.NaN;
		return k * s1 + s2;
	}
	/**
	 * Stirling numbers of the second kind
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return     if n, k &lt;&gt; Doube.NaN returns Stirling2Number( (int)Math.round(n), (int)Math.round(k) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double Stirling2Number(double n, double k) {
		if (Double.isNaN(n) || Double.isNaN(k))
			return Double.NaN;
		return Stirling2Number( (int)Math.round(n), (int)Math.round(k) );
	}
	/**
	 * Worpitzky numbers
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return     if n,k &gt;= 0 and k &lt;= n return Worpitzky number,
	 *             otherwise return Double.NaN.
	 */
	public static double worpitzkyNumber(int n, int k) {
		double result = Double.NaN;
		if (k >= 0 && k <= n) {
			result = 0;
			for (int v = 0; v <= k; v++) {
				result += Math.pow(-1, v+k) * Math.pow(v+1, n) * binomCoeff(k, v);
				if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
			}
		}
		return result;
	}
	/**
	 * Worpitzky numbers
	 *
	 * @param      n                   the n function parameter
	 * @param      k                   the k function parameter
	 *
	 * @return     if n,k &lt;&gt; Double.NaN returns worpitzkyNumber( (int)Math.round(n), (int)Math.round(k) ),
	 *             otherwise return Double.NaN.
	 */
	public static double worpitzkyNumber(double n, double k) {
		if (Double.isNaN(n) || Double.isNaN(k))
			return Double.NaN;
		return worpitzkyNumber( (int)Math.round(n), (int)Math.round(k) );
	}
	/**
	 * Harmonic numer
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &gt; 0 returns harmonic number, otherwise returns 0
	 *             (empty summation operator)
	 */
	public static double harmonicNumber(int n) {
		if (n <= 0)
			return 0;
		if (n == 1)
			return 1;
		double h = 1;
		for (double k = 2.0; k <= n; k++) {
			h += 1.0 / k;
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		}
		return h;
	}
	/**
	 * Harmonic number
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &lt;&gt; Double.NaN returns harmonicNumber( (int)Math.round(n) ),
	 *             otherwise returns Double.NaN
	 */
	public static double harmonicNumber(double n) {
		if (Double.isNaN(n))
			return Double.NaN;
		return harmonicNumber( (int)Math.round(n) );
	}
	/**
	 * Harmonic number 1/1 + 1/2^x + ... + 1/n^x
	 *
	 * @param      x                   the x function parameter
	 * @param      n                   the n function parameter
	 *
	 * @return     if x &lt;&gt; Double.NaN and x &gt;= 0 Harmonic number,
	 *             otherwise returns Double.NaN.
	 */
	public static double harmonicNumber(double x, int n) {
		if  ( (Double.isNaN(x)) || (x < 0) )
			return Double.NaN;
		if (n <= 0)
			return 0;
		if (n == 1)
			return x;
		double h = 1;
		for (double k = 2.0; k <= n; k++) {
			h += 1 / power(k, x);
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		}
		return h;
	}
	/**
	 * Harmonic number 1/1 + 1/2^x + ... + 1/n^x
	 *
	 * @param      x                   the x function parameter
	 * @param      n                   the n function parameter
	 *
	 * @return     if x,n &lt;&gt; Double.NaN returns harmonicNumber( x, (int)Math.round(n) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double harmonicNumber(double x, double n) {
		if ( (Double.isNaN(x)) || (Double.isNaN(n)) )
			return Double.NaN;
		return harmonicNumber( x, (int)Math.round(n) );
	}
	/**
	 * Catalan numbers
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     Catalan numbers
	 */
	public static double catalanNumber(int n) {
		return binomCoeff(2*n, n) * div(1, n+1);
	}
	/**
	 * Catalan numbers
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &lt;&gt; Double.NaN returns catalanNumber( (int)Math.round(n) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double catalanNumber(double n) {
		if (Double.isNaN(n))
			return Double.NaN;
		return catalanNumber( (int)Math.round(n) );
	}
	/**
	 * Fibonacci numbers
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &gt;= 0 returns fibonacci numbers,
	 *             otherwise returns Double.NaN.
	 *             Returns also Double.NaN when MAX RECURSION CALLS
	 *             is exceeded.
	 *
	 * @see mXparser#getMaxAllowedRecursionDepth()
	 * @see mXparser#setMaxAllowedRecursionDepth(int)
	 */
	public static double fibonacciNumber(int n) {
		refreshMaxAllowedRecursionDepth();
		return fibonacciNumber(n, 1);
	}
	private static double fibonacciNumber(int n, int recursionCall) {
		if (recursionCall > MAX_RECURSION_CALLS)
			return Double.NaN;
		if (n < 0)
			return Double.NaN;
		if (n == 0)
			return 0;
		if (n == 1)
			return 1;
		if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		double f1 = fibonacciNumber(n - 1, recursionCall + 1);
		if (Double.isNaN(f1))
			return Double.NaN;
		double f2 = fibonacciNumber(n - 2, recursionCall + 1);
		if (Double.isNaN(f2))
			return Double.NaN;
		return f1 + f2;
	}
	/**
	 * Fibonacci numbers
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &lt;&gt; Double.NaN returns fibonacciNumber( (int)Math.round(n) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double fibonacciNumber(double n) {
		if (Double.isNaN(n))
			return Double.NaN;
		return fibonacciNumber( (int)Math.round(n) );
	}
	/**
	 * Lucas numebrs
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &gt;= 0 returns Lucas numbers,
	 *             otherwise returns Double.NaN.
	 *             Returns also Double.NaN when MAX RECURSION CALLS
	 *             is exceeded.
	 *
	 * @see mXparser#getMaxAllowedRecursionDepth()
	 * @see mXparser#setMaxAllowedRecursionDepth(int)
	 */
	public static double lucasNumber(int n) {
		refreshMaxAllowedRecursionDepth();
		return lucasNumber(n, 1);
	}
	private static double lucasNumber(int n, int recursionCall) {
		if (recursionCall > MAX_RECURSION_CALLS)
			return Double.NaN;
		if (n < 0)
			return Double.NaN;
		if (n == 0)
			return 2;
		if (n == 1)
			return 1;
		if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		double l1 = lucasNumber(n - 1, recursionCall + 1);
		if (Double.isNaN(l1))
			return Double.NaN;
		double l2 = lucasNumber(n - 2, recursionCall + 1);
		if (Double.isNaN(l2))
			return Double.NaN;
		return l1 + l2;
	}
	/**
	 * Lucas numebrs
	 *
	 * @param      n                   the n function parameter
	 *
	 * @return     if n &lt;&gt; Double.NaN returns lucasNumber( (int)Math.round(n) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double lucasNumber(double n) {
		if (Double.isNaN(n))
			return Double.NaN;
		return lucasNumber( (int)Math.round(n) );
	}
	/**
	 * Kronecker delta
	 *
	 * @param      i                   the i function parameter
	 * @param      j                   the j function parameter
	 *
	 * @return     if i,j &lt;&gt; Double.NaN returns Kronecker delta,
	 *             otherwise returns Double.NaN.
	 */
	public static double kroneckerDelta(double i, double j) {
		if (Double.isNaN(i) || Double.isNaN(j))
			return Double.NaN;
		if (i == j)
			return 1;
		else
			return 0;
	}
	/**
	 * Kronecker delta
	 *
	 * @param      i                   the i function parameter
	 * @param      j                   the j function parameter
	 *
	 * @return     Kronecker delta
	 */
	public static double kroneckerDelta(int i, int j) {
		if (i == j)
			return 1;
		else
			return 0;
	}
	/**
	 * Continued fraction
	 *
	 * @param      sequence            the numbers
	 *
	 * @return     if each number form the sequence &lt;&gt; Double.NaN and
	 *             there is no division by 0 while computing returns continued fraction
	 *             value, otherwise returns Double.NaN.
	 */
	public static double continuedFraction(double... sequence) {
		if (sequence == null) return Double.NaN;
		if (sequence.length == 0) return Double.NaN;
		double cf = 0;
		double a;
		if (sequence.length == 1)
			return sequence[0];
		int lastIndex = sequence.length-1;
		for(int i = lastIndex; i >= 0; i--) {
			a = sequence[i];
			if (Double.isNaN(a))
				return Double.NaN;
			if (i == lastIndex) {
				cf = a;
			} else {
				if (cf == 0)
					return Double.NaN;
				cf = a + 1.0 / cf;
			}
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		}
		return cf;
	}
	/**
	 * Private function calculating continued polynomial
	 * recursively.
	 *
	 * @param      n         the polynomial order
	 * @param      x         the x values
	 *
	 * @return     continued polynomial value
	 */
	private static double continuedPolynomial(int n, double[] x) {
		if (x == null) return Double.NaN;
		if (x.length == 0) return Double.NaN;
		if (n == 0)
			return 1;
		if (n == 1)
			return x[0];
		if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		return x[n-1] * continuedPolynomial(n-1, x) + continuedPolynomial(n-2, x);
	}
	/**
	 * Continued polynomial
	 *
	 * @param      x         the x values
	 *
	 * @return     if each number for x is different the Double.NaN
	 *             returns continued polynomial, otherwise returns
	 *             Double.NaN.
	 */
	public static double continuedPolynomial(double... x) {
		if (x == null) return Double.NaN;
		if (x.length == 0) return Double.NaN;
		for (double d : x) {
			if (Double.isNaN(d))
				return Double.NaN;
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		}
		return continuedPolynomial(x.length, x);
	}
	/**
	 * Euler polynomial
	 *
	 * @param      m                   the m parameter
	 * @param      x                   the x parameter
	 *
	 * @return     if x &lt;&gt; Double.NaN and m &gt;= 0 returns polynomial value,
	 *             otherwise returns Double.NaN.
	 */
	public static double eulerPolynomial(int m, double x) {
		if (Double.isNaN(x))
			return Double.NaN;
		double sumTotal = Double.NaN;
		if (m >= 0) {
			sumTotal = 0;
			for (int n = 0; n <= m; n++) {
				double sumPartial = 0;
				for (int k = 0; k <= n; k++) {
					sumPartial += Math.pow(-1, k) * binomCoeff(n, k) * Math.pow(x+k, m);
					if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
				}
				sumPartial /= Math.pow(2, n);
				sumTotal += sumPartial;
			}
		}
		return sumTotal;
	}
	/**
	 * Euler polynomial
	 *
	 * @param      m                   the m parameter
	 * @param      x                   the x parameter
	 *
	 * @return     if x,m &lt;&gt; Double.NaN returns eulerPolynomial( (int)Math.round(m), (int)Math.round(x) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double eulerPolynomial(double m, double x) {
		if (Double.isNaN(m) || Double.isNaN(x))
			return Double.NaN;
		return eulerPolynomial( (int)Math.round(m), x);
	}
	/**
	 * Characteristic function x in (a,b)
	 *
	 * @param      x                   the x value
	 * @param      a                   the left (lower) limit
	 * @param      b                   the right (upper) limit
	 *
	 * @return     if x, a, b &lt;&gt; Double.NaN returns
	 * 			   characteristic function value on the (a,b) range.
	 */
	public static double chi(double x, double a, double b) {
		if (Double.isNaN(x) || Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		double result = Double.NaN;
		if ( (!Double.isNaN(x)) && (!Double.isNaN(a)) && (!Double.isNaN(b)) )
			if ( (x > a) && (x < b) )
				result = 1;
			else
				result = 0;
		return result;
	}
	/**
	 * Characteristic function x in [a,b]
	 *
	 * @param      x                   the x value
	 * @param      a                   the left (lower) limit
	 * @param      b                   the right (upper) limit
	 *
	 * @return     if x, a, b &lt;&gt; Double.NaN returns
	 * 			   characteristic function value on the [a,b] range.
	 */
	public static double chi_LR(double x, double a, double b) {
		if (Double.isNaN(x) || Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		double result = Double.NaN;
		if ( (!Double.isNaN(x)) && (!Double.isNaN(a)) && (!Double.isNaN(b)) )
			if ( (x >= a) && (x <= b) )
				result = 1;
			else
				result = 0;
		return result;
	}
	/**
	 * Characteristic function x in [a,b)
	 *
	 * @param      x                   the x value
	 * @param      a                   the left (lower) limit
	 * @param      b                   the right (upper) limit
	 *
	 * @return     if x, a, b &lt;&gt; Double.NaN returns
	 * 			   characteristic function value on the [a,b) range.
	 */
	public static double chi_L(double x, double a, double b) {
		if (Double.isNaN(x) || Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		double result = Double.NaN;
		if ( (!Double.isNaN(x)) && (!Double.isNaN(a)) && (!Double.isNaN(b)) )
			if ( (x >= a) && (x < b) )
				result = 1;
			else
				result = 0;
		return result;
	}
	/**
	 * Characteristic function x in (a,b]
	 *
	 * @param      x                   the x value
	 * @param      a                   the left (lower) limit
	 * @param      b                   the right (upper) limit
	 *
	 * @return     if x, a, b &lt;&gt; Double.NaN returns
	 * 			   characteristic function value on the (a,b] range.
	 */
	public static double chi_R(double x, double a, double b) {
		if (Double.isNaN(x) || Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		double result = Double.NaN;
		if ( (!Double.isNaN(x)) && (!Double.isNaN(a)) && (!Double.isNaN(b)) )
			if ( (x > a) && (x <= b) )
				result = 1;
			else
				result = 0;
		return result;
	}
	/**
	 * Verifies whether provided number is almost integer
	 *
	 * @see BinaryRelations#DEFAULT_COMPARISON_EPSILON
	 *
	 * @param a  The number to be verified
	 * @return   True if the number is almost integer according to the default epsilon,
	 *           otherwise returns false.
	 */
	public static boolean isAlmostInt(double a) {
		double aint = Math.round(a);
		return abs(a - aint) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON;
	}
	/**
	 * Applies the integer exponent to the base a
	 *
	 * @param a   The base
	 * @param n   The integer exponent
	 * @return    Return a to the power of n, if canonical rounding is enable, the it operates on big numbers
	 */
	private static double powInt(double a, int n) {
		if (Double.isNaN(a)) return Double.NaN;
		if (Double.isInfinite(a)) return Math.pow(a, n);
		if (a == 0) return Math.pow(a, n);
		if (n == 0) return 1;
		if (n == 1) return a;
		if (mXparser.checkIfCanonicalRounding()) {
			BigDecimal da = BigDecimal.valueOf(a);
			try {
				if (n >= 0) return da.pow(n).doubleValue();
				else return BigDecimal.ONE.divide(da, MathContext.DECIMAL128).pow(-n).doubleValue();
			} catch (Throwable e) {
				return Math.pow(a, n);
			}
		} else {
			return Math.pow(a, n);
		}
	}
	/**
	 * Power function a^b
	 *
	 * @param      a                   the a function parameter
	 * @param      b                   the b function parameter
	 *
	 * @return     if a,b &lt;&gt; Double.NaN returns Math.pow(a, b),
	 *             otherwise returns Double.NaN.
	 */
	public static double power(double a, double b) {
		if (Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		if (Double.isInfinite(a)) return Math.pow(a, b);
		if (Double.isInfinite(b)) return Math.pow(a, b);
		double babs = Math.abs(b);
		double bint = Math.round(babs);
		if ( MathFunctions.abs(babs - bint) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON
				&& babs < Integer.MAX_VALUE && -babs > Integer.MIN_VALUE) {
			if (b >= 0) return powInt(a, (int)bint);
			else return powInt(a, -(int)bint);
		} else if (a >= 0)
			return Math.pow(a, b);
		else if (abs(b) >= 1)
			return Math.pow(a, b);
		else if (b == 0)
			return Math.pow(a, b);
		else {
			double ndob = 1.0 / abs(b);
			double nint = Math.round(ndob);
			if ( MathFunctions.abs(ndob-nint) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON ) {
				long n = (long)nint;
				if (n % 2 == 1)
					if (b > 0)
						return -Math.pow( abs(a), 1.0 / ndob);
					else
						return -Math.pow( abs(a), -1.0 / ndob);
				else
					return Double.NaN;
			} else return Double.NaN;
		}
	}
	/**
	 * Nth order root of a number
	 *
	 * @param n   Root order
	 * @param x   Number
	 * @return    Returns root of a number. If calculation is not possible Double.NaN is returned.
	 */
	public static double root(double n, double x) {
		if (Double.isNaN(n) || Double.isNaN(x)) return Double.NaN;
		if (Double.isInfinite(n) || Double.isInfinite(x)) return Double.NaN;
		if (n < -BinaryRelations.DEFAULT_COMPARISON_EPSILON) return Double.NaN;
		if (abs(n) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON) {
			if (abs(x) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON) return 0;
			else if (abs(x-1) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON) return 1;
			else return Double.NaN;
		}
		long nint = (long)floor(n);
		if (nint == 1) return x;
		if (nint == 2) return sqrt(x);
		if (nint % 2 == 1) {
			if ( x >= 0) return Math.pow(x, 1.0 / nint);
			else return -Math.pow( abs(x), 1.0 / nint);
		} else {
			if ( x >= 0) return Math.pow(x, 1.0 / nint);
			else return Double.NaN;
		}
	}
	/**
	 * Tetration, exponential power, power series
	 *
	 * @param a   base
	 * @param n   exponent
	 * @return    Tetration result.
	 */
	public static double tetration(double a, double n) {
		if (Double.isNaN(a)) return Double.NaN;
		if (Double.isNaN(n)) return Double.NaN;
		if (n == Double.POSITIVE_INFINITY) {
			if (BinaryRelations.isEqualOrAlmost(a, 1)) return 1.0;
			if (abs(a - MathConstants.EXP_MINUS_E) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON)
				return MathConstants.EXP_MINUS_1;
			if (abs(a - MathConstants.EXP_1_OVER_E) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON)
				return MathConstants.E;
			if ((a > MathConstants.EXP_MINUS_E) && (a < MathConstants.EXP_1_OVER_E))
				return SpecialFunctions.lambertW( -MathFunctions.ln(a), 0) / ( -MathFunctions.ln(a) );
			if (a > MathConstants.EXP_1_OVER_E) return Double.POSITIVE_INFINITY;
			if (a < MathConstants.EXP_MINUS_E) return Double.NaN;
		}
		if (n < -BinaryRelations.DEFAULT_COMPARISON_EPSILON) return Double.NaN;
		if (abs(n) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON) {
			if (abs(a) > BinaryRelations.DEFAULT_COMPARISON_EPSILON)
				return 1;
			else
				return Double.NaN;
		}
		n = floor(n);
		if (n == 0) {
			if (abs(a) > BinaryRelations.DEFAULT_COMPARISON_EPSILON)
				return 1;
			else
				return Double.NaN;
		}
		if (abs(a) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON) return 0;
		if (n == 1) return a;
		double r = a;
		for (double i = 2; i <= n; i++) {
			r = Math.pow(a, r);
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		}
		return r;
	}
	/**
	 * Modulo operator a % b
	 *
	 * @param      a                   the a function parameter
	 * @param      b                   the b function parameter
	 *
	 * @return     if a,b &lt;&gt; Double.NaN returns a % b.
	 */
	public static double mod(double a, double b) {
		if (Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		return a % b;
	}
	/**
	 * Division a/b
	 *
	 * @param      a                   the a function parameter
	 * @param      b                   the b function parameter
	 *
	 * @return     if a,b &lt;&gt; Double.NaN and b &lt;&gt; 0 returns a/b,
	 *             otherwise return Double.NaN.
	 */
	/*
	public static final double div(double a, double b) {
		if (Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		double result = Double.NaN;
		if (b != 0)
			result = a / b;
		return result;
	}
	*/
	/**
	 * Sine trigonometric function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN return Math.sin(a),
	 *             otherwise return Double.NaN.
	 */
	public static double sin(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		if (mXparser.checkIfDegreesMode())
			a = a * Units.DEGREE_ARC;
		SpecialValueTrigonometric sv = SpecialValueTrigonometric.getSpecialValueTrigonometric(a);
		if (sv != null)
			return sv.sin;
		return Math.sin(a);
	}
	/**
	 * Cosine trigonometric function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.cos(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double cos(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		if (mXparser.checkIfDegreesMode())
			a = a * Units.DEGREE_ARC;
		SpecialValueTrigonometric sv = SpecialValueTrigonometric.getSpecialValueTrigonometric(a);
		if (sv != null)
			return sv.cos;
		return Math.cos(a);
	}
	/**
	 * Tangent trigonometric function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.tan(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double tan(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		if (mXparser.checkIfDegreesMode())
			a = a * Units.DEGREE_ARC;
		SpecialValueTrigonometric sv = SpecialValueTrigonometric.getSpecialValueTrigonometric(a);
		if (sv != null)
			return sv.tan;
		return Math.tan(a);
	}
	/**
	 * Cotangent trigonometric function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and tan(a) &lt;&gt; 0 returns 1 / Math.tan(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double ctan(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		if (mXparser.checkIfDegreesMode())
			a = a * Units.DEGREE_ARC;
		SpecialValueTrigonometric sv = SpecialValueTrigonometric.getSpecialValueTrigonometric(a);
		if (sv != null)
			return sv.ctan;
		double result = Double.NaN;
		double tg = Math.tan(a);
		if (tg != 0)
			result = 1.0 / tg;
		return result;
	}
	/**
	 * Secant trigonometric function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and cos(a) &lt;&gt; 0 returns 1 / Math.cos(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double sec(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		if (mXparser.checkIfDegreesMode())
			a = a * Units.DEGREE_ARC;
		SpecialValueTrigonometric sv = SpecialValueTrigonometric.getSpecialValueTrigonometric(a);
		if (sv != null)
			return sv.sec;
		double result = Double.NaN;
		double cos = Math.cos(a);
		if (cos != 0)
			result = 1.0 / cos;
		return result;
	}
	/**
	 * Cosecant trigonometric function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and sin(a) &lt;&gt; 0 returns 1 / Math.sin(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double cosec(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		if (mXparser.checkIfDegreesMode())
			a = a * Units.DEGREE_ARC;
		SpecialValueTrigonometric sv = SpecialValueTrigonometric.getSpecialValueTrigonometric(a);
		if (sv != null)
			return sv.csc;
		double result = Double.NaN;
		double sin = Math.sin(a);
		if (sin != 0)
			result = 1.0 / sin;
		return result;
	}
	/**
	 * If double is almost integer returns the closes integer, otherwise original value
	 * @param val   Parameter
	 * @return      f double is almost integer returns the closest integer, otherwise original value
	 */
	private static double intIfAlmostIntOtherwiseOrig(double val) {
		double valint = Math.round(val);
		if ( Math.abs(val-valint) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON ) return valint;
		return val;
	}
	/**
	 * Arcus sine - inverse trigonometric sine function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.asin(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double asin(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		SpecialValue sv = SpecialValueTrigonometric.getSpecialValueAsin(a);
		double r;
		if (sv != null) r = sv.fv;
		else r = Math.asin(a);
		if (mXparser.checkIfDegreesMode()) {
			if (sv != null) return sv.fvdeg;
			return intIfAlmostIntOtherwiseOrig(div(r, Units.DEGREE_ARC));
		} else return r;
	}
	/**
	 * Arcus cosine - inverse trigonometric cosine function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.acos(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double acos(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		SpecialValue sv = SpecialValueTrigonometric.getSpecialValueAcos(a);
		double r;
		if (sv != null) r = sv.fv;
		else r = Math.acos(a);
		if (mXparser.checkIfDegreesMode()) {
			if (sv != null) return sv.fvdeg;
			return intIfAlmostIntOtherwiseOrig(div(r, Units.DEGREE_ARC));
		} else return r;
	}
	/**
	 * Arcus tangent - inverse trigonometric tangent function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.atan(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double atan(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		SpecialValue sv = SpecialValueTrigonometric.getSpecialValueAtan(a);
		double r;
		if (sv != null) r = sv.fv;
		else r = Math.atan(a);
		if (mXparser.checkIfDegreesMode()) {
			if (sv != null) return sv.fvdeg;
			return intIfAlmostIntOtherwiseOrig(div(r, Units.DEGREE_ARC));
		}
		else return r;
	}
	/**
	 * Arcus cotangent - inverse trigonometric cotangent function
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and a &lt;&gt; 0 returns Math.atan(1/a),
	 *             otherwise returns Double.NaN.
	 */
	public static double actan(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		SpecialValue sv = SpecialValueTrigonometric.getSpecialValueActan(a);
		double r;
		if (sv != null) r = sv.fv;
		else {
			if (a > 0) r = Math.atan(1/a);
			else if (a < 0) r = Math.atan(1/a) + MathConstants.PI;
			else r = Double.NaN;
		}
		if (mXparser.checkIfDegreesMode()) {
			if (sv != null) return sv.fvdeg;
			return intIfAlmostIntOtherwiseOrig(div(r, Units.DEGREE_ARC));
		}
		else return r;
	}
	/**
	 * Arcus secant - inverse trigonometric secant function
	 *
	 * @param      a                   the a function parameter
	 * @return     Inverse trigonometric secant function
	 */
	public static double asec(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		SpecialValue sv = SpecialValueTrigonometric.getSpecialValueAsec(a);
		double r;
		if (sv != null) r = sv.fv;
		else r = Math.acos(1/a);
		if (mXparser.checkIfDegreesMode()) {
			if (sv != null) return sv.fvdeg;
			return intIfAlmostIntOtherwiseOrig(div(r, Units.DEGREE_ARC));
		}
		else return r;
	}
	/**
	 * Arcus cosecant - inverse trigonometric cosecant function
	 *
	 * @param      a                   the a function parameter
	 * @return     Inverse trigonometric cosecant function
	 */
	public static double acosec(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		SpecialValue sv = SpecialValueTrigonometric.getSpecialValueAcsc(a);
		double r;
		if (sv != null) r = sv.fv;
		else r = Math.asin(1/a);
		if (mXparser.checkIfDegreesMode()) {
			if (sv != null) return sv.fvdeg;
			return intIfAlmostIntOtherwiseOrig(div(r, Units.DEGREE_ARC));
		}
		else return r;
	}
	/**
	 * Natural logarithm
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.log(1/a),
	 *             otherwise returns Double.NaN.
	 */
	public static double ln(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.log(a);
	}
	/**
	 * Binary logarithm
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.log(a)/Math.log(2.0),
	 *             otherwise returns Double.NaN.
	 */
	public static double log2(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.log(a)/Math.log(2.0);
	}
	/**
	 * Common logarithm
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.log10(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double log10(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.log10(a);
	}
	/**
	 * Degrees to radius translation.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.toRadians(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double rad(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.toRadians(a);
	}
	/**
	 * Exponential function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.exp(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double exp(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.exp(a);
	}
	/**
	 * Square root.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.sqrt(a),
	 *             otherwise returns Double.NaN.
	 */
	 public static double sqrt(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.sqrt(a);
	}
	/**
	 * Hyperbolic sine function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.sinh(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double sinh(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.sinh(a);
	}
	/**
	 * Hyperbolic cosine function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.cosh(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double cosh(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.cosh(a);
	}
	/**
	 * Hyperbolic tangent function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.tanh(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double tanh(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.tanh(a);
	}
	/**
	 * Hyperbolic cotangent function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and tanh(a) &lt;&gt; 0 returns 1 / Math.tanh(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double coth(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		double tanh = Math.tanh(a);
		if (tanh != 0)
			result = 1.0 / tanh;
		return result;
	}
	/**
	 * Hyperbolic secant function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and cosh(a) &lt;&gt; 0 returns 1 / Math.cosh(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double sech(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		double cosh = Math.cosh(a);
		if (cosh != 0)
			result = 1.0 / cosh;
		return result;
	}
	/**
	 * Hyperbolic cosecant function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and sinh(a) &lt;&gt; 0 returns 1 / Math.sinh(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double csch(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		double sinh = Math.sinh(a);
		if (sinh != 0)
			result = 1.0 / sinh;
		return result;
	}
	/**
	 * Radius to degrees translation.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.toDegrees(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double deg(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.toDegrees(a);
	}
	/**
	 * Absolute value.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.abs(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double abs(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.abs(a);
	}
	/**
	 * Signum function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.signum(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double sgn(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.signum(a);
	}
	/**
	 * Floor function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.floor(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double floor(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.floor(a);
	}
	/**
	 * Ceiling function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.ceil(a),
	 *             otherwise returns Double.NaN.
	 */
	public static double ceil(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.ceil(a);
	}
	/**
	 * Arcus hyperbolic sine - inverse hyperbolic sine function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.log(a + Math.sqrt(a*a+1)),
	 *             otherwise returns Double.NaN.
	 */
	public static double arsinh(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.log(a + Math.sqrt(a*a+1));
	}
	/**
	 * Arcus hyperbolic cosine - inverse hyperbolic cosine function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN returns Math.log(a + Math.sqrt(a*a-1)),
	 *             otherwise returns Double.NaN.
	 */
	public static double arcosh(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		return Math.log(a + Math.sqrt(a*a-1));
	}
	/**
	 * Arcus hyperbolic tangent - inverse hyperbolic tangent function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and 1-a &lt;&gt; 0 returns 0.5*Math.log( (1+a)/(1-a) ),
	 *             otherwise returns Double.NaN.
	 */
	public static double artanh(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		if (1-a != 0)
			result = 0.5*Math.log( (1+a)/(1-a) );
		return result;
	}
	/**
	 * Arcus hyperbolic tangent - inverse hyperbolic tangent function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and a-1 &lt;&gt; 0 returns 0.5*Math.log( (a+1)/(a-1) );,
	 *             otherwise returns Double.NaN.
	 */
	public static double arcoth(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		if (a-1 != 0)
			result = 0.5*Math.log( (a+1)/(a-1) );
		return result;
	}
	/**
	 * Arcus hyperbolic secant - inverse hyperbolic secant function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and a &lt;&gt; 0 returns Math.log( (1+Math.sqrt(1-a*a))/a);,
	 *             otherwise returns Double.NaN.
	 */
	public static double arsech(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		if (a != 0)
			result = Math.log( (1+Math.sqrt(1-a*a))/a);
		return result;
	}
	/**
	 * Arcus hyperbolic cosecant - inverse hyperbolic cosecant function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and a &lt;&gt; 0 returns Math.log( (1+Math.sqrt(1-a*a))/a);,
	 *             otherwise returns Double.NaN.
	 */
	public static double arcsch(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		if (a != 0)
			result = Math.log( 1/a + Math.sqrt(1+a*a)/Math.abs(a) );
		return result;
	}
	/**
	 * Normalized sinc function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and a &lt;&gt; 0 returns Math.sin(PI*a) / (PI*a);,
	 *             otherwise returns Double.NaN.
	 */
	public static double sa(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double x = MathConstants.PI * a;
		double result = Double.NaN;
		if (x != 0)
			result = Math.sin(x) / (x);
		else
			result =  1.0;
		return result;
	}
	/**
	 * Sinc function.
	 *
	 * @param      a                   the a function parameter
	 *
	 * @return     if a &lt;&gt; Double.NaN and a &lt;&gt; 0 returns Math.sin(a) / (a),
	 *             otherwise returns Double.NaN.
	 */
	public static double sinc(double a) {
		if (Double.isNaN(a))
			return Double.NaN;
		double result = Double.NaN;
		if (a != 0)
			if (mXparser.checkIfDegreesMode())
				result = Math.sin(a * Units.DEGREE_ARC) / (a);
			else
				result = Math.sin(a) / (a);
		else
			result =  1.0;
		return result;
	}
	/**
	 * General logarithm.
	 *
	 * @param      a                   the a function parameter (base)
	 * @param      b                   the b function parameter (number)
	 *
	 * @return     if a,b &lt;&gt; Double.NaN and log(b) &lt;&gt; 0 returns Math.log(a) / Math.log(b),
	 *             otherwise returns Double.NaN.
	 */
	public static double log(double a, double b) {
		if (Double.isNaN(a) || Double.isNaN(b))
			return Double.NaN;
		double result = Double.NaN;
		double logb = Math.log(b);
		if (logb != 0)
			result = Math.log(a) / logb;
		return result;
	}
	/**
	 * Double rounding
	 *
	 * @param value    double value to be rounded
	 * @param places   decimal places
	 * @return         Rounded value
	 */
 	public static double round(double value, int places) {
		if (Double.isNaN(value)) return Double.NaN;
		if (Double.isInfinite(value)) return value;
		if (places < 0) return Double.NaN;
		try {
			BigDecimal bd = new BigDecimal(Double.toString(value));
			bd = bd.setScale(places, RoundingMode.HALF_UP);
			return bd.doubleValue();
		} catch (Throwable e) {
			return roundHalfUp(value, places);
		}
 	}
	/**
	 * Double half up rounding
	 *
	 * @param value    double value to be rounded
	 * @param places   decimal places
	 * @return         Rounded value
	 */
 	public static double roundHalfUp(double value, int places) {
 		if (Double.isNaN(value)) return Double.NaN;
		if (places < 0) return Double.NaN;
		if (value == Double.NEGATIVE_INFINITY) return Double.NEGATIVE_INFINITY;
		if (value == Double.POSITIVE_INFINITY) return Double.POSITIVE_INFINITY;
		if (value == 0) return 0;
		double sign = 1;
 		double origValue = value;
 		if (value < 0) {
 			sign = -1;
 			value = -value;
 		}
 		int ulpPosition = MathFunctions.ulpDecimalDigitsBefore(value);
 		if (ulpPosition <= 0) return sign * Math.floor(value);
 		if (places > ulpPosition) return origValue;
 		double multiplier = 1;
 		for (int place = 0; place < places; place++) {
 			multiplier = Math.floor(multiplier * 10);
 			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
 		}
 		double valueMultiplied = value * multiplier;
 		double valueFloor = Math.floor(valueMultiplied);
 		if (Math.abs(valueMultiplied - valueFloor) >= 0.5) valueFloor = Math.floor(valueFloor + 1);
 		return Math.floor(sign * valueFloor) / multiplier;
 	}
	/**
	 * Double down rounding
	 *
	 * @param value    double value to be rounded
	 * @param places   decimal places
	 * @return         Rounded value
	 */
 	public static double roundDown(double value, int places) {
 		if (Double.isNaN(value)) return Double.NaN;
		if (places < 0) return Double.NaN;
		if (value == Double.NEGATIVE_INFINITY) return Double.NEGATIVE_INFINITY;
		if (value == Double.POSITIVE_INFINITY) return Double.POSITIVE_INFINITY;
		if (value == 0) return 0;
		double sign = 1;
 		double origValue = value;
 		if (value < 0) {
 			sign = -1;
 			value = -value;
 		}
 		int ulpPosition = MathFunctions.ulpDecimalDigitsBefore(value);
 		if (ulpPosition <= 0) return sign * Math.floor(value);
 		if (places > ulpPosition) return origValue;
 		double multiplier = 1;
 		for (int place = 0; place < places; place++) {
 			multiplier = Math.floor(multiplier * 10);
 			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
 		}
 		double valueMultiplied = value * multiplier;
 		double valueFloor = Math.floor(valueMultiplied);
 		return Math.floor(sign * valueFloor) / multiplier;
 	}
 	/**
 	 * Unit in the last place rounding, see
 	 * 0.1 + 0.1 + 0.1 vs roundUlp(0.1 + 0.1 + 0.1)
 	 *
 	 * @param number   Double number that is to be rounded
 	 *
 	 * @return    Double number with rounded ulp
 	 *
 	 * @see MathFunctions#decimalDigitsBefore(double)
 	 * @see MathFunctions#ulp(double)
 	 */
	public static double roundUlp(double number) {
		if ( (Double.isNaN(number) ) || (Double.isInfinite(number)) || (number == 0) )
			return number;
		else {
			int precision = MathFunctions.ulpDecimalDigitsBefore(number);
			if (precision >= 1)
				return MathFunctions.round(number, precision-5);
			else if (precision == 0)
				return MathFunctions.round(number, 0);
			else return number;
		}
	}
 	/**
 	 * Returns integer part of a double value.
 	 * @param x  Number
 	 * @return For non-negative x returns Math.floor(x),
 	 *         otherwise returns -Math.floor(-x)
 	 */
 	public static double integerPart(double x) {
 		if (x > 0) return Math.floor(x);
 		else if (x < 0) return -Math.floor(-x);
 		else return 0;
 	}
 	/**
 	 * For very small number returns the position of
 	 * first significant digit, ie 0.1 = 1, 0.01 = 2
 	 *
 	 * @param anyValue Double value, small one.
 	 * @return Number of digits, number of places.
 	 */
	public static int decimalDigitsBefore(double anyValue) {
		double value;

		if (anyValue >= 0) value = anyValue;
		else value = -anyValue;

		if (value >= 1) return -1;

		if (value < 1e-200) {
			if (value < 1e-322) return 322;
			if (value < 1e-321) return 321;
			if (value < 1e-320) return 320;
			if (value < 1e-319) return 319;
			if (value < 1e-318) return 318;
			if (value < 1e-317) return 317;
			if (value < 1e-316) return 316;
			if (value < 1e-315) return 315;
			if (value < 1e-314) return 314;
			if (value < 1e-313) return 313;
			if (value < 1e-312) return 312;
			if (value < 1e-311) return 311;
			if (value < 1e-310) return 310;
			if (value < 1e-309) return 309;
			if (value < 1e-308) return 308;
			if (value < 1e-307) return 307;
			if (value < 1e-306) return 306;
			if (value < 1e-305) return 305;
			if (value < 1e-304) return 304;
			if (value < 1e-303) return 303;
			if (value < 1e-302) return 302;
			if (value < 1e-301) return 301;
			if (value < 1e-300) return 300;
			if (value < 1e-299) return 299;
			if (value < 1e-298) return 298;
			if (value < 1e-297) return 297;
			if (value < 1e-296) return 296;
			if (value < 1e-295) return 295;
			if (value < 1e-294) return 294;
			if (value < 1e-293) return 293;
			if (value < 1e-292) return 292;
			if (value < 1e-291) return 291;
			if (value < 1e-290) return 290;
			if (value < 1e-289) return 289;
			if (value < 1e-288) return 288;
			if (value < 1e-287) return 287;
			if (value < 1e-286) return 286;
			if (value < 1e-285) return 285;
			if (value < 1e-284) return 284;
			if (value < 1e-283) return 283;
			if (value < 1e-282) return 282;
			if (value < 1e-281) return 281;
			if (value < 1e-280) return 280;
			if (value < 1e-279) return 279;
			if (value < 1e-278) return 278;
			if (value < 1e-277) return 277;
			if (value < 1e-276) return 276;
			if (value < 1e-275) return 275;
			if (value < 1e-274) return 274;
			if (value < 1e-273) return 273;
			if (value < 1e-272) return 272;
			if (value < 1e-271) return 271;
			if (value < 1e-270) return 270;
			if (value < 1e-269) return 269;
			if (value < 1e-268) return 268;
			if (value < 1e-267) return 267;
			if (value < 1e-266) return 266;
			if (value < 1e-265) return 265;
			if (value < 1e-264) return 264;
			if (value < 1e-263) return 263;
			if (value < 1e-262) return 262;
			if (value < 1e-261) return 261;
			if (value < 1e-260) return 260;
			if (value < 1e-259) return 259;
			if (value < 1e-258) return 258;
			if (value < 1e-257) return 257;
			if (value < 1e-256) return 256;
			if (value < 1e-255) return 255;
			if (value < 1e-254) return 254;
			if (value < 1e-253) return 253;
			if (value < 1e-252) return 252;
			if (value < 1e-251) return 251;
			if (value < 1e-250) return 250;
			if (value < 1e-249) return 249;
			if (value < 1e-248) return 248;
			if (value < 1e-247) return 247;
			if (value < 1e-246) return 246;
			if (value < 1e-245) return 245;
			if (value < 1e-244) return 244;
			if (value < 1e-243) return 243;
			if (value < 1e-242) return 242;
			if (value < 1e-241) return 241;
			if (value < 1e-240) return 240;
			if (value < 1e-239) return 239;
			if (value < 1e-238) return 238;
			if (value < 1e-237) return 237;
			if (value < 1e-236) return 236;
			if (value < 1e-235) return 235;
			if (value < 1e-234) return 234;
			if (value < 1e-233) return 233;
			if (value < 1e-232) return 232;
			if (value < 1e-231) return 231;
			if (value < 1e-230) return 230;
			if (value < 1e-229) return 229;
			if (value < 1e-228) return 228;
			if (value < 1e-227) return 227;
			if (value < 1e-226) return 226;
			if (value < 1e-225) return 225;
			if (value < 1e-224) return 224;
			if (value < 1e-223) return 223;
			if (value < 1e-222) return 222;
			if (value < 1e-221) return 221;
			if (value < 1e-220) return 220;
			if (value < 1e-219) return 219;
			if (value < 1e-218) return 218;
			if (value < 1e-217) return 217;
			if (value < 1e-216) return 216;
			if (value < 1e-215) return 215;
			if (value < 1e-214) return 214;
			if (value < 1e-213) return 213;
			if (value < 1e-212) return 212;
			if (value < 1e-211) return 211;
			if (value < 1e-210) return 210;
			if (value < 1e-209) return 209;
			if (value < 1e-208) return 208;
			if (value < 1e-207) return 207;
			if (value < 1e-206) return 206;
			if (value < 1e-205) return 205;
			if (value < 1e-204) return 204;
			if (value < 1e-203) return 203;
			if (value < 1e-202) return 202;
			if (value < 1e-201) return 201;
			/* if (value < 1e-200) */ return 200;
		}
		if (value < 1e-100) {
			if (value < 1e-199) return 199;
			if (value < 1e-198) return 198;
			if (value < 1e-197) return 197;
			if (value < 1e-196) return 196;
			if (value < 1e-195) return 195;
			if (value < 1e-194) return 194;
			if (value < 1e-193) return 193;
			if (value < 1e-192) return 192;
			if (value < 1e-191) return 191;
			if (value < 1e-190) return 190;
			if (value < 1e-189) return 189;
			if (value < 1e-188) return 188;
			if (value < 1e-187) return 187;
			if (value < 1e-186) return 186;
			if (value < 1e-185) return 185;
			if (value < 1e-184) return 184;
			if (value < 1e-183) return 183;
			if (value < 1e-182) return 182;
			if (value < 1e-181) return 181;
			if (value < 1e-180) return 180;
			if (value < 1e-179) return 179;
			if (value < 1e-178) return 178;
			if (value < 1e-177) return 177;
			if (value < 1e-176) return 176;
			if (value < 1e-175) return 175;
			if (value < 1e-174) return 174;
			if (value < 1e-173) return 173;
			if (value < 1e-172) return 172;
			if (value < 1e-171) return 171;
			if (value < 1e-170) return 170;
			if (value < 1e-169) return 169;
			if (value < 1e-168) return 168;
			if (value < 1e-167) return 167;
			if (value < 1e-166) return 166;
			if (value < 1e-165) return 165;
			if (value < 1e-164) return 164;
			if (value < 1e-163) return 163;
			if (value < 1e-162) return 162;
			if (value < 1e-161) return 161;
			if (value < 1e-160) return 160;
			if (value < 1e-159) return 159;
			if (value < 1e-158) return 158;
			if (value < 1e-157) return 157;
			if (value < 1e-156) return 156;
			if (value < 1e-155) return 155;
			if (value < 1e-154) return 154;
			if (value < 1e-153) return 153;
			if (value < 1e-152) return 152;
			if (value < 1e-151) return 151;
			if (value < 1e-150) return 150;
			if (value < 1e-149) return 149;
			if (value < 1e-148) return 148;
			if (value < 1e-147) return 147;
			if (value < 1e-146) return 146;
			if (value < 1e-145) return 145;
			if (value < 1e-144) return 144;
			if (value < 1e-143) return 143;
			if (value < 1e-142) return 142;
			if (value < 1e-141) return 141;
			if (value < 1e-140) return 140;
			if (value < 1e-139) return 139;
			if (value < 1e-138) return 138;
			if (value < 1e-137) return 137;
			if (value < 1e-136) return 136;
			if (value < 1e-135) return 135;
			if (value < 1e-134) return 134;
			if (value < 1e-133) return 133;
			if (value < 1e-132) return 132;
			if (value < 1e-131) return 131;
			if (value < 1e-130) return 130;
			if (value < 1e-129) return 129;
			if (value < 1e-128) return 128;
			if (value < 1e-127) return 127;
			if (value < 1e-126) return 126;
			if (value < 1e-125) return 125;
			if (value < 1e-124) return 124;
			if (value < 1e-123) return 123;
			if (value < 1e-122) return 122;
			if (value < 1e-121) return 121;
			if (value < 1e-120) return 120;
			if (value < 1e-119) return 119;
			if (value < 1e-118) return 118;
			if (value < 1e-117) return 117;
			if (value < 1e-116) return 116;
			if (value < 1e-115) return 115;
			if (value < 1e-114) return 114;
			if (value < 1e-113) return 113;
			if (value < 1e-112) return 112;
			if (value < 1e-111) return 111;
			if (value < 1e-110) return 110;
			if (value < 1e-109) return 109;
			if (value < 1e-108) return 108;
			if (value < 1e-107) return 107;
			if (value < 1e-106) return 106;
			if (value < 1e-105) return 105;
			if (value < 1e-104) return 104;
			if (value < 1e-103) return 103;
			if (value < 1e-102) return 102;
			if (value < 1e-101) return 101;
			/* if (value < 1e-100) */ return 100;
		}

		if (value < 1e-50) {
			if (value < 1e-99) return 99;
			if (value < 1e-98) return 98;
			if (value < 1e-97) return 97;
			if (value < 1e-96) return 96;
			if (value < 1e-95) return 95;
			if (value < 1e-94) return 94;
			if (value < 1e-93) return 93;
			if (value < 1e-92) return 92;
			if (value < 1e-91) return 91;
			if (value < 1e-90) return 90;
			if (value < 1e-89) return 89;
			if (value < 1e-88) return 88;
			if (value < 1e-87) return 87;
			if (value < 1e-86) return 86;
			if (value < 1e-85) return 85;
			if (value < 1e-84) return 84;
			if (value < 1e-83) return 83;
			if (value < 1e-82) return 82;
			if (value < 1e-81) return 81;
			if (value < 1e-80) return 80;
			if (value < 1e-79) return 79;
			if (value < 1e-78) return 78;
			if (value < 1e-77) return 77;
			if (value < 1e-76) return 76;
			if (value < 1e-75) return 75;
			if (value < 1e-74) return 74;
			if (value < 1e-73) return 73;
			if (value < 1e-72) return 72;
			if (value < 1e-71) return 71;
			if (value < 1e-70) return 70;
			if (value < 1e-69) return 69;
			if (value < 1e-68) return 68;
			if (value < 1e-67) return 67;
			if (value < 1e-66) return 66;
			if (value < 1e-65) return 65;
			if (value < 1e-64) return 64;
			if (value < 1e-63) return 63;
			if (value < 1e-62) return 62;
			if (value < 1e-61) return 61;
			if (value < 1e-60) return 60;
			if (value < 1e-59) return 59;
			if (value < 1e-58) return 58;
			if (value < 1e-57) return 57;
			if (value < 1e-56) return 56;
			if (value < 1e-55) return 55;
			if (value < 1e-54) return 54;
			if (value < 1e-53) return 53;
			if (value < 1e-52) return 52;
			if (value < 1e-51) return 51;
			/* if (value < 1e-50) */ return 50;
		}

		if (value < 1e-20) {
			if (value < 1e-49) return 49;
			if (value < 1e-48) return 48;
			if (value < 1e-47) return 47;
			if (value < 1e-46) return 46;
			if (value < 1e-45) return 45;
			if (value < 1e-44) return 44;
			if (value < 1e-43) return 43;
			if (value < 1e-42) return 42;
			if (value < 1e-41) return 41;
			if (value < 1e-40) return 40;
			if (value < 1e-39) return 39;
			if (value < 1e-38) return 38;
			if (value < 1e-37) return 37;
			if (value < 1e-36) return 36;
			if (value < 1e-35) return 35;
			if (value < 1e-34) return 34;
			if (value < 1e-33) return 33;
			if (value < 1e-32) return 32;
			if (value < 1e-31) return 31;
			if (value < 1e-30) return 30;
			if (value < 1e-29) return 29;
			if (value < 1e-28) return 28;
			if (value < 1e-27) return 27;
			if (value < 1e-26) return 26;
			if (value < 1e-25) return 25;
			if (value < 1e-24) return 24;
			if (value < 1e-23) return 23;
			if (value < 1e-22) return 22;
			if (value < 1e-21) return 21;
			/* if (value < 1e-20) */ return 20;
		}

		if (value < 1e-10) {
			if (value < 1e-19) return 19;
			if (value < 1e-18) return 18;
			if (value < 1e-17) return 17;
			if (value < 1e-16) return 16;
			if (value < 1e-15) return 15;
			if (value < 1e-14) return 14;
			if (value < 1e-13) return 13;
			if (value < 1e-12) return 12;
			if (value < 1e-11) return 11;
			/* if (value < 1e-10) */ return 10;
		}

		if (value < 1e-5) {
			if (value < 1e-9) return 9;
			if (value < 1e-8) return 8;
			if (value < 1e-7) return 7;
			if (value < 1e-6) return 6;
			/* if (value < 1e-5) */ return 5;
		}

		if (value < 1e-4) return 4;
		if (value < 1e-3) return 3;
		if (value < 1e-2) return 2;
		if (value < 1e-1) return 1;
		if (value < 1e-0) return 0;

		return -1;
	}
	/**
	 * Unit in the last place(ULP) for double
	 * @param value Double number
	 * @return ULP for a given double.
	 */
	public static double ulp(double value) {
		return Math.ulp(value);
	}
	/**
	 * Unit in The Last Place - number of decimal digits before
	 * @param value   Double number
	 * @return        Positive number of digits N for ulp = 1e-{N+1},
	 *                if ulp is &gt; 1 then -1 is returned.
	 *                Returned proper value is always between -1 and +322.
	 *                If value is NaN then -2 is returned.
	 */
	public static int ulpDecimalDigitsBefore(double value) {
		if (Double.isNaN(value)) return -2;
		double u = ulp(value);
		return decimalDigitsBefore(u);
	}
	/**
	 * Length of a number represented in a standard decimal format
	 * @param value   A given number
	 * @return        Length of a number represented in a standard decimal format
	 *                including decimal separator, excluding leading zeros (integer part),
	 *                excluding trailing zeros (fractional part)
	 */
	public static int decimalNumberLength(double value) {
		return DECIMAL_FORMAT.format(value).length();
	}
	/**
	 * Fractional part length of a number represented in a standard decimal format
	 * @param value   A given number
	 * @return        Fractional part length of a number represented in a standard decimal
	 *                format excluding decimal separator, excluding trailing zeros (fractional part)
	 */
	public static int fractionalPartLength(double value) {
		if (Double.isNaN(value)) return 0;
		if (Double.isInfinite(value)) return 0;
		if (ulpDecimalDigitsBefore(value) <= 0) return 0;
		String valueStr = DECIMAL_FORMAT.format(value);
		int dotPos = valueStr.indexOf('.');
		if (dotPos >= 0) return valueStr.length() - 1 - dotPos;
		return 0;
	}
	/**
	 * Intelligent rounding of a number within the decimal position of the ULP (Unit in the Last Place),
	 * provided that the result is significantly shortened in the standard decimal notation. Examples:
	 * 30.499999999999992 is rounded to 30.5, but 30.499999999999122 will not be rounded. Rounding is
	 * made to the decimal position of the ULP minus 2 on condition that the resulted number is shortened
	 * by at least 9 places.
	 * @param value   A given number
	 * @return        Returns an intelligently rounded number when the decimal position of ULP
	 *                is a minimum of 11 and when rounded to the position of ULP - 2, shortens
	 *                the number by a minimum of 9 places. Otherwise, returns original number.
	 */
	public static double lengthRound(double value) {
		if (Double.isNaN(value)) return value;
		if (Double.isInfinite(value)) return value;
		if (value == 0d || value == -1d || value == 1d || value == -2d || value == 2d || value == -3d || value == 3d) return value;
		if (value == -4d || value == 4d || value == -5d || value == 5d || value == -6d || value == 6d) return value;
		if (value == -7d || value == 7d || value == -8d || value == 8d || value == -9d || value == 9d) return value;
		if (value == -10d || value == 10d || value == -11d || value == 11d || value == -12d || value == 12d) return value;
		if (ulpDecimalDigitsBefore(value) < 6) return value;
		int decPartLen = fractionalPartLength(value);
		if (decPartLen < 11) return value;
		double valueRound = round(value, decPartLen - 2);
		int decPartLenRound = fractionalPartLength(valueRound);
		if (decPartLen - decPartLenRound >= 9) return valueRound;
		return value;
	}
	/**
	 * Returns the first non-NaN value
	 *
	 * @param values   List of values
	 * @return         Returns the first non-NaN value, if list is null
	 *                 then returns Double.NaN, if list contains no elements
	 *                 then returns Double.NaN.
	 */
	public static double coalesce(double[] values) {
		if (values == null) return Double.NaN;
		for (double v : values) {
			if (!Double.isNaN(v)) return v;
			if (mXparser.isCurrentCalculationCancelled()) return Double.NaN;
		}
		return Double.NaN;
	}
	/**
	 * Check whether double value is almost integer.
	 * @param x   Number
	 * @return True if double value is almost integer, otherwise false.
	 *         {@link BinaryRelations#DEFAULT_COMPARISON_EPSILON}
	 *
	 * @see BinaryRelations#DEFAULT_COMPARISON_EPSILON
	 */
	public static boolean isInteger(double x) {
		if (Double.isNaN(x)) return false;
		if (x == Double.POSITIVE_INFINITY) return false;
		if (x == Double.NEGATIVE_INFINITY) return false;
		if (x < 0) x = -x;
		double round = Math.round(x);
		return Math.abs(x - round) < BinaryRelations.DEFAULT_COMPARISON_EPSILON;
	}
	/**
	 * Check whether two double values are almost equal.
	 * @param a   First number
	 * @param b   Second number
	 * @return True if double values are almost equal, otherwise false.
	 *         {@link BinaryRelations#DEFAULT_COMPARISON_EPSILON}
	 *
	 * @see BinaryRelations#DEFAULT_COMPARISON_EPSILON
	 */
	public static boolean almostEqual(double a, double b) {
		if (Double.isNaN(a)) return false;
		if (Double.isNaN(b)) return false;
		if (a == b) return true;
		return Math.abs(a - b) <= BinaryRelations.DEFAULT_COMPARISON_EPSILON;
	}
	/**
	 * Calculates function f(x0) (given as expression) assigning Argument x = x0;
	 *
	 *
	 * @param      f                   the expression
	 * @param      x                   the argument
	 * @param      x0                  the argument value
	 *
	 * @return     f.calculate()
	 *
	 * @see        Expression
	 */
	public static double getFunctionValue(Expression f, Argument x, double x0) {
		x.setArgumentValue(x0);
		return f.calculate();
	}

	/**
	 * Returns array of double values of the function f(i)
	 * calculated on the range: i = from to i = to by step = delta
	 *
	 * @param f            Function expression
	 * @param index        Index argument
	 * @param from         'from' value
	 * @param to           'to' value
	 * @param delta        'delta' step definition
	 * @return             Array of function values
	 */
	public static double[] getFunctionValues(Expression f, Argument index, double from, double to, double delta) {
		if (Double.isNaN(delta) || Double.isNaN(from) || Double.isNaN(to) || delta == 0)
			return null;
		int n = 0;
		double[] values;
		if (to >= from && delta > 0) {
			for (double i = from; i < to; i+=delta)
				n++;
			n++;
			values = new double[n];
			int j = 0;
			for (double i = from; i < to; i+=delta) {
				values[j] = getFunctionValue(f, index, i);
				j++;
			}
			values[j] = getFunctionValue(f, index, to);
		} else if (to <= from && delta < 0) {
			for (double i = from; i > to; i+=delta)
				n++;
			n++;
			values = new double[n];
			int j = 0;
			for (double i = from; i > to; i+=delta) {
				values[j] = getFunctionValue(f, index, i);
				j++;
			}
			values[j] = getFunctionValue(f, index, to);
		} else if (from == to) {
			n = 1;
			values = new double[n];
			values[0] = getFunctionValue(f, index, from);
		} else values = null;
		return values;
	}
}