/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.integration.async;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.atomic.AtomicInteger;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.hamcrest.junit.MatcherAssert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Query;
import org.neo4j.driver.Record;
import org.neo4j.driver.SessionConfig;
import org.neo4j.driver.Value;
import org.neo4j.driver.Values;
import org.neo4j.driver.async.AsyncSession;
import org.neo4j.driver.async.AsyncTransaction;
import org.neo4j.driver.async.ResultCursor;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.NoSuchRecordException;
import org.neo4j.driver.exceptions.ResultConsumedException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.internal.InternalBookmark;
import org.neo4j.driver.internal.util.Iterables;
import org.neo4j.driver.summary.QueryType;
import org.neo4j.driver.summary.ResultSummary;
import org.neo4j.driver.types.Node;
import org.neo4j.driver.util.DatabaseExtension;
import org.neo4j.driver.util.ParallelizableIT;
import org.neo4j.driver.util.TestUtil;

@ParallelizableIT
class AsyncTransactionIT {
    @RegisterExtension
    static final DatabaseExtension neo4j = new DatabaseExtension();
    private AsyncSession session;

    AsyncTransactionIT() {
    }

    @BeforeEach
    void setUp() {
        this.session = neo4j.driver().asyncSession();
    }

    @AfterEach
    void tearDown() {
        this.session.closeAsync();
    }

    @Test
    void shouldBePossibleToCommitEmptyTx() {
        Bookmark bookmarkBefore = this.session.lastBookmark();
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        MatcherAssert.assertThat((Object)((Void)TestUtil.await(tx.commitAsync())), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
        Bookmark bookmarkAfter = this.session.lastBookmark();
        Assertions.assertNotNull((Object)bookmarkAfter);
        Assertions.assertNotEquals((Object)bookmarkBefore, (Object)bookmarkAfter);
    }

    @Test
    void shouldBePossibleToRollbackEmptyTx() {
        Bookmark bookmarkBefore = this.session.lastBookmark();
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        MatcherAssert.assertThat((Object)((Void)TestUtil.await(tx.rollbackAsync())), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
        Bookmark bookmarkAfter = this.session.lastBookmark();
        Assertions.assertEquals((Object)bookmarkBefore, (Object)bookmarkAfter);
    }

    @Test
    void shouldBePossibleToRunSingleQueryAndCommit() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node {id: 42}) RETURN n"));
        Record record = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record);
        Node node = record.get(0).asNode();
        Assertions.assertEquals((Object)"Node", (Object)Iterables.single((Iterable)node.labels()));
        Assertions.assertEquals((int)42, (int)node.get("id").asInt());
        Assertions.assertNull(TestUtil.await(cursor.nextAsync()));
        Assertions.assertNull(TestUtil.await(tx.commitAsync()));
        Assertions.assertEquals((int)1, (int)this.countNodes(42));
    }

    @Test
    void shouldBePossibleToRunSingleQueryAndRollback() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node {id: 4242}) RETURN n"));
        Record record = (Record)TestUtil.await(cursor.nextAsync());
        Assertions.assertNotNull((Object)record);
        Node node = record.get(0).asNode();
        Assertions.assertEquals((Object)"Node", (Object)Iterables.single((Iterable)node.labels()));
        Assertions.assertEquals((int)4242, (int)node.get("id").asInt());
        Assertions.assertNull(TestUtil.await(cursor.nextAsync()));
        Assertions.assertNull(TestUtil.await(tx.rollbackAsync()));
        Assertions.assertEquals((int)0, (int)this.countNodes(4242));
    }

    @Test
    void shouldBePossibleToRunMultipleQueriesAndCommit() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor1 = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node {id: 1})"));
        Assertions.assertNull(TestUtil.await(cursor1.nextAsync()));
        ResultCursor cursor2 = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node {id: 2})"));
        Assertions.assertNull(TestUtil.await(cursor2.nextAsync()));
        ResultCursor cursor3 = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node {id: 2})"));
        Assertions.assertNull(TestUtil.await(cursor3.nextAsync()));
        Assertions.assertNull(TestUtil.await(tx.commitAsync()));
        Assertions.assertEquals((int)1, (int)this.countNodes(1));
        Assertions.assertEquals((int)2, (int)this.countNodes(2));
    }

    @Test
    void shouldBePossibleToRunMultipleQueriesAndCommitWithoutWaiting() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE (n:Node {id: 1})");
        tx.runAsync("CREATE (n:Node {id: 2})");
        tx.runAsync("CREATE (n:Node {id: 1})");
        Assertions.assertNull(TestUtil.await(tx.commitAsync()));
        Assertions.assertEquals((int)1, (int)this.countNodes(2));
        Assertions.assertEquals((int)2, (int)this.countNodes(1));
    }

    @Test
    void shouldBePossibleToRunMultipleQueriesAndRollback() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor1 = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node {id: 1})"));
        Assertions.assertNull(TestUtil.await(cursor1.nextAsync()));
        ResultCursor cursor2 = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node {id: 42})"));
        Assertions.assertNull(TestUtil.await(cursor2.nextAsync()));
        Assertions.assertNull(TestUtil.await(tx.rollbackAsync()));
        Assertions.assertEquals((int)0, (int)this.countNodes(1));
        Assertions.assertEquals((int)0, (int)this.countNodes(42));
    }

    @Test
    void shouldBePossibleToRunMultipleQueriesAndRollbackWithoutWaiting() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE (n:Node {id: 1})");
        tx.runAsync("CREATE (n:Node {id: 42})");
        Assertions.assertNull(TestUtil.await(tx.rollbackAsync()));
        Assertions.assertEquals((int)0, (int)this.countNodes(1));
        Assertions.assertEquals((int)0, (int)this.countNodes(42));
    }

    @Test
    void shouldFailToCommitAfterSingleWrongQuery() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(tx.runAsync("RETURN")));
        MatcherAssert.assertThat((Object)e, (Matcher)Matchers.is(org.neo4j.driver.internal.util.Matchers.syntaxError()));
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
    }

    @Test
    void shouldAllowRollbackAfterSingleWrongQuery() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(tx.runAsync("RETURN")));
        MatcherAssert.assertThat((Object)e, (Matcher)Matchers.is(org.neo4j.driver.internal.util.Matchers.syntaxError()));
        MatcherAssert.assertThat((Object)((Void)TestUtil.await(tx.rollbackAsync())), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
    }

    @Test
    void shouldFailToCommitAfterCoupleCorrectAndSingleWrongQuery() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor1 = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node) RETURN n"));
        Record record1 = (Record)TestUtil.await(cursor1.nextAsync());
        Assertions.assertNotNull((Object)record1);
        Assertions.assertTrue((boolean)record1.get(0).asNode().hasLabel("Node"));
        ResultCursor cursor2 = (ResultCursor)TestUtil.await(tx.runAsync("RETURN 42"));
        Record record2 = (Record)TestUtil.await(cursor2.nextAsync());
        Assertions.assertNotNull((Object)record2);
        Assertions.assertEquals((int)42, (int)record2.get(0).asInt());
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(tx.runAsync("RETURN")));
        MatcherAssert.assertThat((Object)e, (Matcher)Matchers.is(org.neo4j.driver.internal.util.Matchers.syntaxError()));
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
    }

    @Test
    void shouldAllowRollbackAfterCoupleCorrectAndSingleWrongQuery() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor1 = (ResultCursor)TestUtil.await(tx.runAsync("RETURN 4242"));
        Record record1 = (Record)TestUtil.await(cursor1.nextAsync());
        Assertions.assertNotNull((Object)record1);
        Assertions.assertEquals((int)4242, (int)record1.get(0).asInt());
        ResultCursor cursor2 = (ResultCursor)TestUtil.await(tx.runAsync("CREATE (n:Node) DELETE n RETURN 42"));
        Record record2 = (Record)TestUtil.await(cursor2.nextAsync());
        Assertions.assertNotNull((Object)record2);
        Assertions.assertEquals((int)42, (int)record2.get(0).asInt());
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(tx.runAsync("RETURN")));
        MatcherAssert.assertThat((Object)e, (Matcher)Matchers.is(org.neo4j.driver.internal.util.Matchers.syntaxError()));
        MatcherAssert.assertThat((Object)((Void)TestUtil.await(tx.rollbackAsync())), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
    }

    @Test
    void shouldNotAllowNewQueriesAfterAnIncorrectQuery() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        Exception e1 = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(tx.runAsync("RETURN")));
        MatcherAssert.assertThat((Object)e1, (Matcher)Matchers.is(org.neo4j.driver.internal.util.Matchers.syntaxError()));
        ClientException e2 = (ClientException)Assertions.assertThrows(ClientException.class, () -> tx.runAsync("CREATE ()"));
        MatcherAssert.assertThat((Object)e2.getMessage(), (Matcher)Matchers.startsWith((String)"Cannot run more queries in this transaction"));
    }

    @Test
    void shouldFailBoBeginTxWithInvalidBookmark() {
        AsyncSession session = neo4j.driver().asyncSession(SessionConfig.builder().withBookmarks(new Bookmark[]{InternalBookmark.parse((String)"InvalidBookmark")}).build());
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(session.beginTransactionAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"InvalidBookmark"));
    }

    @Test
    void shouldFailToCommitWhenCommitted() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE ()");
        Assertions.assertNull(TestUtil.await(tx.commitAsync()));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"transaction has been committed"));
    }

    @Test
    void shouldFailToRollbackWhenRolledBack() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE ()");
        Assertions.assertNull(TestUtil.await(tx.rollbackAsync()));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.rollbackAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"transaction has been rolled back"));
    }

    @Test
    void shouldFailToCommitWhenRolledBack() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE ()");
        Assertions.assertNull(TestUtil.await(tx.rollbackAsync()));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"transaction has been rolled back"));
    }

    @Test
    void shouldFailToRollbackWhenCommitted() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE ()");
        Assertions.assertNull(TestUtil.await(tx.commitAsync()));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.rollbackAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"transaction has been committed"));
    }

    @Test
    void shouldExposeQueryKeysForColumnsWithAliases() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("RETURN 1 AS one, 2 AS two, 3 AS three, 4 AS five"));
        Assertions.assertEquals(Arrays.asList("one", "two", "three", "five"), (Object)cursor.keys());
    }

    @Test
    void shouldExposeQueryKeysForColumnsWithoutAliases() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("RETURN 1, 2, 3, 5"));
        Assertions.assertEquals(Arrays.asList("1", "2", "3", "5"), (Object)cursor.keys());
    }

    @Test
    void shouldExposeResultSummaryForSimpleQuery() {
        String query = "CREATE (p1:Person {name: $name1})-[:KNOWS]->(p2:Person {name: $name2}) RETURN p1, p2";
        Value params = Values.parameters((Object[])new Object[]{"name1", "Bob", "name2", "John"});
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync(query, params));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertEquals((Object)new Query(query, params), (Object)summary.query());
        Assertions.assertEquals((int)2, (int)summary.counters().nodesCreated());
        Assertions.assertEquals((int)2, (int)summary.counters().labelsAdded());
        Assertions.assertEquals((int)2, (int)summary.counters().propertiesSet());
        Assertions.assertEquals((int)1, (int)summary.counters().relationshipsCreated());
        Assertions.assertEquals((Object)QueryType.READ_WRITE, (Object)summary.queryType());
        Assertions.assertFalse((boolean)summary.hasPlan());
        Assertions.assertFalse((boolean)summary.hasProfile());
        Assertions.assertNull((Object)summary.plan());
        Assertions.assertNull((Object)summary.profile());
        Assertions.assertEquals((int)0, (int)summary.notifications().size());
        MatcherAssert.assertThat((Object)summary, org.neo4j.driver.internal.util.Matchers.containsResultAvailableAfterAndResultConsumedAfter());
    }

    @Test
    void shouldExposeResultSummaryForExplainQuery() {
        String query = "EXPLAIN MATCH (n) RETURN n";
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync(query));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertEquals((Object)new Query(query), (Object)summary.query());
        Assertions.assertEquals((int)0, (int)summary.counters().nodesCreated());
        Assertions.assertEquals((int)0, (int)summary.counters().propertiesSet());
        Assertions.assertEquals((Object)QueryType.READ_ONLY, (Object)summary.queryType());
        Assertions.assertTrue((boolean)summary.hasPlan());
        Assertions.assertFalse((boolean)summary.hasProfile());
        Assertions.assertNotNull((Object)summary.plan());
        MatcherAssert.assertThat((Object)summary.plan().toString().toLowerCase(), (Matcher)Matchers.containsString((String)"scan"));
        Assertions.assertNull((Object)summary.profile());
        Assertions.assertEquals((int)0, (int)summary.notifications().size());
        MatcherAssert.assertThat((Object)summary, org.neo4j.driver.internal.util.Matchers.containsResultAvailableAfterAndResultConsumedAfter());
    }

    @Test
    void shouldExposeResultSummaryForProfileQuery() {
        String query = "PROFILE MERGE (n {name: $name}) ON CREATE SET n.created = timestamp() ON MATCH SET n.counter = coalesce(n.counter, 0) + 1";
        Value params = Values.parameters((Object[])new Object[]{"name", "Bob"});
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync(query, params));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertEquals((Object)new Query(query, params), (Object)summary.query());
        Assertions.assertEquals((int)1, (int)summary.counters().nodesCreated());
        Assertions.assertEquals((int)2, (int)summary.counters().propertiesSet());
        Assertions.assertEquals((int)0, (int)summary.counters().relationshipsCreated());
        Assertions.assertEquals((Object)QueryType.WRITE_ONLY, (Object)summary.queryType());
        Assertions.assertTrue((boolean)summary.hasPlan());
        Assertions.assertTrue((boolean)summary.hasProfile());
        Assertions.assertNotNull((Object)summary.plan());
        Assertions.assertNotNull((Object)summary.profile());
        String profileAsString = summary.profile().toString().toLowerCase();
        MatcherAssert.assertThat((Object)profileAsString, (Matcher)Matchers.containsString((String)"hits"));
        Assertions.assertEquals((int)0, (int)summary.notifications().size());
        MatcherAssert.assertThat((Object)summary, org.neo4j.driver.internal.util.Matchers.containsResultAvailableAfterAndResultConsumedAfter());
    }

    @Test
    void shouldPeekRecordFromCursor() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("UNWIND ['a', 'b', 'c'] AS x RETURN x"));
        Assertions.assertEquals((Object)"a", (Object)((Record)TestUtil.await(cursor.peekAsync())).get(0).asString());
        Assertions.assertEquals((Object)"a", (Object)((Record)TestUtil.await(cursor.peekAsync())).get(0).asString());
        Assertions.assertEquals((Object)"a", (Object)((Record)TestUtil.await(cursor.nextAsync())).get(0).asString());
        Assertions.assertEquals((Object)"b", (Object)((Record)TestUtil.await(cursor.peekAsync())).get(0).asString());
        Assertions.assertEquals((Object)"b", (Object)((Record)TestUtil.await(cursor.peekAsync())).get(0).asString());
        Assertions.assertEquals((Object)"b", (Object)((Record)TestUtil.await(cursor.peekAsync())).get(0).asString());
        Assertions.assertEquals((Object)"b", (Object)((Record)TestUtil.await(cursor.nextAsync())).get(0).asString());
        Assertions.assertEquals((Object)"c", (Object)((Record)TestUtil.await(cursor.nextAsync())).get(0).asString());
        Assertions.assertNull(TestUtil.await(cursor.peekAsync()));
        Assertions.assertNull(TestUtil.await(cursor.nextAsync()));
        TestUtil.await(tx.rollbackAsync());
    }

    @Test
    void shouldForEachWithEmptyCursor() {
        this.testForEach("MATCH (n:SomeReallyStrangeLabel) RETURN n", 0);
    }

    @Test
    void shouldForEachWithNonEmptyCursor() {
        this.testForEach("UNWIND range(1, 12555) AS x CREATE (n:Node {id: x}) RETURN n", 12555);
    }

    @Test
    void shouldFailForEachWhenActionFails() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("RETURN 'Hi!'"));
        RuntimeException error = new RuntimeException();
        RuntimeException e = (RuntimeException)Assertions.assertThrows(RuntimeException.class, () -> TestUtil.await(cursor.forEachAsync(record -> {
            throw error;
        })));
        Assertions.assertEquals((Object)error, (Object)e);
    }

    @Test
    void shouldConvertToListWithEmptyCursor() {
        this.testList("CREATE (:Person)-[:KNOWS]->(:Person)", Collections.emptyList());
    }

    @Test
    void shouldConvertToListWithNonEmptyCursor() {
        this.testList("UNWIND [1, '1', 2, '2', 3, '3'] AS x RETURN x", Arrays.asList(1L, "1", 2L, "2", 3L, "3"));
    }

    @Test
    void shouldConvertToTransformedListWithEmptyCursor() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("CREATE ()"));
        List maps = (List)TestUtil.await(cursor.listAsync(record -> record.get(0).asMap()));
        Assertions.assertEquals((int)0, (int)maps.size());
    }

    @Test
    void shouldConvertToTransformedListWithNonEmptyCursor() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("UNWIND ['a', 'b', 'c'] AS x RETURN x"));
        List strings = (List)TestUtil.await(cursor.listAsync(record -> record.get(0).asString() + "!"));
        Assertions.assertEquals(Arrays.asList("a!", "b!", "c!"), (Object)strings);
    }

    @Test
    void shouldFailWhenListTransformationFunctionFails() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("RETURN 'Hello'"));
        IOException error = new IOException("World");
        Exception e = (Exception)Assertions.assertThrows(Exception.class, () -> TestUtil.await(cursor.listAsync(record -> {
            throw new CompletionException(error);
        })));
        Assertions.assertEquals((Object)error, (Object)e);
    }

    @Test
    void shouldFailToCommitWhenServerIsRestarted() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        TestUtil.await(tx.runAsync("CREATE ()"));
        neo4j.stopDb();
        Assertions.assertThrows(ServiceUnavailableException.class, () -> TestUtil.await(tx.commitAsync()));
    }

    @Test
    void shouldFailSingleWithEmptyCursor() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("MATCH (n:NoSuchLabel) RETURN n"));
        NoSuchRecordException e = (NoSuchRecordException)Assertions.assertThrows(NoSuchRecordException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"result is empty"));
    }

    @Test
    void shouldFailSingleWithMultiRecordCursor() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("UNWIND ['a', 'b'] AS x RETURN x"));
        NoSuchRecordException e = (NoSuchRecordException)Assertions.assertThrows(NoSuchRecordException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.startsWith((String)"Expected a result with a single record"));
    }

    @Test
    void shouldReturnSingleWithSingleRecordCursor() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("RETURN 'Hello!'"));
        Record record = (Record)TestUtil.await(cursor.singleAsync());
        Assertions.assertEquals((Object)"Hello!", (Object)record.get(0).asString());
    }

    @Test
    void shouldPropagateFailureFromFirstRecordInSingleAsync() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("UNWIND [0] AS x RETURN 10 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"/ by zero"));
    }

    @Test
    void shouldNotPropagateFailureFromSecondRecordInSingleAsync() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("UNWIND [1, 0] AS x RETURN 10 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(cursor.singleAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"/ by zero"));
    }

    @Test
    void shouldConsumeEmptyCursor() {
        this.testConsume("MATCH (n:NoSuchLabel) RETURN n");
    }

    @Test
    void shouldConsumeNonEmptyCursor() {
        this.testConsume("RETURN 42");
    }

    @Test
    void shouldFailToRunQueryAfterCommit() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE (:MyLabel)");
        Assertions.assertNull(TestUtil.await(tx.commitAsync()));
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("MATCH (n:MyLabel) RETURN count(n)"));
        Assertions.assertEquals((int)1, (int)((Record)TestUtil.await(cursor.singleAsync())).get(0).asInt());
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.runAsync("CREATE (:MyOtherLabel)")));
        Assertions.assertEquals((Object)"Cannot run more queries in this transaction, it has been committed", (Object)e.getMessage());
    }

    @Test
    void shouldFailToRunQueryAfterRollback() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE (:MyLabel)");
        Assertions.assertNull(TestUtil.await(tx.rollbackAsync()));
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("MATCH (n:MyLabel) RETURN count(n)"));
        Assertions.assertEquals((int)0, (int)((Record)TestUtil.await(cursor.singleAsync())).get(0).asInt());
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.runAsync("CREATE (:MyOtherLabel)")));
        Assertions.assertEquals((Object)"Cannot run more queries in this transaction, it has been rolled back", (Object)e.getMessage());
    }

    @Test
    void shouldUpdateSessionBookmarkAfterCommit() {
        Bookmark bookmarkBefore = this.session.lastBookmark();
        TestUtil.await(this.session.beginTransactionAsync().thenCompose(tx -> tx.runAsync("CREATE (:MyNode)").thenCompose(ignore -> tx.commitAsync())));
        Bookmark bookmarkAfter = this.session.lastBookmark();
        Assertions.assertNotNull((Object)bookmarkAfter);
        Assertions.assertNotEquals((Object)bookmarkBefore, (Object)bookmarkAfter);
    }

    @Test
    void shouldFailToCommitWhenQueriesFail() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("CREATE (:TestNode)");
        tx.runAsync("CREATE (:TestNode)");
        tx.runAsync("RETURN 10 / 0");
        tx.runAsync("CREATE (:TestNode)");
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
        TestUtil.assertNoCircularReferences((Throwable)e);
        Assertions.assertEquals((Object)"Transaction can't be committed. It has been rolled back either because of an error or explicit termination", (Object)e.getMessage());
    }

    @Test
    void shouldFailToCommitWhenRunFailed() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("RETURN ILLEGAL");
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
        TestUtil.assertNoCircularReferences((Throwable)e);
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"Transaction can't be committed"));
    }

    @Test
    void shouldFailToCommitWhenBlockedRunFailed() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ClientException runException = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.runAsync("RETURN 42 / 0")));
        ClientException commitException = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
        MatcherAssert.assertThat((Object)runException.getMessage(), (Matcher)Matchers.containsString((String)"/ by zero"));
        TestUtil.assertNoCircularReferences((Throwable)commitException);
        MatcherAssert.assertThat((Object)commitException.getMessage(), (Matcher)Matchers.containsString((String)"Transaction can't be committed"));
    }

    @Test
    void shouldRollbackSuccessfullyWhenRunFailed() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("RETURN ILLEGAL");
        TestUtil.await(tx.rollbackAsync());
    }

    @Test
    void shouldRollbackSuccessfullyWhenBlockedRunFailed() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.runAsync("RETURN 42 / 0")));
        TestUtil.await(tx.rollbackAsync());
    }

    @Test
    void shouldPropagatePullAllFailureFromCommit() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("UNWIND [1, 2, 3, 'Hi'] AS x RETURN 10 / x");
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
        TestUtil.assertNoCircularReferences((Throwable)e);
        MatcherAssert.assertThat((Object)e.code(), (Matcher)Matchers.containsString((String)"TypeError"));
    }

    @Test
    void shouldPropagateBlockedPullAllFailureFromCommit() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        TestUtil.await(tx.runAsync("UNWIND [1, 2, 3, 'Hi'] AS x RETURN 10 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.commitAsync()));
        TestUtil.assertNoCircularReferences((Throwable)e);
        MatcherAssert.assertThat((Object)e.code(), (Matcher)Matchers.containsString((String)"TypeError"));
    }

    @Test
    void shouldPropagatePullAllFailureFromRollback() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        tx.runAsync("UNWIND [1, 2, 3, 'Hi'] AS x RETURN 10 / x");
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.rollbackAsync()));
        MatcherAssert.assertThat((Object)e.code(), (Matcher)Matchers.containsString((String)"TypeError"));
    }

    @Test
    void shouldPropagateBlockedPullAllFailureFromRollback() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        TestUtil.await(tx.runAsync("UNWIND [1, 2, 3, 'Hi'] AS x RETURN 10 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(tx.rollbackAsync()));
        MatcherAssert.assertThat((Object)e.code(), (Matcher)Matchers.containsString((String)"TypeError"));
    }

    @Test
    void shouldRollbackWhenPullAllFailureIsConsumed() {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync("UNWIND [1, 0] AS x RETURN 5 / x"));
        ClientException e = (ClientException)Assertions.assertThrows(ClientException.class, () -> TestUtil.await(cursor.consumeAsync()));
        MatcherAssert.assertThat((Object)e.getMessage(), (Matcher)Matchers.containsString((String)"/ by zero"));
        Assertions.assertNull(TestUtil.await(tx.rollbackAsync()));
    }

    private int countNodes(Object id) {
        ResultCursor cursor = (ResultCursor)TestUtil.await(this.session.runAsync("MATCH (n:Node {id: $id}) RETURN count(n)", Values.parameters((Object[])new Object[]{"id", id})));
        return ((Record)TestUtil.await(cursor.singleAsync())).get(0).asInt();
    }

    private void testForEach(String query, int expectedSeenRecords) {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync(query));
        AtomicInteger recordsSeen = new AtomicInteger();
        CompletionStage forEachDone = cursor.forEachAsync(record -> recordsSeen.incrementAndGet());
        ResultSummary summary = (ResultSummary)TestUtil.await(forEachDone);
        Assertions.assertNotNull((Object)summary);
        Assertions.assertEquals((Object)query, (Object)summary.query().text());
        Assertions.assertEquals(Collections.emptyMap(), (Object)summary.query().parameters().asMap());
        Assertions.assertEquals((int)expectedSeenRecords, (int)recordsSeen.get());
    }

    private <T> void testList(String query, List<T> expectedList) {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync(query));
        List records = (List)TestUtil.await(cursor.listAsync());
        ArrayList<Object> actualList = new ArrayList<Object>();
        for (Record record : records) {
            actualList.add(record.get(0).asObject());
        }
        Assertions.assertEquals(expectedList, actualList);
    }

    private void testConsume(String query) {
        AsyncTransaction tx = (AsyncTransaction)TestUtil.await(this.session.beginTransactionAsync());
        ResultCursor cursor = (ResultCursor)TestUtil.await(tx.runAsync(query));
        ResultSummary summary = (ResultSummary)TestUtil.await(cursor.consumeAsync());
        Assertions.assertNotNull((Object)summary);
        Assertions.assertEquals((Object)query, (Object)summary.query().text());
        Assertions.assertEquals(Collections.emptyMap(), (Object)summary.query().parameters().asMap());
        Assertions.assertThrows(ResultConsumedException.class, () -> TestUtil.await(cursor.nextAsync()));
    }
}

