/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.packstream;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.driver.internal.packstream.PackInput;
import org.neo4j.driver.internal.packstream.PackOutput;
import org.neo4j.driver.internal.packstream.PackStream;
import org.neo4j.driver.internal.packstream.PackType;
import org.neo4j.driver.internal.util.Iterables;
import org.neo4j.driver.internal.util.io.BufferedChannelInput;
import org.neo4j.driver.internal.util.io.ChannelOutput;

public class PackStreamTest {
    public static Map<String, Object> asMap(Object ... keysAndValues) {
        LinkedHashMap map = Iterables.newLinkedHashMapWithSize((int)(keysAndValues.length / 2));
        String key = null;
        for (Object keyOrValue : keysAndValues) {
            if (key == null) {
                key = keyOrValue.toString();
                continue;
            }
            map.put(key, keyOrValue);
            key = null;
        }
        return map;
    }

    private PackStream.Unpacker newUnpacker(byte[] bytes) {
        ByteArrayInputStream input = new ByteArrayInputStream(bytes);
        return new PackStream.Unpacker((PackInput)new BufferedChannelInput(Channels.newChannel(input)));
    }

    @Test
    void testCanPackAndUnpackNull() throws Throwable {
        Machine machine = new Machine();
        machine.packer().packNull();
        byte[] bytes = machine.output();
        MatcherAssert.assertThat((Object)bytes, (Matcher)CoreMatchers.equalTo((Object)new byte[]{-64}));
        PackStream.Unpacker unpacker = this.newUnpacker(bytes);
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.NULL));
    }

    @Test
    void testCanPackAndUnpackTrue() throws Throwable {
        Machine machine = new Machine();
        machine.packer().pack(true);
        byte[] bytes = machine.output();
        MatcherAssert.assertThat((Object)bytes, (Matcher)CoreMatchers.equalTo((Object)new byte[]{-61}));
        PackStream.Unpacker unpacker = this.newUnpacker(bytes);
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.BOOLEAN));
        MatcherAssert.assertThat((Object)unpacker.unpackBoolean(), (Matcher)CoreMatchers.equalTo((Object)true));
    }

    @Test
    void testCanPackAndUnpackFalse() throws Throwable {
        Machine machine = new Machine();
        machine.packer().pack(false);
        byte[] bytes = machine.output();
        MatcherAssert.assertThat((Object)bytes, (Matcher)CoreMatchers.equalTo((Object)new byte[]{-62}));
        PackStream.Unpacker unpacker = this.newUnpacker(bytes);
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.BOOLEAN));
        MatcherAssert.assertThat((Object)unpacker.unpackBoolean(), (Matcher)CoreMatchers.equalTo((Object)false));
    }

    @Test
    void testCanPackAndUnpackTinyIntegers() throws Throwable {
        Machine machine = new Machine();
        for (long i = -16L; i < 128L; ++i) {
            machine.reset();
            machine.packer().pack(i);
            byte[] bytes = machine.output();
            MatcherAssert.assertThat((Object)bytes.length, (Matcher)CoreMatchers.equalTo((Object)1));
            PackStream.Unpacker unpacker = this.newUnpacker(bytes);
            PackType packType = unpacker.peekNextType();
            MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.INTEGER));
            MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)i));
        }
    }

    @Test
    void testCanPackAndUnpackShortIntegers() throws Throwable {
        Machine machine = new Machine();
        for (long i = -32768L; i < 32768L; ++i) {
            machine.reset();
            machine.packer().pack(i);
            byte[] bytes = machine.output();
            MatcherAssert.assertThat((Object)bytes.length, (Matcher)Matchers.lessThanOrEqualTo((Comparable)Integer.valueOf(3)));
            PackStream.Unpacker unpacker = this.newUnpacker(bytes);
            PackType packType = unpacker.peekNextType();
            MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.INTEGER));
            MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)i));
        }
    }

    @Test
    void testCanPackAndUnpackPowersOfTwoAsIntegers() throws Throwable {
        Machine machine = new Machine();
        for (int i = 0; i < 32; ++i) {
            long n = (long)Math.pow(2.0, i);
            machine.reset();
            machine.packer().pack(n);
            PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
            PackType packType = unpacker.peekNextType();
            MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.INTEGER));
            MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)n));
        }
    }

    @Test
    void testCanPackAndUnpackPowersOfTwoPlusABitAsDoubles() throws Throwable {
        Machine machine = new Machine();
        for (int i = 0; i < 32; ++i) {
            double n = Math.pow(2.0, i) + 0.5;
            machine.reset();
            machine.packer().pack(n);
            PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
            PackType packType = unpacker.peekNextType();
            MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.FLOAT));
            MatcherAssert.assertThat((Object)unpacker.unpackDouble(), (Matcher)CoreMatchers.equalTo((Object)n));
        }
    }

    @Test
    void testCanPackAndUnpackPowersOfTwoMinusABitAsDoubles() throws Throwable {
        Machine machine = new Machine();
        for (int i = 0; i < 32; ++i) {
            double n = Math.pow(2.0, i) - 0.5;
            machine.reset();
            machine.packer().pack(n);
            PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
            PackType packType = unpacker.peekNextType();
            MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.FLOAT));
            MatcherAssert.assertThat((Object)unpacker.unpackDouble(), (Matcher)CoreMatchers.equalTo((Object)n));
        }
    }

    @Test
    void testCanPackAndUnpackByteArrays() throws Throwable {
        Machine machine = new Machine();
        this.testByteArrayPackingAndUnpacking(machine, 0);
        for (int i = 0; i < 24; ++i) {
            this.testByteArrayPackingAndUnpacking(machine, (int)Math.pow(2.0, i));
        }
    }

    private void testByteArrayPackingAndUnpacking(Machine machine, int length) throws Throwable {
        byte[] array = new byte[length];
        machine.reset();
        machine.packer().pack(array);
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.BYTES));
        Assertions.assertArrayEquals((byte[])array, (byte[])unpacker.unpackBytes());
    }

    @Test
    void testCanPackAndUnpackStrings() throws Throwable {
        Machine machine = new Machine();
        for (int i = 0; i < 24; ++i) {
            String string = new String(new byte[(int)Math.pow(2.0, i)]);
            machine.reset();
            machine.packer().pack(string);
            PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
            PackType packType = unpacker.peekNextType();
            MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.STRING));
            MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)string));
        }
    }

    @Test
    void testCanPackAndUnpackBytes() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.pack("ABCDEFGHIJ".getBytes());
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.BYTES));
        Assertions.assertArrayEquals((byte[])"ABCDEFGHIJ".getBytes(), (byte[])unpacker.unpackBytes());
    }

    @Test
    void testCanPackAndUnpackString() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.pack("ABCDEFGHIJ");
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.STRING));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"ABCDEFGHIJ"));
    }

    @Test
    void testCanPackAndUnpackSpecialString() throws Throwable {
        Machine machine = new Machine();
        String code = "Mj\u00f6lnir";
        PackStream.Packer packer = machine.packer();
        packer.pack(code);
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.STRING));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)code));
    }

    @Test
    void testCanPackAndUnpackListOneItemAtATime() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.packListHeader(3);
        packer.pack(12L);
        packer.pack(13L);
        packer.pack(14L);
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.LIST));
        MatcherAssert.assertThat((Object)unpacker.unpackListHeader(), (Matcher)CoreMatchers.equalTo((Object)3L));
        MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)12L));
        MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)13L));
        MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)14L));
    }

    @Test
    void testCanPackAndUnpackListOfString() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.pack(Arrays.asList("eins", "zwei", "drei"));
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.LIST));
        MatcherAssert.assertThat((Object)unpacker.unpackListHeader(), (Matcher)CoreMatchers.equalTo((Object)3L));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"eins"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"zwei"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"drei"));
    }

    @Test
    void testCanPackAndUnpackListOfSpecialStrings() throws Throwable {
        this.assertPackStringLists(3, "Mj\u00f6lnir");
        this.assertPackStringLists(126, "Mj\u00f6lnir");
        this.assertPackStringLists(3000, "Mj\u00f6lnir");
        this.assertPackStringLists(32768, "Mj\u00f6lnir");
    }

    @Test
    void testCanPackAndUnpackListOfStringOneByOne() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.packListHeader(3);
        packer.pack("eins");
        packer.pack("zwei");
        packer.pack("drei");
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.LIST));
        MatcherAssert.assertThat((Object)unpacker.unpackListHeader(), (Matcher)CoreMatchers.equalTo((Object)3L));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"eins"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"zwei"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"drei"));
    }

    @Test
    void testCanPackAndUnpackListOfSpecialStringOneByOne() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.packListHeader(3);
        packer.pack("Mj\u00f6lnir");
        packer.pack("Mj\u00f6lnir");
        packer.pack("Mj\u00f6lnir");
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.LIST));
        MatcherAssert.assertThat((Object)unpacker.unpackListHeader(), (Matcher)CoreMatchers.equalTo((Object)3L));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"Mj\u00f6lnir"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"Mj\u00f6lnir"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"Mj\u00f6lnir"));
    }

    @Test
    void testCanPackAndUnpackMap() throws Throwable {
        this.assertMap(2);
        this.assertMap(126);
        this.assertMap(2439);
        this.assertMap(32768);
    }

    @Test
    void testCanPackAndUnpackStruct() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.packStructHeader(3, (byte)78);
        packer.pack(12L);
        packer.pack(Arrays.asList("Person", "Employee"));
        packer.pack(PackStreamTest.asMap("name", "Alice", "age", 33));
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.STRUCT));
        MatcherAssert.assertThat((Object)unpacker.unpackStructHeader(), (Matcher)CoreMatchers.equalTo((Object)3L));
        MatcherAssert.assertThat((Object)unpacker.unpackStructSignature(), (Matcher)CoreMatchers.equalTo((Object)78));
        MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)12L));
        MatcherAssert.assertThat((Object)unpacker.unpackListHeader(), (Matcher)CoreMatchers.equalTo((Object)2L));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"Person"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"Employee"));
        MatcherAssert.assertThat((Object)unpacker.unpackMapHeader(), (Matcher)CoreMatchers.equalTo((Object)2L));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"name"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"Alice"));
        MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"age"));
        MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)33L));
    }

    @Test
    void testCanPackAndUnpackStructsOfDifferentSizes() throws Throwable {
        this.assertStruct(2);
        this.assertStruct(126);
        this.assertStruct(2439);
        Assertions.assertThrows(PackStream.Overflow.class, () -> this.assertStruct(65536));
    }

    @Test
    void testCanDoStreamingListUnpacking() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.pack(Arrays.asList(1, 2, 3, Arrays.asList(4, 5)));
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        long size = unpacker.unpackListHeader();
        long a = unpacker.unpackLong();
        long b = unpacker.unpackLong();
        long c = unpacker.unpackLong();
        long innerSize = unpacker.unpackListHeader();
        long d = unpacker.unpackLong();
        long e = unpacker.unpackLong();
        Assertions.assertEquals((long)4L, (long)size);
        Assertions.assertEquals((long)2L, (long)innerSize);
        Assertions.assertEquals((long)1L, (long)a);
        Assertions.assertEquals((long)2L, (long)b);
        Assertions.assertEquals((long)3L, (long)c);
        Assertions.assertEquals((long)4L, (long)d);
        Assertions.assertEquals((long)5L, (long)e);
    }

    @Test
    void testCanDoStreamingStructUnpacking() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.packStructHeader(4, (byte)126);
        packer.pack(1L);
        packer.pack(2L);
        packer.pack(3L);
        packer.pack(Arrays.asList(4, 5));
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        long size = unpacker.unpackStructHeader();
        byte signature = unpacker.unpackStructSignature();
        long a = unpacker.unpackLong();
        long b = unpacker.unpackLong();
        long c = unpacker.unpackLong();
        long innerSize = unpacker.unpackListHeader();
        long d = unpacker.unpackLong();
        long e = unpacker.unpackLong();
        Assertions.assertEquals((long)4L, (long)size);
        Assertions.assertEquals((int)126, (int)signature);
        Assertions.assertEquals((long)2L, (long)innerSize);
        Assertions.assertEquals((long)1L, (long)a);
        Assertions.assertEquals((long)2L, (long)b);
        Assertions.assertEquals((long)3L, (long)c);
        Assertions.assertEquals((long)4L, (long)d);
        Assertions.assertEquals((long)5L, (long)e);
    }

    @Test
    void testCanDoStreamingMapUnpacking() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.packMapHeader(2);
        packer.pack("name");
        packer.pack("Bob");
        packer.pack("cat_ages");
        packer.pack(Arrays.asList(4.3, true));
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        long size = unpacker.unpackMapHeader();
        String k1 = unpacker.unpackString();
        String v1 = unpacker.unpackString();
        String k2 = unpacker.unpackString();
        long innerSize = unpacker.unpackListHeader();
        double d = unpacker.unpackDouble();
        boolean e = unpacker.unpackBoolean();
        Assertions.assertEquals((long)2L, (long)size);
        Assertions.assertEquals((long)2L, (long)innerSize);
        Assertions.assertEquals((Object)"name", (Object)k1);
        Assertions.assertEquals((Object)"Bob", (Object)v1);
        Assertions.assertEquals((Object)"cat_ages", (Object)k2);
        Assertions.assertEquals((double)4.3, (double)d, (double)1.0E-4);
        Assertions.assertTrue((boolean)e);
    }

    @Test
    void handlesDataCrossingBufferBoundaries() throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.pack(Long.MAX_VALUE);
        packer.pack(Long.MAX_VALUE);
        ReadableByteChannel ch = Channels.newChannel(new ByteArrayInputStream(machine.output()));
        PackStream.Unpacker unpacker = new PackStream.Unpacker((PackInput)new BufferedChannelInput(11, ch));
        long first = unpacker.unpackLong();
        long second = unpacker.unpackLong();
        Assertions.assertEquals((long)Long.MAX_VALUE, (long)first);
        Assertions.assertEquals((long)Long.MAX_VALUE, (long)second);
    }

    @Test
    void testCanPeekOnNextType() throws Throwable {
        this.assertPeekType(PackType.STRING, "a string");
        this.assertPeekType(PackType.INTEGER, 123);
        this.assertPeekType(PackType.FLOAT, 123.123);
        this.assertPeekType(PackType.BOOLEAN, true);
        this.assertPeekType(PackType.LIST, Arrays.asList(1, 2, 3));
        this.assertPeekType(PackType.MAP, PackStreamTest.asMap("l", 3));
    }

    @Test
    void shouldFailForUnknownValue() throws IOException {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        Assertions.assertThrows(PackStream.UnPackable.class, () -> packer.pack((Object)new MyRandomClass()));
    }

    private void assertPeekType(PackType type, Object value) throws IOException {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.pack(value);
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        Assertions.assertEquals((Object)type, (Object)unpacker.peekNextType());
    }

    private void assertPackStringLists(int size, String value) throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        ArrayList<String> strings = new ArrayList<String>(size);
        for (int i = 0; i < size; ++i) {
            strings.add(i, value);
        }
        packer.pack(strings);
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.LIST));
        MatcherAssert.assertThat((Object)unpacker.unpackListHeader(), (Matcher)CoreMatchers.equalTo((Object)size));
        for (int i = 0; i < size; ++i) {
            MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)"Mj\u00f6lnir"));
        }
    }

    private void assertMap(int size) throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        HashMap<String, Integer> map = new HashMap<String, Integer>();
        for (int i = 0; i < size; ++i) {
            map.put(Integer.toString(i), i);
        }
        packer.pack(map);
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.MAP));
        MatcherAssert.assertThat((Object)unpacker.unpackMapHeader(), (Matcher)CoreMatchers.equalTo((Object)size));
        for (int i = 0; i < size; ++i) {
            MatcherAssert.assertThat((Object)unpacker.unpackString(), (Matcher)CoreMatchers.equalTo((Object)Long.toString(unpacker.unpackLong())));
        }
    }

    private void assertStruct(int size) throws Throwable {
        Machine machine = new Machine();
        PackStream.Packer packer = machine.packer();
        packer.packStructHeader(size, (byte)78);
        for (int i = 0; i < size; ++i) {
            packer.pack((long)i);
        }
        PackStream.Unpacker unpacker = this.newUnpacker(machine.output());
        PackType packType = unpacker.peekNextType();
        MatcherAssert.assertThat((Object)packType, (Matcher)CoreMatchers.equalTo((Object)PackType.STRUCT));
        MatcherAssert.assertThat((Object)unpacker.unpackStructHeader(), (Matcher)CoreMatchers.equalTo((Object)size));
        MatcherAssert.assertThat((Object)unpacker.unpackStructSignature(), (Matcher)CoreMatchers.equalTo((Object)78));
        for (int i = 0; i < size; ++i) {
            MatcherAssert.assertThat((Object)unpacker.unpackLong(), (Matcher)CoreMatchers.equalTo((Object)i));
        }
    }

    private static class MyRandomClass {
        private MyRandomClass() {
        }
    }

    private static class Machine {
        private final ByteArrayOutputStream output = new ByteArrayOutputStream();
        private final WritableByteChannel writable = Channels.newChannel(this.output);
        private final PackStream.Packer packer = new PackStream.Packer((PackOutput)new ChannelOutput(this.writable));

        Machine() {
        }

        public void reset() {
            this.output.reset();
        }

        public byte[] output() {
            return this.output.toByteArray();
        }

        PackStream.Packer packer() {
            return this.packer;
        }
    }
}

