/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons;

import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import com.google.common.primitives.Doubles;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public class Histogram {
    private final double min;
    private final double max;
    private final int numColumns;
    private final int[] counts;
    private final double[] percentages;

    Histogram(double min, double max, int numColumns, double[] data) {
        Preconditions.checkArgument((max > min ? 1 : 0) != 0);
        Preconditions.checkArgument((numColumns > 0 ? 1 : 0) != 0);
        this.min = min;
        this.max = max;
        this.numColumns = numColumns;
        this.counts = this.createCountsArray(data);
        this.percentages = this.computePercentages(this.counts);
    }

    private int[] createCountsArray(double[] dataInput) {
        int[] columnCounts = new int[this.numColumns];
        for (double score : dataInput) {
            double fraction = (score - this.min) / (this.max - this.min);
            int bucket = (int)((double)this.numColumns * fraction);
            bucket = Math.min(this.numColumns - 1, bucket);
            int n = bucket = Math.max(0, bucket);
            columnCounts[n] = columnCounts[n] + 1;
        }
        return columnCounts;
    }

    private double[] computePercentages(int[] counts) {
        double totalCount = IntStream.of(counts).sum();
        double[] fractionsOfTotal = new double[this.numColumns];
        for (int i = 0; i < this.numColumns; ++i) {
            fractionsOfTotal[i] = (double)counts[i] / totalCount;
        }
        return fractionsOfTotal;
    }

    public double earthMoverDistanceTo(Histogram other) {
        Preconditions.checkNotNull((Object)other);
        Preconditions.checkArgument((boolean)DoubleMath.fuzzyEquals((double)other.min, (double)this.min, (double)1.0E-6));
        Preconditions.checkArgument((boolean)DoubleMath.fuzzyEquals((double)other.max, (double)this.max, (double)1.0E-6));
        Preconditions.checkArgument((other.numColumns == this.numColumns ? 1 : 0) != 0);
        return Histogram.earthMoverDistanceBetween(this.percentages, other.percentages);
    }

    private static double earthMoverDistanceBetween(double[] percentages1, double[] percentages2) {
        double totalEarthMoved = 0.0;
        double currentExcessOrDearth = 0.0;
        int numColumns = percentages1.length;
        for (int i = 0; i < numColumns; ++i) {
            totalEarthMoved += Math.abs(currentExcessOrDearth += percentages1[i] - percentages2[i]);
        }
        Preconditions.checkArgument((boolean)DoubleMath.fuzzyEquals((double)currentExcessOrDearth, (double)0.0, (double)5.0E-4), (Object)("FAILURE: Earth Mover Distance does not work when the input arrays have different sums, the difference was: " + currentExcessOrDearth));
        return totalEarthMoved;
    }

    public double min() {
        return this.min;
    }

    public double max() {
        return this.max;
    }

    public int numColumns() {
        return this.numColumns;
    }

    public double columnWidth() {
        return (this.max - this.min) / (double)this.numColumns;
    }

    public String toString(int digitsAfterDecimalPlace) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < this.numColumns; ++i) {
            sb.append(this.describeColumn(i, digitsAfterDecimalPlace) + "\n");
        }
        return sb.toString();
    }

    public String toString() {
        return this.toString(3);
    }

    public String describeColumn(int i, int digitsAfterDecimalPlace) {
        this.checkIndexBounds(i);
        String minSide = i == 0 ? String.format("-[%." + digitsAfterDecimalPlace + "f", this.floorOfColumn(i)) : String.format("[%." + digitsAfterDecimalPlace + "f", this.floorOfColumn(i));
        String maxSide = i == this.numColumns - 1 ? String.format("%." + digitsAfterDecimalPlace + "f]+", this.ceilingOfColumn(i)) : String.format("%." + digitsAfterDecimalPlace + "f]", this.ceilingOfColumn(i));
        String count = " : " + this.counts[i];
        return minSide + "-" + maxSide + count;
    }

    public double floorOfColumn(int i) {
        this.checkIndexBounds(i);
        return (double)i * this.columnWidth() + this.min;
    }

    public double ceilingOfColumn(int i) {
        this.checkIndexBounds(i);
        return (double)(i + 1) * this.columnWidth() + this.min;
    }

    private void checkIndexBounds(int i) {
        Preconditions.checkArgument((i >= 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((i < this.numColumns ? 1 : 0) != 0);
    }

    public int[] counts() {
        return this.counts;
    }

    public double[] percentages() {
        return this.percentages;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Double minimum;
        private Double maximum;
        private Integer numColumns;
        private double[] data;

        public Builder min(double min) {
            Preconditions.checkState((boolean)Objects.isNull(this.minimum), (Object)"Minimum was already set");
            this.minimum = min;
            return this;
        }

        public Builder max(double max) {
            Preconditions.checkState((boolean)Objects.isNull(this.maximum), (Object)"Maximum was already set");
            this.maximum = max;
            return this;
        }

        public Builder numColumns(int numColumns) {
            Preconditions.checkState((boolean)Objects.isNull(this.numColumns), (Object)"Number of columns were already set");
            this.numColumns = numColumns;
            return this;
        }

        public Builder fromRawData(double[] data) {
            Preconditions.checkState((boolean)Objects.isNull(this.data), (Object)"Data import was already set");
            this.data = data;
            return this;
        }

        public <T> Builder fromCollection(Collection<T> data, Function<T, Double> dataScorer) {
            Preconditions.checkState((boolean)Objects.isNull(this.data), (Object)"Data import was already set");
            List scoredData = data.stream().map(dataScorer).filter(Objects::nonNull).collect(Collectors.toList());
            this.data = Doubles.toArray(scoredData);
            return this;
        }

        public Histogram build() {
            Preconditions.checkState((boolean)Objects.nonNull(this.minimum), (Object)"Building a Histogram requires setting the minimum");
            Preconditions.checkState((boolean)Objects.nonNull(this.maximum), (Object)"Building a Histogram requires setting the maximum");
            Preconditions.checkState((boolean)Objects.nonNull(this.numColumns), (Object)"Building a Histogram requires setting numColumns");
            Preconditions.checkState((boolean)Objects.nonNull(this.data), (Object)"Building a Histogram requires adding data");
            return new Histogram(this.minimum, this.maximum, this.numColumns, this.data);
        }
    }
}

