/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons;

import com.google.common.base.Preconditions;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.Instant;
import java.util.Base64;
import java.util.Objects;
import org.mitre.caasd.commons.Acceleration;
import org.mitre.caasd.commons.Course;
import org.mitre.caasd.commons.Distance;
import org.mitre.caasd.commons.HasPosition;
import org.mitre.caasd.commons.HasTime;
import org.mitre.caasd.commons.LatLong;
import org.mitre.caasd.commons.Speed;

public class KineticPosition
implements HasTime,
HasPosition {
    private static final Base64.Encoder BASE_64_ENCODER = Base64.getEncoder().withoutPadding();
    private final long epochTime;
    private final double latitude;
    private final double longitude;
    private final double altitudeInFeet;
    private final double climbRateInFtPerMin;
    private final double courseInDegrees;
    private final double turnRateInDegreesPerSecond;
    private final double speedInKnots;
    private final double accelerationInKnotsPerSec;

    public KineticPosition(Instant time, LatLong location, Distance altitude, Speed climbRate, Course course, double turnRate, Speed speed, Acceleration accel) {
        Objects.requireNonNull(time);
        Objects.requireNonNull(location);
        Objects.requireNonNull(altitude);
        Objects.requireNonNull(speed);
        this.epochTime = time.toEpochMilli();
        this.latitude = location.latitude();
        this.longitude = location.longitude();
        this.altitudeInFeet = altitude.inFeet();
        this.climbRateInFtPerMin = climbRate.inFeetPerMinutes();
        this.courseInDegrees = course.inDegrees();
        this.speedInKnots = speed.inKnots();
        this.accelerationInKnotsPerSec = accel.speedDeltaPerSecond().inKnots();
        this.turnRateInDegreesPerSecond = turnRate;
    }

    public KineticPosition(byte[] seventyTwoBytes) {
        Objects.requireNonNull(seventyTwoBytes);
        Preconditions.checkArgument((seventyTwoBytes.length == 72 ? 1 : 0) != 0, (Object)"Must use exactly 72 bytes");
        ByteBuffer buffer = ByteBuffer.wrap(seventyTwoBytes);
        this.epochTime = buffer.getLong();
        this.latitude = buffer.getDouble();
        this.longitude = buffer.getDouble();
        this.altitudeInFeet = buffer.getDouble();
        this.climbRateInFtPerMin = buffer.getDouble();
        this.courseInDegrees = buffer.getDouble();
        this.turnRateInDegreesPerSecond = buffer.getDouble();
        this.speedInKnots = buffer.getDouble();
        this.accelerationInKnotsPerSec = buffer.getDouble();
        LatLong.checkLatitude(this.latitude);
        LatLong.checkLongitude(this.longitude);
    }

    @Override
    public Instant time() {
        return Instant.ofEpochMilli(this.epochTime);
    }

    @Override
    public LatLong latLong() {
        return LatLong.of(this.latitude, this.longitude);
    }

    public Distance altitude() {
        return Distance.ofFeet(this.altitudeInFeet);
    }

    public Speed climbRate() {
        return Speed.of(this.climbRateInFtPerMin, Speed.Unit.FEET_PER_MINUTE);
    }

    public Speed speed() {
        return Speed.of(this.speedInKnots, Speed.Unit.KNOTS);
    }

    public Acceleration acceleration() {
        return Acceleration.of(Speed.ofKnots(this.accelerationInKnotsPerSec));
    }

    public Course course() {
        return Course.ofDegrees(this.courseInDegrees);
    }

    public double turnRate() {
        return this.turnRateInDegreesPerSecond;
    }

    public Distance turnRadius() {
        return KineticPosition.turnRadius(this.speed(), this.turnRate());
    }

    public static Distance turnRadius(Speed speed, double turnRateInDegreesPerSecond) {
        Objects.requireNonNull(speed);
        if (turnRateInDegreesPerSecond == 0.0) {
            return Distance.ofNauticalMiles(Double.POSITIVE_INFINITY);
        }
        double secToTurn360 = 360.0 / Math.abs(turnRateInDegreesPerSecond);
        long asMilliSec = (long)secToTurn360 * 1000L;
        Distance distTraveled = speed.times(Duration.ofMillis(asMilliSec));
        Distance radius = distTraveled.times(0.15915494309189535);
        return turnRateInDegreesPerSecond > 0.0 ? radius : radius.times(-1.0);
    }

    public byte[] toBytes() {
        return ByteBuffer.allocate(72).putLong(this.epochTime).putDouble(this.latitude).putDouble(this.longitude).putDouble(this.altitudeInFeet).putDouble(this.climbRateInFtPerMin).putDouble(this.courseInDegrees).putDouble(this.turnRateInDegreesPerSecond).putDouble(this.speedInKnots).putDouble(this.accelerationInKnotsPerSec).array();
    }

    public String toBase64() {
        return BASE_64_ENCODER.encodeToString(this.toBytes());
    }

    public static KineticPosition fromBase64(String str) {
        byte[] bytes = Base64.getDecoder().decode(str);
        return new KineticPosition(bytes);
    }

    public static KineticPosition fromBytes(byte[] seventyTwoBytes) {
        return new KineticPosition(seventyTwoBytes);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        KineticPosition that = (KineticPosition)o;
        if (this.epochTime != that.epochTime) {
            return false;
        }
        if (Double.compare(that.latitude, this.latitude) != 0) {
            return false;
        }
        if (Double.compare(that.longitude, this.longitude) != 0) {
            return false;
        }
        if (Double.compare(that.altitudeInFeet, this.altitudeInFeet) != 0) {
            return false;
        }
        if (Double.compare(that.climbRateInFtPerMin, this.climbRateInFtPerMin) != 0) {
            return false;
        }
        if (Double.compare(that.courseInDegrees, this.courseInDegrees) != 0) {
            return false;
        }
        if (Double.compare(that.turnRateInDegreesPerSecond, this.turnRateInDegreesPerSecond) != 0) {
            return false;
        }
        if (Double.compare(that.speedInKnots, this.speedInKnots) != 0) {
            return false;
        }
        return Double.compare(that.accelerationInKnotsPerSec, this.accelerationInKnotsPerSec) == 0;
    }

    public int hashCode() {
        int result = (int)(this.epochTime ^ this.epochTime >>> 32);
        long temp = Double.doubleToLongBits(this.latitude);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.longitude);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.altitudeInFeet);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.climbRateInFtPerMin);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.courseInDegrees);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.turnRateInDegreesPerSecond);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.speedInKnots);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        temp = Double.doubleToLongBits(this.accelerationInKnotsPerSec);
        result = 31 * result + (int)(temp ^ temp >>> 32);
        return result;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(KineticPosition seed) {
        return KineticPosition.builder().time(seed.time()).latLong(seed.latLong()).altitude(seed.altitude()).climbRate(seed.climbRate()).course(seed.course()).turnRate(seed.turnRate()).speed(seed.speed()).acceleration(seed.acceleration());
    }

    public static class Builder {
        private Long epochTime;
        private LatLong latLong;
        private Double altitudeInFeet;
        private Double climbRateInFtPerMin;
        private Double speedInKnots;
        private Double accelerationInKnotsPerSec;
        private Double courseInDegrees;
        private Double turnRateInDegreesPerSec;

        public KineticPosition build() {
            Objects.requireNonNull(this.epochTime, "epochTime was not set");
            Objects.requireNonNull(this.latLong, "latLong was not set");
            Objects.requireNonNull(this.altitudeInFeet, "altitude was not set");
            Objects.requireNonNull(this.climbRateInFtPerMin, "climbRate was not set");
            Objects.requireNonNull(this.speedInKnots, "speed was not set");
            Objects.requireNonNull(this.accelerationInKnotsPerSec, "acceleration was not set");
            Objects.requireNonNull(this.courseInDegrees, "course was not set");
            Objects.requireNonNull(this.turnRateInDegreesPerSec, "turn rate was not set");
            return new KineticPosition(Instant.ofEpochMilli(this.epochTime), this.latLong, Distance.ofFeet(this.altitudeInFeet), Speed.of(this.climbRateInFtPerMin, Speed.Unit.FEET_PER_MINUTE), Course.ofDegrees(this.courseInDegrees), this.turnRateInDegreesPerSec, Speed.of(this.speedInKnots, Speed.Unit.KNOTS), Acceleration.of(Speed.ofKnots(this.accelerationInKnotsPerSec)));
        }

        public Builder time(Instant time) {
            Preconditions.checkState((boolean)Objects.isNull(this.epochTime), (Object)"time was already set");
            return this.butTime(time);
        }

        public Builder butTime(Instant time) {
            this.epochTime = time.toEpochMilli();
            return this;
        }

        public Builder latLong(LatLong location) {
            Preconditions.checkState((boolean)Objects.isNull(this.latLong), (Object)"latLong was already set");
            return this.butLatLong(location);
        }

        public Builder butLatLong(LatLong location) {
            Objects.requireNonNull(location);
            return this.butLatLong(location.latitude(), location.longitude());
        }

        public Builder latLong(double latitude, double longitude) {
            Preconditions.checkState((boolean)Objects.isNull(this.latLong), (Object)"latLong was already set");
            return this.butLatLong(latitude, longitude);
        }

        public Builder butLatLong(double latitude, double longitude) {
            LatLong.checkLatitude(latitude);
            LatLong.checkLongitude(longitude);
            this.latLong = LatLong.of(latitude, longitude);
            return this;
        }

        public Builder altitude(Distance altitude) {
            Preconditions.checkState((boolean)Objects.isNull(this.altitudeInFeet), (Object)"altitude was already set");
            return this.butAltitude(altitude);
        }

        public Builder butAltitude(Distance altitude) {
            this.altitudeInFeet = altitude.inFeet();
            return this;
        }

        public Builder speed(Speed speed) {
            Preconditions.checkState((boolean)Objects.isNull(this.speedInKnots), (Object)"speed was already set");
            return this.butSpeed(speed);
        }

        public Builder butSpeed(Speed speed) {
            this.speedInKnots = speed.inKnots();
            return this;
        }

        public Builder acceleration(Acceleration accel) {
            Preconditions.checkState((boolean)Objects.isNull(this.accelerationInKnotsPerSec), (Object)"acceleration was already set");
            return this.butAcceleration(accel);
        }

        public Builder butAcceleration(Acceleration accel) {
            this.accelerationInKnotsPerSec = accel.speedDeltaPerSecond().inKnots();
            return this;
        }

        public Builder climbRate(Speed climbRate) {
            Preconditions.checkState((boolean)Objects.isNull(this.climbRateInFtPerMin), (Object)"climbRate was already set");
            return this.butClimbRate(climbRate);
        }

        public Builder butClimbRate(Speed climbRate) {
            this.climbRateInFtPerMin = climbRate.inFeetPerMinutes();
            return this;
        }

        public Builder course(Course course) {
            Preconditions.checkState((boolean)Objects.isNull(this.courseInDegrees), (Object)"course was already set");
            return this.butCourse(course);
        }

        public Builder butCourse(Course course) {
            this.courseInDegrees = course.inDegrees();
            return this;
        }

        public Builder turnRate(double degreesPerSec) {
            Preconditions.checkState((boolean)Objects.isNull(this.turnRateInDegreesPerSec), (Object)"turnRate was already set");
            return this.butTurnRate(degreesPerSec);
        }

        public Builder butTurnRate(double degreesPerSec) {
            this.turnRateInDegreesPerSec = degreesPerSec;
            return this;
        }
    }
}

