/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons;

import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Base64;
import java.util.Collection;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.stream.Stream;
import org.apache.commons.math3.util.FastMath;
import org.mitre.caasd.commons.Course;
import org.mitre.caasd.commons.Distance;
import org.mitre.caasd.commons.LatLong64;
import org.mitre.caasd.commons.Spherical;

public class LatLong
implements Comparable<LatLong>,
Serializable {
    private static final long serialVersionUID = 1L;
    private final double latitude;
    private final double longitude;
    private static final String COLLECTION_CANNOT_BE_NULL = "The collection of LatLong locations cannot be null";
    private static final String COLLECTION_CANNOT_BE_EMPTY = "The collection of LatLong locations cannot be empty";

    private LatLong() {
        this(0.0, 0.0);
    }

    public LatLong(Double latitude, Double longitude) {
        Preconditions.checkNotNull((Object)latitude);
        Preconditions.checkNotNull((Object)longitude);
        LatLong.checkLatitude(latitude);
        LatLong.checkLongitude(longitude);
        this.latitude = latitude;
        this.longitude = longitude;
    }

    public static LatLong of(Double latitude, Double longitude) {
        return new LatLong(latitude, longitude);
    }

    public static LatLong fromBytes(byte[] exactly16Bytes) {
        int i;
        Objects.requireNonNull(exactly16Bytes);
        Preconditions.checkArgument((exactly16Bytes.length == 16 ? 1 : 0) != 0, (Object)"Must use exactly 16 bytes");
        long bigBits = 0L;
        long smallBits = 0L;
        for (i = 0; i < 8; ++i) {
            bigBits = bigBits << 8 | (long)(exactly16Bytes[i] & 0xFF);
        }
        for (i = 8; i < 16; ++i) {
            smallBits = smallBits << 8 | (long)(exactly16Bytes[i] & 0xFF);
        }
        double longitude = Double.longBitsToDouble(bigBits);
        double latitude = Double.longBitsToDouble(smallBits);
        return LatLong.of(longitude, latitude);
    }

    public static LatLong fromBase64Str(String base64Encoding) {
        return LatLong.fromBytes(Base64.getUrlDecoder().decode(base64Encoding));
    }

    public static void checkLatitude(double latitude) {
        if (!(-90.0 <= latitude) || !(latitude <= 90.0)) {
            throw new IllegalArgumentException("Latitude is out of range: " + latitude);
        }
    }

    public static void checkLongitude(double longitude) {
        if (!(-180.0 <= longitude) || !(longitude <= 180.0)) {
            throw new IllegalArgumentException("Longitude is out of range: " + longitude);
        }
    }

    public double latitude() {
        return this.latitude;
    }

    public double longitude() {
        return this.longitude;
    }

    public String toString() {
        return "(" + this.latitude + "," + this.longitude + ")";
    }

    public byte[] toBytes() {
        return ByteBuffer.allocate(16).putDouble(this.latitude).putDouble(this.longitude).array();
    }

    public String toBase64() {
        return Base64.getUrlEncoder().withoutPadding().encodeToString(this.toBytes());
    }

    public LatLong64 compress() {
        return LatLong64.fromLatLong(this);
    }

    public Distance distanceTo(LatLong that) {
        return Distance.ofNauticalMiles(this.distanceInNM(that));
    }

    public double distanceInNM(LatLong that) {
        return Spherical.distanceInNM(this, that);
    }

    public double courseInDegrees(LatLong that) {
        return Spherical.courseInDegrees(this.latitude, this.longitude, that.latitude(), that.longitude());
    }

    public Course courseTo(LatLong that) {
        return Spherical.courseBtw(this, that);
    }

    public boolean isWithin(Distance distance, LatLong location) {
        return this.distanceTo(location).isLessThanOrEqualTo(distance);
    }

    public LatLong projectOut(Double direction, Double distance) {
        return Spherical.projectOut(this.latitude, this.longitude, direction, distance);
    }

    public LatLong project(Course direction, Distance distance) {
        return Spherical.projectOut(this, direction, distance);
    }

    public LatLong projectOut(Double direction, Double distance, Double curvature) {
        return Spherical.projectOut(this.latitude, this.longitude, direction, distance, curvature);
    }

    public LatLong greatCircleOrigin(Double course) {
        return Spherical.greatCircleOrigin(this.latitude, this.longitude, course);
    }

    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + (int)(Double.doubleToLongBits(this.latitude) ^ Double.doubleToLongBits(this.latitude) >>> 32);
        hash = 89 * hash + (int)(Double.doubleToLongBits(this.longitude) ^ Double.doubleToLongBits(this.longitude) >>> 32);
        return hash;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        LatLong other = (LatLong)obj;
        if (Double.doubleToLongBits(this.latitude) != Double.doubleToLongBits(other.latitude)) {
            return false;
        }
        return Double.doubleToLongBits(this.longitude) == Double.doubleToLongBits(other.longitude);
    }

    public static LatLong avgLatLong(LatLong one, LatLong two) {
        Preconditions.checkNotNull((Object)one);
        Preconditions.checkNotNull((Object)two);
        SphericalUnitVector vectorOne = new SphericalUnitVector(one);
        SphericalUnitVector vectorTwo = new SphericalUnitVector(two);
        double avgX = (vectorOne.x + vectorTwo.x) / 2.0;
        double avgY = (vectorOne.y + vectorTwo.y) / 2.0;
        double avgZ = (vectorOne.z + vectorTwo.z) / 2.0;
        double avgLong = FastMath.atan2((double)avgY, (double)avgX);
        double avgSqareRoot = FastMath.hypot((double)avgX, (double)avgY);
        double avgLat = FastMath.atan2((double)avgZ, (double)avgSqareRoot);
        return LatLong.of(Math.toDegrees(avgLat), Math.toDegrees(avgLong));
    }

    public static LatLong quickAvgLatLong(LatLong one, LatLong two) {
        double averageLat = (one.latitude + two.latitude) / 2.0;
        double averageLong = Math.abs(one.longitude - two.longitude) > 180.0 ? (one.longitude + 180.0 + (two.longitude + 180.0)) / 2.0 : (one.longitude + two.longitude) / 2.0;
        return LatLong.of(averageLat, averageLong);
    }

    public static LatLong avgLatLong(LatLong ... locations) {
        Objects.requireNonNull(locations);
        if (locations.length == 0) {
            throw new NoSuchElementException("Average LatLong not defined when empty");
        }
        double x = 0.0;
        double y = 0.0;
        double z = 0.0;
        for (LatLong location : locations) {
            SphericalUnitVector vector = new SphericalUnitVector(location);
            x += vector.x;
            y += vector.y;
            z += vector.z;
        }
        double avgLong = FastMath.atan2((double)(y /= (double)locations.length), (double)(x /= (double)locations.length));
        double avgSqareRoot = FastMath.hypot((double)x, (double)y);
        double avgLat = FastMath.atan2((double)(z /= (double)locations.length), (double)avgSqareRoot);
        return LatLong.of(Math.toDegrees(avgLat), Math.toDegrees(avgLong));
    }

    public static LatLong avgLatLong(Collection<LatLong> locations) {
        Objects.requireNonNull(locations);
        LatLong[] asArray = locations.toArray(new LatLong[0]);
        return LatLong.avgLatLong(asArray);
    }

    public static LatLong quickAvgLatLong(LatLong ... locations) {
        Objects.requireNonNull(locations);
        if (locations.length == 0) {
            throw new NoSuchElementException("The input array was empty");
        }
        if (locations.length == 1) {
            return locations[0];
        }
        double avgLatitude = Stream.of(locations).mapToDouble(loc -> loc.latitude).average().getAsDouble();
        double[] longitudes = Stream.of(locations).mapToDouble(loc -> loc.longitude()).toArray();
        double curAvgLongitude = longitudes[0];
        for (int i = 1; i < longitudes.length; ++i) {
            curAvgLongitude = LatLong.avgLong(curAvgLongitude, i, longitudes[i], 1);
        }
        return LatLong.of(avgLatitude, curAvgLongitude);
    }

    public static LatLong quickAvgLatLong(Collection<LatLong> locations) {
        Objects.requireNonNull(locations);
        LatLong[] asArray = locations.toArray(new LatLong[0]);
        return LatLong.quickAvgLatLong(asArray);
    }

    private static double avgLong(double longitudeA, int weightA, double longitudeB, int weightB) {
        double w1 = (double)weightA / (double)(weightA + weightB);
        double w2 = (double)weightB / (double)(weightA + weightB);
        double averageLong = Math.abs(longitudeA - longitudeB) > 180.0 ? w1 * (longitudeA + 180.0) + w2 * (longitudeB + 180.0) : w1 * longitudeA + w2 * longitudeB;
        return averageLong;
    }

    public static Double maxLatitude(Collection<LatLong> locations) {
        Preconditions.checkNotNull(locations, (Object)COLLECTION_CANNOT_BE_NULL);
        Preconditions.checkArgument((!locations.isEmpty() ? 1 : 0) != 0, (Object)COLLECTION_CANNOT_BE_EMPTY);
        return locations.stream().map(LatLong::latitude).reduce(-1.7976931348623157E308, Math::max);
    }

    public static Double minLatitude(Collection<LatLong> locations) {
        Preconditions.checkNotNull(locations, (Object)COLLECTION_CANNOT_BE_NULL);
        Preconditions.checkArgument((!locations.isEmpty() ? 1 : 0) != 0, (Object)COLLECTION_CANNOT_BE_EMPTY);
        return locations.stream().map(LatLong::latitude).reduce((Double)Double.MAX_VALUE, Math::min);
    }

    public static Double maxLongitude(Collection<LatLong> locations) {
        Preconditions.checkNotNull(locations, (Object)COLLECTION_CANNOT_BE_NULL);
        Preconditions.checkArgument((!locations.isEmpty() ? 1 : 0) != 0, (Object)COLLECTION_CANNOT_BE_EMPTY);
        return locations.stream().map(LatLong::longitude).reduce(-1.7976931348623157E308, Math::max);
    }

    public static Double minLongitude(Collection<LatLong> locations) {
        Preconditions.checkNotNull(locations, (Object)COLLECTION_CANNOT_BE_NULL);
        Preconditions.checkArgument((!locations.isEmpty() ? 1 : 0) != 0, (Object)COLLECTION_CANNOT_BE_EMPTY);
        return locations.stream().map(LatLong::longitude).reduce((Double)Double.MAX_VALUE, Math::min);
    }

    @Override
    public int compareTo(LatLong other) {
        return ComparisonChain.start().compare(this.latitude, other.latitude).compare(this.longitude, other.longitude).result();
    }

    private static class SphericalUnitVector {
        final double x;
        final double y;
        final double z;

        private SphericalUnitVector(LatLong location) {
            double latInRadian = Math.toRadians(location.latitude);
            double longInRadian = Math.toRadians(location.longitude);
            this.x = FastMath.cos((double)latInRadian) * FastMath.cos((double)longInRadian);
            this.y = FastMath.cos((double)latInRadian) * FastMath.sin((double)longInRadian);
            this.z = FastMath.sin((double)latInRadian);
        }
    }
}

