/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons.out;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Multiset;
import com.google.common.collect.TreeMultiset;
import com.google.common.io.Files;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.mitre.caasd.commons.Time;
import org.mitre.caasd.commons.fileutil.FileUtils;
import org.mitre.caasd.commons.util.DemotedException;

public class GzFileSink<T>
implements Consumer<T>,
Closeable {
    static final String IN_PROGRESS_PREFIX = "UNDER_CONSTRUCTION_";
    private final String outputDir;
    private final Function<T, String> fileNamer;
    private final Function<T, String> toString;
    private final Map<String, TrackedPrintWriter> openWriters;
    private final Multiset<String> targetCounts;
    private final Duration expirationTime;
    private final int maxOpenWriters;
    private final BlockingQueue<T> queue;
    private final ScheduledExecutorService executor;
    private boolean isClosed = false;

    public GzFileSink(String outputDir, Function<T, String> toString, Function<T, String> fileNamer, Duration expirationTime) {
        this.outputDir = Objects.requireNonNull(outputDir);
        this.toString = Objects.requireNonNull(toString);
        this.fileNamer = Objects.requireNonNull(fileNamer);
        this.expirationTime = Objects.requireNonNull(expirationTime);
        this.maxOpenWriters = 100;
        this.openWriters = new TreeMap<String, TrackedPrintWriter>();
        this.targetCounts = TreeMultiset.create();
        this.queue = new ArrayBlockingQueue<T>(5000);
        this.executor = this.buildExecutor();
        this.scheduleStreamCloser();
        this.scheduleDataWriting();
    }

    private void scheduleStreamCloser() {
        this.executor.scheduleWithFixedDelay(this::closeStaleStreamTargets, 0L, 1L, TimeUnit.SECONDS);
    }

    private void scheduleDataWriting() {
        this.executor.scheduleWithFixedDelay(this::drainQueueAndWriteRecords, 0L, 1L, TimeUnit.SECONDS);
    }

    private void closeStaleStreamTargets() {
        this.staleOutputTargets().stream().forEach(this::closeAndRemoveWriter);
    }

    private List<String> staleOutputTargets() {
        List<String> staleStreamTargets = this.openWriters.entrySet().stream().filter(entry -> ((TrackedPrintWriter)entry.getValue()).isStale(this.expirationTime)).map(Map.Entry::getKey).collect(Collectors.toList());
        return staleStreamTargets;
    }

    @Override
    public void accept(T inputRecord) {
        Preconditions.checkState((!this.isClosed ? 1 : 0) != 0, (Object)"Cannot add data to a closed GzFileSink");
        this.putInBuffer(inputRecord);
    }

    public int numOpenWriters() {
        return this.openWriters.size();
    }

    private void putInBuffer(T inputRecord) {
        try {
            this.queue.put(inputRecord);
        }
        catch (InterruptedException ex) {
            throw DemotedException.demote("Thread interrupted while waiting to add item to maxxed out writeQueue", ex);
        }
    }

    public synchronized void drainQueueAndWriteRecords() {
        ArrayList recordsToPublish = Lists.newArrayList();
        this.queue.drainTo(recordsToPublish);
        for (Object t : recordsToPublish) {
            this.sendToGzFile(t);
        }
    }

    private void sendToGzFile(T inputRecord) {
        String targetFilename = this.fileNamer.apply(inputRecord);
        TrackedPrintWriter targetStream = this.openWriters.containsKey(targetFilename) ? this.openWriters.get(targetFilename) : this.newGzStreamFor(targetFilename);
        targetStream.write(this.toString.apply(inputRecord) + "\n");
        targetStream.flush();
    }

    private TrackedPrintWriter newGzStreamFor(String filename) {
        Preconditions.checkState((this.openWriters.size() <= this.maxOpenWriters ? 1 : 0) != 0, (Object)("Cannot open new gz file because " + this.openWriters.size() + " streams are already open.  Could calling flushAndCloseCurrentFiles() help?"));
        FileUtils.makeDirIfMissing(this.outputDir);
        try {
            int count = this.targetCounts.count((Object)filename);
            String actualFilename = count > 0 ? IN_PROGRESS_PREFIX + filename + "_" + count + ".gz" : IN_PROGRESS_PREFIX + filename + ".gz";
            File target = new File(this.outputDir + File.separator + actualFilename);
            TrackedPrintWriter tpw = new TrackedPrintWriter(target);
            this.openWriters.put(filename, tpw);
            this.targetCounts.add((Object)filename);
            return tpw;
        }
        catch (IOException ioe) {
            throw DemotedException.demote(ioe);
        }
    }

    @Override
    public void close() throws IOException {
        this.isClosed = true;
        this.executor.shutdownNow();
        this.drainQueueAndWriteRecords();
        this.closeAllWriters();
    }

    private void closeAllWriters() {
        ArrayList closeKeys = Lists.newArrayList(this.openWriters.keySet());
        for (String targetToClose : closeKeys) {
            this.closeAndRemoveWriter(targetToClose);
        }
    }

    public void flushAndCloseCurrentFiles() {
        this.drainQueueAndWriteRecords();
        this.closeAllWriters();
    }

    private void closeAndRemoveWriter(String targetFile) {
        try {
            TrackedPrintWriter closeMe = this.openWriters.remove(targetFile);
            closeMe.close();
            File hasInProgressName = closeMe.targetFile();
            File afterRename = new File(this.outputDir + File.separator + hasInProgressName.getName().substring(IN_PROGRESS_PREFIX.length()));
            Files.move((File)hasInProgressName, (File)afterRename);
        }
        catch (IOException ioe) {
            throw DemotedException.demote(ioe);
        }
    }

    private ScheduledExecutorService buildExecutor() {
        ScheduledExecutorService ses = Executors.newScheduledThreadPool(1);
        Runtime.getRuntime().addShutdownHook(new DrainFlushAndShutdown(this, ses));
        return ses;
    }

    private static class TrackedPrintWriter
    implements AutoCloseable {
        private final File targetFile;
        private final PrintWriter writer;
        private Instant timeOfLastWrite;

        TrackedPrintWriter(File targetFile) throws IOException {
            this.targetFile = Objects.requireNonNull(targetFile);
            this.writer = FileUtils.buildGzWriter(targetFile);
            this.timeOfLastWrite = Instant.now();
        }

        @Override
        public void close() throws IOException {
            this.writer.close();
        }

        void write(String str) {
            this.timeOfLastWrite = Instant.now();
            this.writer.write(str);
        }

        void flush() {
            this.writer.flush();
        }

        Duration timeSinceLastWrite() {
            return Time.durationBtw(Instant.now(), this.timeOfLastWrite);
        }

        boolean isStale(Duration timeLimit) {
            return Time.theDuration(this.timeSinceLastWrite()).isGreaterThan(timeLimit);
        }

        File targetFile() {
            return this.targetFile;
        }
    }

    private static class DrainFlushAndShutdown
    extends Thread {
        final GzFileSink archiver;
        final ExecutorService exec;

        public DrainFlushAndShutdown(GzFileSink archiver, ExecutorService exec) {
            this.archiver = Objects.requireNonNull(archiver);
            this.exec = Objects.requireNonNull(exec);
        }

        @Override
        public void run() {
            this.exec.shutdownNow();
            this.archiver.drainQueueAndWriteRecords();
            try {
                this.archiver.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }
}

