/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons;

import com.google.common.base.Preconditions;
import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import org.mitre.caasd.commons.Distance;
import org.mitre.caasd.commons.LatLong;

public class Speed
implements Serializable,
Comparable<Speed> {
    public static final Speed ZERO = Speed.of(0.0, Unit.KNOTS);
    public static final Speed ZERO_FEET_PER_MIN = Speed.of(0.0, Unit.FEET_PER_MINUTE);
    private static final double SEC_PER_HOUR = 3600.0;
    private final Distance.Unit distanceUnit;
    private final double amountPerSecond;

    private Speed() {
        this(Distance.Unit.NAUTICAL_MILES, 0.0);
    }

    public Speed(Distance dist, Duration timeDelta) {
        Preconditions.checkNotNull((Object)dist);
        Preconditions.checkNotNull((Object)timeDelta);
        Preconditions.checkArgument((!timeDelta.isNegative() ? 1 : 0) != 0);
        Preconditions.checkArgument((!timeDelta.isZero() ? 1 : 0) != 0);
        this.distanceUnit = dist.nativeUnit();
        double msElapsed = timeDelta.toMillis();
        this.amountPerSecond = dist.in(this.distanceUnit) / (msElapsed / 1000.0);
    }

    private Speed(Distance dist, double msElapsed) {
        this(dist.nativeUnit(), dist.in(dist.nativeUnit()) / (msElapsed / 1000.0));
    }

    private Speed(Distance.Unit distUnit, double amountPerSecond) {
        this.distanceUnit = distUnit;
        this.amountPerSecond = amountPerSecond;
    }

    @Override
    public int compareTo(Speed speed) {
        return Double.compare(this.inMetersPerSecond(), speed.inMetersPerSecond());
    }

    public static Speed ofMilesPerHour(double amount) {
        return Speed.of(amount, Unit.MILES_PER_HOUR);
    }

    public static Speed ofKilometersPerHour(double amount) {
        return Speed.of(amount, Unit.KILOMETERS_PER_HOUR);
    }

    public static Speed ofMetersPerSecond(double amount) {
        return Speed.of(amount, Unit.METERS_PER_SECOND);
    }

    public static Speed ofFeetPerSecond(double amount) {
        return Speed.of(amount, Unit.FEET_PER_SECOND);
    }

    public static Speed ofFeetPerMinute(double amount) {
        return Speed.of(amount, Unit.FEET_PER_MINUTE);
    }

    public static Speed ofKnots(double amount) {
        return Speed.of(amount, Unit.KNOTS);
    }

    public static Speed of(double amount, Unit unit) {
        return new Speed(Distance.of(amount, unit.distUnit), unit.secondsPerTimeUnit * 1000.0);
    }

    public static Speed between(LatLong pos1, Instant time1, LatLong pos2, Instant time2) {
        Distance distance = pos1.distanceTo(pos2);
        Duration timeDelta = Duration.between(time1, time2).abs();
        return new Speed(distance, timeDelta);
    }

    public double inKnots() {
        return this.in(Unit.KNOTS);
    }

    public double inMetersPerSecond() {
        return this.in(Unit.METERS_PER_SECOND);
    }

    public double inFeetPerSecond() {
        return this.in(Unit.FEET_PER_SECOND);
    }

    public double inFeetPerMinutes() {
        return this.in(Unit.FEET_PER_MINUTE);
    }

    public double inKilometersPerHour() {
        return this.in(Unit.KILOMETERS_PER_HOUR);
    }

    public double inMilesPerHour() {
        return this.in(Unit.MILES_PER_HOUR);
    }

    public boolean isLessThan(Speed otherSpd) {
        return this.inMetersPerSecond() < otherSpd.inMetersPerSecond();
    }

    public boolean isLessThanOrEqualTo(Speed otherSpd) {
        return this.inMetersPerSecond() <= otherSpd.inMetersPerSecond();
    }

    public boolean isGreaterThan(Speed otherSpd) {
        return this.inMetersPerSecond() > otherSpd.inMetersPerSecond();
    }

    public boolean isGreaterThanOrEqualTo(Speed otherSpd) {
        return this.inMetersPerSecond() >= otherSpd.inMetersPerSecond();
    }

    public double in(Unit unit) {
        double conversionFactor = unit.distUnit.unitsPerMeter() / this.distanceUnit.unitsPerMeter();
        return this.amountPerSecond * unit.secondsPerTimeUnit * conversionFactor;
    }

    public String toString() {
        return this.toString(3);
    }

    public String toString(int digitsAfterDecimalPlace) {
        return this.toString(digitsAfterDecimalPlace, Speed.speedUnitFor(this.distanceUnit));
    }

    public String toString(int digitsAfterDecimalPlace, Unit unit) {
        return String.format("%." + digitsAfterDecimalPlace + "f" + unit.abbreviation, this.in(unit));
    }

    public static Unit speedUnitFor(Distance.Unit distanceUnit) {
        switch (distanceUnit) {
            case NAUTICAL_MILES: {
                return Unit.KNOTS;
            }
            case FEET: {
                return Unit.FEET_PER_MINUTE;
            }
            case KILOMETERS: {
                return Unit.KILOMETERS_PER_HOUR;
            }
            case METERS: {
                return Unit.METERS_PER_SECOND;
            }
            case MILES: {
                return Unit.MILES_PER_HOUR;
            }
        }
        throw new IllegalArgumentException("unknown unit: " + String.valueOf((Object)distanceUnit));
    }

    public static Speed fromString(String parseMe) {
        Preconditions.checkNotNull((Object)parseMe);
        parseMe = parseMe.trim();
        Unit speedUnit = Speed.unitFromString(parseMe);
        if (speedUnit == null) {
            throw new IllegalArgumentException("Could not parse Speed Unit from: " + parseMe);
        }
        String parseMeWithoutUnitsSuffix = parseMe.substring(0, parseMe.length() - speedUnit.abbreviation.length());
        double amount = Double.parseDouble(parseMeWithoutUnitsSuffix);
        return Speed.of(amount, speedUnit);
    }

    public static Unit unitFromString(String parseMe) {
        for (Unit unit : Unit.values()) {
            if (!parseMe.endsWith(unit.abbreviation)) continue;
            return unit;
        }
        return null;
    }

    public Distance times(Duration amountOfTime) {
        Preconditions.checkArgument((!amountOfTime.isNegative() ? 1 : 0) != 0);
        double amount = (double)amountOfTime.toMillis() / 1000.0 * this.amountPerSecond;
        return Distance.of(amount, this.distanceUnit);
    }

    public Speed times(double scalar) {
        return new Speed(this.distanceUnit, this.amountPerSecond * scalar);
    }

    public Speed plus(Speed otherSpeed) {
        switch (this.distanceUnit) {
            case NAUTICAL_MILES: {
                return Speed.of(this.inKnots() + otherSpeed.inKnots(), Unit.KNOTS);
            }
            case METERS: {
                return Speed.of(this.inMetersPerSecond() + otherSpeed.inMetersPerSecond(), Unit.METERS_PER_SECOND);
            }
            case FEET: {
                return Speed.of(this.inFeetPerSecond() + otherSpeed.inFeetPerSecond(), Unit.FEET_PER_SECOND);
            }
            case MILES: {
                return Speed.of(this.inMilesPerHour() + otherSpeed.inMilesPerHour(), Unit.MILES_PER_HOUR);
            }
            case KILOMETERS: {
                return Speed.of(this.inKilometersPerHour() + otherSpeed.inKilometersPerHour(), Unit.KILOMETERS_PER_HOUR);
            }
        }
        throw new IllegalStateException("Unkown distanceUnit: " + String.valueOf((Object)this.distanceUnit));
    }

    public Speed minus(Speed otherSpeed) {
        return this.plus(otherSpeed.times(-1.0));
    }

    public Speed abs() {
        return this.isNegative() ? this.times(-1.0) : this;
    }

    public Duration timeToTravel(Distance dist) {
        Preconditions.checkNotNull((Object)dist);
        double amountInDesiredUnit = dist.in(this.distanceUnit);
        double timeInSec = amountInDesiredUnit / this.amountPerSecond;
        double timeInMs = (long)(timeInSec * 1000.0);
        return Duration.ofMillis((long)timeInMs);
    }

    public int hashCode() {
        int hash = 7;
        hash = 67 * hash + Objects.hashCode((Object)this.distanceUnit);
        hash = 67 * hash + (int)(Double.doubleToLongBits(this.amountPerSecond) ^ Double.doubleToLongBits(this.amountPerSecond) >>> 32);
        return hash;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        Speed other = (Speed)obj;
        if (Double.doubleToLongBits(this.amountPerSecond) != Double.doubleToLongBits(other.amountPerSecond)) {
            return false;
        }
        return this.distanceUnit == other.distanceUnit;
    }

    public boolean isPositive() {
        return this.amountPerSecond > 0.0;
    }

    public boolean isNegative() {
        return this.amountPerSecond < 0.0;
    }

    public boolean isZero() {
        return this.amountPerSecond == 0.0;
    }

    public static enum Unit {
        KNOTS(Distance.Unit.NAUTICAL_MILES, 3600.0, "kn"),
        METERS_PER_SECOND(Distance.Unit.METERS, 1.0, "mps"),
        FEET_PER_SECOND(Distance.Unit.FEET, 1.0, "fps"),
        FEET_PER_MINUTE(Distance.Unit.FEET, 60.0, "fpm"),
        MILES_PER_HOUR(Distance.Unit.MILES, 3600.0, "mph"),
        KILOMETERS_PER_HOUR(Distance.Unit.KILOMETERS, 3600.0, "kph");

        private final Distance.Unit distUnit;
        final double secondsPerTimeUnit;
        final String abbreviation;

        private Unit(Distance.Unit distUnit, double secPerTimeUnit, String suffix) {
            this.distUnit = distUnit;
            this.secondsPerTimeUnit = secPerTimeUnit;
            this.abbreviation = suffix;
        }

        public Distance.Unit distanceUnit() {
            return this.distUnit;
        }

        public double secondsPerTimeUnit() {
            return this.secondsPerTimeUnit;
        }

        public String abbreviation() {
            return this.abbreviation;
        }
    }
}

