/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons.maps;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.net.URL;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import javax.imageio.ImageIO;
import org.mitre.caasd.commons.fileutil.FileUtils;
import org.mitre.caasd.commons.maps.TileAddress;
import org.mitre.caasd.commons.maps.TileServer;
import org.mitre.caasd.commons.util.DemotedException;

public class LocallyCachingTileServer
implements TileServer {
    private static final int CACHE_SIZE = 64;
    private static final Duration DEFAULT_CACHE_LIFETIME = Duration.ofDays(7L);
    private static final String DEFAULT_CACHE_DIR = "mapTileCache";
    private final TileServer inner;
    private final LoadingCache<TileAddress, BufferedImage> cache;

    public LocallyCachingTileServer(TileServer server, Duration maxAge, String cacheDirectory) {
        Objects.requireNonNull(server);
        Objects.requireNonNull(cacheDirectory);
        this.inner = server;
        CacheLoader<TileAddress, BufferedImage> onCacheMiss = LocallyCachingTileServer.makeCacheLoader(this.inner, cacheDirectory, maxAge);
        this.cache = CacheBuilder.newBuilder().maximumSize(64L).recordStats().build(onCacheMiss);
    }

    public LocallyCachingTileServer(TileServer server, Duration maxAg) {
        this(server, maxAg, DEFAULT_CACHE_DIR);
    }

    public LocallyCachingTileServer(TileServer server) {
        this(server, DEFAULT_CACHE_LIFETIME, DEFAULT_CACHE_DIR);
    }

    @Override
    public int maxZoomLevel() {
        return this.inner.maxZoomLevel();
    }

    @Override
    public int maxTileSize() {
        return this.inner.maxTileSize();
    }

    @Override
    public URL getUrlFor(TileAddress ta) {
        return this.inner.getUrlFor(ta);
    }

    @Override
    public BufferedImage downloadMap(TileAddress ta) {
        try {
            return (BufferedImage)this.cache.get((Object)ta);
        }
        catch (ExecutionException ex) {
            throw DemotedException.demote(ex);
        }
    }

    public Cache<TileAddress, BufferedImage> cache() {
        return this.cache;
    }

    public static CacheLoader<TileAddress, BufferedImage> makeCacheLoader(final TileServer fallbackTileServer, final String cacheDir, final Duration maxAge) {
        return new CacheLoader<TileAddress, BufferedImage>(){

            public BufferedImage load(TileAddress ta) throws Exception {
                FileUtils.makeDirIfMissing(cacheDir);
                File cachedImageFile = new File(cacheDir + File.separator + String.valueOf(ta) + ".png");
                if (cachedImageFile.exists() && LocallyCachingTileServer.fileIsRecent(cachedImageFile, maxAge)) {
                    return ImageIO.read(cachedImageFile);
                }
                BufferedImage img = fallbackTileServer.downloadMap(ta);
                ImageIO.write((RenderedImage)img, "png", cachedImageFile);
                return img;
            }
        };
    }

    private static boolean fileIsRecent(File file, Duration maxAge) {
        Instant lastModified = Instant.ofEpochMilli(file.lastModified());
        Duration fileAge = Duration.between(lastModified, Instant.now());
        return fileAge.toMillis() < maxAge.toMillis();
    }
}

