/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons.maps;

import com.google.common.base.Preconditions;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.imageio.ImageIO;
import org.mitre.caasd.commons.Course;
import org.mitre.caasd.commons.Distance;
import org.mitre.caasd.commons.LatLong;
import org.mitre.caasd.commons.Spherical;
import org.mitre.caasd.commons.maps.FeatureSet;
import org.mitre.caasd.commons.maps.MapBuilder;
import org.mitre.caasd.commons.maps.MapFeature;
import org.mitre.caasd.commons.maps.MapFeatures;
import org.mitre.caasd.commons.maps.PixelLatLong;
import org.mitre.caasd.commons.maps.TileAddress;
import org.mitre.caasd.commons.maps.TileServer;
import org.mitre.caasd.commons.util.DemotedException;

public class MapImage {
    final LatLong center;
    final LatLong topLeft;
    final LatLong bottomRight;
    final PixelLatLong zeroPixel;
    final int zoomLevel;
    final int tileSize;
    final BufferedImage undecoratedImage;

    public MapImage(TileServer tileServer, LatLong center, Distance width) {
        this.center = Objects.requireNonNull(center);
        Distance halfWidth = width.times(0.5);
        this.topLeft = center.project(Course.WEST, halfWidth).project(Course.NORTH, halfWidth);
        this.bottomRight = center.project(Course.EAST, halfWidth).project(Course.SOUTH, halfWidth);
        this.tileSize = tileServer.maxTileSize();
        this.zoomLevel = MapImage.computeZoomLevel(width);
        this.undecoratedImage = MapImage.buildCroppedImage(tileServer, this.topLeft, this.bottomRight, this.zoomLevel);
        this.zeroPixel = new PixelLatLong(this.topLeft, this.zoomLevel, this.tileSize);
    }

    public MapImage(TileServer tileServer, LatLong center, int widthInPixels, int zoomLvl) {
        this.center = Objects.requireNonNull(center);
        this.zoomLevel = zoomLvl;
        this.tileSize = tileServer.maxTileSize();
        int half = widthInPixels / 2;
        PixelLatLong cp = new PixelLatLong(center, zoomLvl, this.tileSize);
        this.topLeft = new PixelLatLong(cp.x() - (double)half, cp.y() - (double)half, zoomLvl, this.tileSize).latLong();
        this.bottomRight = new PixelLatLong(cp.x() + (double)half, cp.y() + (double)half, zoomLvl, this.tileSize).latLong();
        this.undecoratedImage = MapImage.buildCroppedImage(tileServer, this.topLeft, this.bottomRight, zoomLvl);
        this.zeroPixel = new PixelLatLong(this.topLeft, zoomLvl, this.tileSize);
    }

    private static BufferedImage buildCroppedImage(TileServer tileServer, LatLong topLeft, LatLong bottomRight, int zoomLevel) {
        int tileSize = tileServer.maxTileSize();
        List<TileAddress> tilesInMap = TileAddress.tileAddressesSpanning(topLeft, bottomRight, zoomLevel);
        BufferedImage combinedTileImage = tileServer.downloadAndCombineTiles(tilesInMap);
        PixelLatLong topLeftOfCombined = TileAddress.of(topLeft, zoomLevel).topLeftPixel(tileSize);
        PixelLatLong excerptTopLeft = new PixelLatLong(topLeft, zoomLevel, tileSize);
        PixelLatLong excerptBottomRight = new PixelLatLong(bottomRight, zoomLevel, tileSize);
        BufferedImage undecoratedImage = combinedTileImage.getSubimage(excerptTopLeft.x(topLeftOfCombined), excerptTopLeft.y(topLeftOfCombined), (int)(excerptBottomRight.x() - excerptTopLeft.x()), (int)(excerptBottomRight.y() - excerptTopLeft.y()));
        return undecoratedImage;
    }

    private static int computeZoomLevel(Distance width) {
        Distance currentDistance = Distance.ofNauticalMiles(Math.PI * 2 * Spherical.EARTH_RADIUS_NM);
        int zoomLevel = 0;
        while (width.isLessThan(currentDistance)) {
            ++zoomLevel;
            currentDistance = currentDistance.times(0.5);
        }
        return zoomLevel;
    }

    public BufferedImage plot() {
        return this.copyOfImage();
    }

    public BufferedImage plot(Iterable<MapFeature> features) {
        BufferedImage copy = this.copyOfImage();
        Graphics2D g = (Graphics2D)copy.getGraphics();
        for (MapFeature f : features) {
            f.drawOn(g, this.zeroPixel);
        }
        return copy;
    }

    public BufferedImage plot(FeatureSet ... sets) {
        BufferedImage copy = this.copyOfImage();
        Graphics2D g = (Graphics2D)copy.getGraphics();
        for (FeatureSet set : sets) {
            for (MapFeature mp : set) {
                mp.drawOn(g, this.zeroPixel);
            }
        }
        return copy;
    }

    public void plotToFile(File targetFile) {
        this.plotToFile(FeatureSet.noMapFeatures(), targetFile);
    }

    public void plotToFile(Iterable<MapFeature> features, File targetFile) {
        Objects.requireNonNull(features);
        Objects.requireNonNull(targetFile);
        MapImage.writeImageToFile(this.plot(features), targetFile);
    }

    public void plotToFile(File targetFile, FeatureSet ... sets) {
        Objects.requireNonNull(sets);
        Objects.requireNonNull(targetFile);
        MapImage.writeImageToFile(this.plot(sets), targetFile);
    }

    public static void writeImageToFile(BufferedImage img, File targetFile) {
        Objects.requireNonNull(img);
        Objects.requireNonNull(targetFile);
        String filename = targetFile.getName();
        Preconditions.checkArgument((filename.endsWith(".jpg") || filename.endsWith(".png") ? 1 : 0) != 0, (Object)"Must write to a \".jpg\" file or a \".png\" file");
        String imageFormat = filename.substring(filename.lastIndexOf(".") + 1);
        try {
            ImageIO.write((RenderedImage)img, imageFormat, targetFile);
        }
        catch (IOException ioe) {
            throw DemotedException.demote(ioe);
        }
    }

    public static BufferedImage plotLocationData(List<LatLong> locations, Distance mapWidth) {
        LatLong center = LatLong.avgLatLong(locations.toArray(new LatLong[0]));
        List<MapFeature> pointData = locations.stream().map(loc -> MapFeatures.circle(loc, Color.GREEN, 6, 1.0f)).collect(Collectors.toList());
        return MapBuilder.newMapBuilder().mapBoxDarkMode().center(center).width(mapWidth).addFeatures(pointData).toImage();
    }

    private BufferedImage copyOfImage() {
        BufferedImage copy = new BufferedImage(this.undecoratedImage.getWidth(), this.undecoratedImage.getHeight(), 5);
        Graphics2D g = (Graphics2D)copy.getGraphics();
        g.drawImage(this.undecoratedImage, null, null);
        return copy;
    }
}

