/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.caasd.commons.math.locationfit;

import com.google.common.base.Preconditions;
import java.time.Duration;
import java.time.Instant;
import java.util.DoubleSummaryStatistics;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.commons.math3.analysis.polynomials.PolynomialFunction;
import org.mitre.caasd.commons.Acceleration;
import org.mitre.caasd.commons.Course;
import org.mitre.caasd.commons.LatLong;
import org.mitre.caasd.commons.Speed;
import org.mitre.caasd.commons.Spherical;
import org.mitre.caasd.commons.math.CurveFitters;

class LatLongFitter {
    private final int POLYNOMIAL_DEGREE = 2;
    private final PolynomialFunction latFunc;
    private final PolynomialFunction lonFunc;
    private boolean crossesDateLine;

    LatLongFitter(List<Double> weights, List<Double> timesAsEpochMs, List<LatLong> locations) {
        Objects.requireNonNull(weights);
        Objects.requireNonNull(timesAsEpochMs);
        Objects.requireNonNull(locations);
        Preconditions.checkArgument((timesAsEpochMs.size() == locations.size() ? 1 : 0) != 0, (Object)"list sizes must be the same");
        List<Double> lats = locations.stream().map(loc -> loc.latitude()).collect(Collectors.toList());
        this.latFunc = CurveFitters.weightedFit(2, weights, timesAsEpochMs, lats);
        List<Double> rawLongs = locations.stream().map(loc -> loc.longitude()).collect(Collectors.toList());
        this.crossesDateLine = LatLongFitter.crossesDateLine(rawLongs);
        List<Double> fittableLongs = this.crossesDateLine ? LatLongFitter.moddedLongs(rawLongs) : rawLongs;
        this.lonFunc = CurveFitters.weightedFit(2, weights, timesAsEpochMs, fittableLongs);
    }

    static boolean crossesDateLine(List<Double> longs) {
        DoubleSummaryStatistics stats = longs.stream().mapToDouble(x -> x).summaryStatistics();
        return stats.getMax() - stats.getMin() > 350.0;
    }

    static List<Double> moddedLongs(List<Double> rawLongitudes) {
        return rawLongitudes.stream().map(lon -> Spherical.mod(lon, 360.0)).collect(Collectors.toList());
    }

    static Double unMod(double coercedLongitude) {
        Preconditions.checkArgument((0.0 <= coercedLongitude && coercedLongitude <= 360.0 ? 1 : 0) != 0);
        return coercedLongitude > 180.0 ? coercedLongitude - 360.0 : coercedLongitude;
    }

    private LatLong fitLocationAt(Double timeInEpochMs) {
        double latitude = this.latFunc.value(timeInEpochMs.doubleValue());
        double longitude = this.lonFunc.value(timeInEpochMs.doubleValue());
        longitude = this.crossesDateLine ? LatLongFitter.unMod(longitude) : LatLong.clampLongitude(longitude);
        return LatLong.of(latitude, longitude);
    }

    KineticValues fitKineticsAtTimeZero() {
        LatLong location = this.fitLocationAt(0.0);
        long TIME_STEP_IN_MS = 1000L;
        LatLong priorLocation = this.fitLocationAt(-1000.0);
        LatLong futureLocation = this.fitLocationAt(1000.0);
        Instant priorTime = Instant.EPOCH.minusMillis(1000L);
        Instant futureTime = Instant.EPOCH.plusMillis(1000L);
        Speed speed = Speed.between(priorLocation, priorTime, futureLocation, futureTime);
        Course course = Spherical.courseBtw(priorLocation, futureLocation);
        Speed priorSpeed = Speed.between(priorLocation, priorTime, location, Instant.EPOCH);
        Speed futureSpeed = Speed.between(location, Instant.EPOCH, futureLocation, futureTime);
        Speed speedDelta = futureSpeed.minus(priorSpeed);
        Acceleration acceleration = Acceleration.of(speedDelta, Duration.ofMillis(1000L));
        return new KineticValues(location, speed, course, acceleration);
    }

    double deduceTurnRate() {
        double TIME_STEP_IN_MS = 500.0;
        LatLong stepBack = this.fitLocationAt(-500.0);
        LatLong current = this.fitLocationAt(0.0);
        LatLong stepForward = this.fitLocationAt(500.0);
        Course priorToCurrent = Spherical.courseBtw(stepBack, current);
        Course currentToNext = Spherical.courseBtw(current, stepForward);
        Course delta = Course.angleBetween(currentToNext, priorToCurrent);
        return delta.inDegrees();
    }

    static class KineticValues {
        private final LatLong location;
        private final Speed speed;
        private final Course course;
        private final Acceleration acceleration;

        public KineticValues(LatLong location, Speed speed, Course course, Acceleration acceleration) {
            Objects.requireNonNull(location);
            Objects.requireNonNull(speed);
            Objects.requireNonNull(course);
            Objects.requireNonNull(acceleration);
            this.location = location;
            this.speed = speed;
            this.course = course;
            this.acceleration = acceleration;
        }

        public LatLong latLong() {
            return this.location;
        }

        public Speed speed() {
            return this.speed;
        }

        public Course course() {
            return this.course;
        }

        public Acceleration acceleration() {
            return this.acceleration;
        }
    }
}

