/*
 * Decompiled with CFR 0.152.
 */
package org.mitre.jose.jwk;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.nimbusds.jose.Algorithm;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.Curve;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.KeyType;
import com.nimbusds.jose.jwk.KeyUse;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.math.BigInteger;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.text.ParseException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.bouncycastle.util.io.pem.PemWriter;
import org.mitre.jose.jwk.ECKeyMaker;
import org.mitre.jose.jwk.KeyIdGenerator;
import org.mitre.jose.jwk.OKPKeyMaker;
import org.mitre.jose.jwk.OctetSequenceKeyMaker;
import org.mitre.jose.jwk.RSAKeyMaker;

public class Launcher {
    private static Options options;
    private static List<Curve> ecCurves;
    private static List<Curve> okpCurves;
    private static List<KeyType> keyTypes;

    public static void main(String[] args) {
        Security.addProvider((Provider)new BouncyCastleProvider());
        options = new Options();
        Launcher.configureCommandLineOptions(options);
        DefaultParser parser = new DefaultParser();
        try {
            CommandLine cmd = parser.parse(options, args);
            String kty = cmd.getOptionValue("t");
            String size = cmd.getOptionValue("s");
            String use = cmd.getOptionValue("u");
            String alg = cmd.getOptionValue("a");
            String crv = cmd.getOptionValue("c");
            boolean keySet = cmd.hasOption("S");
            boolean pubKey = cmd.hasOption("p");
            String outFile = cmd.getOptionValue("o");
            String pubOutFile = cmd.getOptionValue("P");
            boolean printX509 = cmd.hasOption("x");
            String kid = cmd.getOptionValue("i");
            KeyIdGenerator generator = Strings.isNullOrEmpty((String)kid) ? (cmd.hasOption("i") || cmd.hasOption("I") ? KeyIdGenerator.NONE : KeyIdGenerator.get(cmd.getOptionValue("g"))) : KeyIdGenerator.specified(kid);
            if (kty == null) {
                throw Launcher.printUsageAndExit("Key type must be supplied.");
            }
            KeyType keyType = KeyType.parse((String)kty);
            KeyUse keyUse = Launcher.validateKeyUse(use);
            JWSAlgorithm keyAlg = null;
            if (!Strings.isNullOrEmpty((String)alg)) {
                keyAlg = JWSAlgorithm.parse((String)alg);
            }
            JWK jwk = Launcher.makeKey(size, generator, crv, keyType, keyUse, (Algorithm)keyAlg);
            Launcher.outputKey(keySet, pubKey, outFile, pubOutFile, printX509, jwk);
        }
        catch (NumberFormatException e) {
            throw Launcher.printUsageAndExit("Invalid key size: " + e.getMessage());
        }
        catch (org.apache.commons.cli.ParseException e) {
            throw Launcher.printUsageAndExit("Failed to parse arguments: " + e.getMessage());
        }
        catch (ParseException e) {
            throw Launcher.printUsageAndExit("Could not parse existing KeySet: " + e.getMessage());
        }
        catch (IOException e) {
            throw Launcher.printUsageAndExit("Could not read existing KeySet: " + e.getMessage());
        }
    }

    private static void configureCommandLineOptions(Options options) {
        options.addOption("t", "type", true, "Key Type, one of: " + keyTypes.stream().map(KeyType::getValue).collect(Collectors.joining(", ")));
        options.addOption("s", "size", true, "Key Size in bits, required for " + KeyType.RSA.getValue() + " and " + KeyType.OCT.getValue() + " key types. Must be an integer divisible by 8");
        options.addOption("c", "curve", true, "Key Curve, required for " + KeyType.EC.getValue() + " or " + KeyType.OKP.getValue() + " key type. Must be one of " + ecCurves.stream().map(Curve::getName).collect(Collectors.joining(", ")) + " for EC keys or one of " + okpCurves.stream().map(Curve::getName).collect(Collectors.joining(", ")) + " for OKP keys.");
        options.addOption("u", "usage", true, "Usage, one of: enc, sig (optional)");
        options.addOption("a", "algorithm", true, "Algorithm (optional)");
        OptionGroup idGroup = new OptionGroup();
        idGroup.addOption(new Option("i", "id", true, "Key ID (optional), one will be generated if not defined"));
        idGroup.addOption(new Option("I", "noGenerateId", false, "<deprecated> Don't generate a Key ID. (Deprecated, use '-g none' instead.)"));
        idGroup.addOption(new Option("g", "idGenerator", true, "Key ID generation method (optional). Can be one of: " + KeyIdGenerator.values().stream().map(KeyIdGenerator::getName).collect(Collectors.joining(", ")) + ". If omitted, generator method defaults to '" + KeyIdGenerator.TIMESTAMP.getName() + "'."));
        options.addOptionGroup(idGroup);
        options.addOption("p", "showPubKey", false, "Display public key separately (if applicable)");
        options.addOption("S", "keySet", false, "Wrap the generated key in a KeySet");
        options.addOption("x", "x509", false, "Display keys in X509 PEM format");
        options.addOption("o", "output", true, "Write output to file. Will append to existing KeySet if -S is used. Key material will not be displayed to console.");
        options.addOption("P", "pubKeyOutput", true, "Write public key to separate file. Will append to existing KeySet if -S is used. Key material will not be displayed to console. '-o/--output' must be declared as well.");
    }

    private static KeyUse validateKeyUse(String use) {
        try {
            return KeyUse.parse((String)use);
        }
        catch (ParseException e) {
            throw Launcher.printUsageAndExit("Invalid key usage, must be 'sig' or 'enc', got " + use);
        }
    }

    private static JWK makeKey(String size, KeyIdGenerator kid, String crv, KeyType keyType, KeyUse keyUse, Algorithm keyAlg) {
        JWK jwk;
        if (keyType.equals((Object)KeyType.RSA)) {
            jwk = Launcher.makeRsaKey(kid, size, keyType, keyUse, keyAlg);
        } else if (keyType.equals((Object)KeyType.OCT)) {
            jwk = Launcher.makeOctKey(kid, size, keyType, keyUse, keyAlg);
        } else if (keyType.equals((Object)KeyType.EC)) {
            jwk = Launcher.makeEcKey(kid, crv, keyType, keyUse, keyAlg);
        } else if (keyType.equals((Object)KeyType.OKP)) {
            jwk = Launcher.makeOkpKey(kid, crv, keyType, keyUse, keyAlg);
        } else {
            throw Launcher.printUsageAndExit("Unknown key type: " + keyType);
        }
        return jwk;
    }

    private static JWK makeOkpKey(KeyIdGenerator kid, String crv, KeyType keyType, KeyUse keyUse, Algorithm keyAlg) {
        if (Strings.isNullOrEmpty((String)crv)) {
            throw Launcher.printUsageAndExit("Curve is required for key type " + keyType);
        }
        Curve keyCurve = Curve.parse((String)crv);
        if (!okpCurves.contains(keyCurve)) {
            throw Launcher.printUsageAndExit("Curve " + crv + " is not valid for key type " + keyType);
        }
        return OKPKeyMaker.make(keyCurve, keyUse, keyAlg, kid);
    }

    private static JWK makeEcKey(KeyIdGenerator kid, String crv, KeyType keyType, KeyUse keyUse, Algorithm keyAlg) {
        if (Strings.isNullOrEmpty((String)crv)) {
            throw Launcher.printUsageAndExit("Curve is required for key type " + keyType);
        }
        Curve keyCurve = Curve.parse((String)crv);
        if (!ecCurves.contains(keyCurve)) {
            throw Launcher.printUsageAndExit("Curve " + crv + " is not valid for key type " + keyType);
        }
        return ECKeyMaker.make(keyCurve, keyUse, keyAlg, kid);
    }

    private static JWK makeOctKey(KeyIdGenerator kid, String size, KeyType keyType, KeyUse keyUse, Algorithm keyAlg) {
        if (Strings.isNullOrEmpty((String)size)) {
            throw Launcher.printUsageAndExit("Key size (in bits) is required for key type " + keyType);
        }
        Integer keySize = Integer.decode(size);
        if (keySize % 8 != 0) {
            throw Launcher.printUsageAndExit("Key size (in bits) must be divisible by 8, got " + keySize);
        }
        return OctetSequenceKeyMaker.make(keySize, keyUse, keyAlg, kid);
    }

    private static JWK makeRsaKey(KeyIdGenerator kid, String size, KeyType keyType, KeyUse keyUse, Algorithm keyAlg) {
        if (Strings.isNullOrEmpty((String)size)) {
            throw Launcher.printUsageAndExit("Key size (in bits) is required for key type " + keyType);
        }
        Integer keySize = Integer.decode(size);
        if (keySize % 8 != 0) {
            throw Launcher.printUsageAndExit("Key size (in bits) must be divisible by 8, got " + keySize);
        }
        return RSAKeyMaker.make(keySize, keyUse, keyAlg, kid);
    }

    private static void outputKey(boolean keySet, boolean pubKey, String outFile, String pubOutFile, boolean printX509, JWK jwk) throws IOException, ParseException {
        Gson gson = new GsonBuilder().setPrettyPrinting().create();
        if (outFile == null) {
            System.out.println("Full key:");
            Launcher.printKey(keySet, jwk, gson);
            if (pubKey) {
                System.out.println();
                JWK pub = jwk.toPublicJWK();
                if (pub != null) {
                    System.out.println("Public key:");
                    Launcher.printKey(keySet, pub, gson);
                } else {
                    System.out.println("No public key.");
                }
            }
            if (printX509) {
                try {
                    KeyType keyType = jwk.getKeyType();
                    if (keyType.equals((Object)KeyType.RSA)) {
                        Certificate cert = Launcher.selfSign(jwk.toRSAKey().toPublicKey(), jwk.toRSAKey().toPrivateKey(), jwk.getKeyID() != null ? jwk.getKeyID() : jwk.computeThumbprint().toString(), "SHA256withRSA");
                        Launcher.writePEMToConsole(jwk.toRSAKey().toPublicKey(), jwk.toRSAKey().toPrivateKey(), cert);
                    }
                    if (keyType.equals((Object)KeyType.EC)) {
                        Certificate cert = Launcher.selfSign(jwk.toECKey().toPublicKey(), jwk.toECKey().toPrivateKey(), jwk.getKeyID() != null ? jwk.getKeyID() : jwk.computeThumbprint().toString(), "SHA256withECDSA");
                        Launcher.writePEMToConsole(jwk.toECKey().toPublicKey(), jwk.toECKey().toPrivateKey(), cert);
                    }
                    throw Launcher.printUsageAndExit("Unknown key type for X509 encoding: " + keyType);
                }
                catch (JOSEException e) {
                    throw Launcher.printUsageAndExit("Error extracting keypair for X509: " + e.getMessage());
                }
            }
        } else {
            Launcher.writeKeyToFile(keySet, outFile, pubOutFile, jwk, gson);
        }
    }

    private static void writePEMToConsole(PublicKey publicKey, PrivateKey privateKey, Certificate cert) {
        try {
            System.out.println();
            System.out.println("X509 Formatted Keys:");
            PemWriter pw = new PemWriter((Writer)new OutputStreamWriter(System.out));
            if (publicKey != null) {
                pw.writeObject((PemObjectGenerator)new PemObject("PUBLIC KEY", publicKey.getEncoded()));
            }
            if (privateKey != null) {
                pw.writeObject((PemObjectGenerator)new PemObject("PRIVATE KEY", privateKey.getEncoded()));
            }
            if (cert != null) {
                pw.writeObject((PemObjectGenerator)new PemObject("CERTIFICATE", cert.getEncoded()));
            }
            pw.flush();
            pw.close();
        }
        catch (IOException | CertificateEncodingException e) {
            throw Launcher.printUsageAndExit("Error printing X509 format: " + e.getMessage());
        }
    }

    public static Certificate selfSign(PublicKey pub, PrivateKey priv, String subjectDN, String signatureAlgorithm) {
        try {
            X500Name dn = new X500Name("CN=" + URLEncoder.encode(subjectDN, Charset.defaultCharset()));
            BigInteger certSerialNumber = BigInteger.valueOf(Instant.now().toEpochMilli());
            ContentSigner contentSigner = new JcaContentSignerBuilder(signatureAlgorithm).build(priv);
            Instant startDate = Instant.now();
            Instant endDate = startDate.plus(300L, ChronoUnit.DAYS);
            JcaX509v3CertificateBuilder certBuilder = new JcaX509v3CertificateBuilder(dn, certSerialNumber, Date.from(startDate), Date.from(endDate), dn, pub);
            return new JcaX509CertificateConverter().getCertificate(certBuilder.build(contentSigner));
        }
        catch (CertificateException | OperatorCreationException e) {
            throw Launcher.printUsageAndExit("Unable to create certificate: " + e.getMessage());
        }
    }

    private static void writeKeyToFile(boolean keySet, String outFile, String pubOutFile, JWK jwk, Gson gson) throws IOException, ParseException {
        JsonElement pubJson;
        JsonElement json;
        File output = new File(outFile);
        if (keySet) {
            List existingKeys = output.exists() ? JWKSet.load((File)output).getKeys() : Collections.emptyList();
            ArrayList<JWK> jwkList = new ArrayList<JWK>(existingKeys);
            jwkList.add(jwk);
            JWKSet jwkSet = new JWKSet(jwkList);
            json = JsonParser.parseString((String)jwkSet.toJSONObject(false).toJSONString());
            pubJson = JsonParser.parseString((String)jwkSet.toJSONObject(true).toJSONString());
        } else {
            json = JsonParser.parseString((String)jwk.toJSONString());
            pubJson = JsonParser.parseString((String)jwk.toPublicJWK().toJSONString());
        }
        try (BufferedWriter os = new BufferedWriter(new FileWriter(output));){
            os.write(gson.toJson(json));
        }
        if (pubOutFile != null) {
            os = new BufferedWriter(new FileWriter(pubOutFile));
            try {
                os.write(gson.toJson(pubJson));
            }
            finally {
                ((Writer)os).close();
            }
        }
    }

    private static void printKey(boolean keySet, JWK jwk, Gson gson) {
        if (keySet) {
            JWKSet jwkSet = new JWKSet(jwk);
            JsonElement json = JsonParser.parseString((String)jwkSet.toJSONObject(false).toJSONString());
            System.out.println(gson.toJson(json));
        } else {
            JsonElement json = JsonParser.parseString((String)jwk.toJSONString());
            System.out.println(gson.toJson(json));
        }
    }

    private static IllegalArgumentException printUsageAndExit(String message) {
        if (message != null) {
            System.err.println(message);
        }
        ImmutableList optionOrder = ImmutableList.of((Object)"t", (Object)"s", (Object)"c", (Object)"u", (Object)"a", (Object)"i", (Object)"g", (Object)"I", (Object)"p", (Object)"S", (Object)"o", (Object)"P", (Object[])new String[]{"x"});
        HelpFormatter formatter = new HelpFormatter();
        formatter.setOptionComparator(Comparator.comparingInt(arg_0 -> Launcher.lambda$printUsageAndExit$0((List)optionOrder, arg_0)));
        formatter.printHelp("java -jar json-web-key-generator.jar -t <keyType> [options]", options);
        System.exit(1);
        return new IllegalArgumentException("Program was called with invalid arguments");
    }

    private static /* synthetic */ int lambda$printUsageAndExit$0(List optionOrder, Option o) {
        return optionOrder.indexOf(o.getOpt());
    }

    static {
        ecCurves = Arrays.asList(Curve.P_256, Curve.SECP256K1, Curve.P_384, Curve.P_521);
        okpCurves = Arrays.asList(Curve.Ed25519, Curve.Ed448, Curve.X25519, Curve.X448);
        keyTypes = Arrays.asList(KeyType.RSA, KeyType.OCT, KeyType.EC, KeyType.OKP);
    }
}

