/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.media.server.component.audio;

import java.io.IOException;
import java.util.ArrayList;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.mobicents.media.MediaSink;
import org.mobicents.media.MediaSource;
import org.mobicents.media.server.component.Mixer;
import org.mobicents.media.server.impl.AbstractSink;
import org.mobicents.media.server.impl.AbstractSource;
import org.mobicents.media.server.scheduler.Scheduler;
import org.mobicents.media.server.scheduler.Task;
import org.mobicents.media.server.spi.format.AudioFormat;
import org.mobicents.media.server.spi.format.Format;
import org.mobicents.media.server.spi.format.FormatFactory;
import org.mobicents.media.server.spi.format.Formats;
import org.mobicents.media.server.spi.memory.Frame;
import org.mobicents.media.server.spi.memory.Memory;

/**
 * Implements audio mixer.
 * 
 * @author kulikov
 */
public class AudioMixer implements Mixer {
    private final static int POOL_SIZE = 100;

    //scheduler for mixer job scheduling
    private Scheduler scheduler;
    
    //the format of the output stream.
    private AudioFormat format = FormatFactory.createAudioFormat("LINEAR", 8000, 16, 1);
    private Formats formats = new Formats();

    //The pool of input streams
    private ArrayList<Input> pool = new ArrayList(POOL_SIZE);
    //Active input streams
    private ArrayList<Input> inputs = new ArrayList(100);
    
    //output stream
    private final Output output;

    private long period = 20000000L;
    private int packetSize = (int)(period / 1000000) * format.getSampleRate()/1000 * format.getSampleSize() / 8;

    //frames for mixing
    private Frame[] frames;

    private MixTask mixer;
    private volatile boolean started = false;

    protected long mixCount = 0;

    public AudioMixer(Scheduler scheduler) {
        this.scheduler = scheduler;
        formats.add(format);

        mixer = new MixTask(scheduler);
        output = new Output(scheduler);
        
        for (int i = 0; i < POOL_SIZE; i++) {
            pool.add(new Input(scheduler));
        }
    }

    /**
     * Gets the format of the output stream.
     * 
     * @return the audio format descriptor.
     */
    public Format getFormat() {
        return format;
    }

    /**
     * Modifies format of the output stream.
     *
     * @param format the format descriptor.
     */
    public void setFormat(Format format) {
        this.format = (AudioFormat) format;
        packetSize = (int)(period / 1000000) * this.format.getSampleRate()/1000 * this.format.getSampleSize() / 8;
    }

    /**
     * Creates new input for this mixer.
     *
     * @return Input as media sink object.
     */
    public synchronized MediaSink newInput() {
        Input input = pool.remove(0);
        input.buffer.clear();
        inputs.add(input);
        return input;
    }

    /**
     * Gets the mixed stream.
     *
     * @return the output stream
     */
    public MediaSource getOutput() {
        return output;
    }

    /**
     * Gets the used packet size.
     * 
     * @return the packet size value in bytes.
     */
    protected int getPacketSize() {
        return this.packetSize;
    }

    /**
     * Releases unused input stream
     *
     * @param input the input stream previously created
     */
    public synchronized void release(MediaSink input) {
        ((Input)input).recycle();
    }

    public synchronized void start() {
/*        if (!started) {
            started = true;
            output.buffer.clear();
            mixer.setDeadLine(scheduler.getClock().getTime() + 1);
            scheduler.submit(mixer);
            mixCount = 0;
//            output.start();
            for (Input input: inputs) {
                input.start();
            }
        }
         * 
         */
        startMixer();
    }

    public void startMixer() {
        if (!started) {
            started = true;
            output.buffer.clear();
            mixer.setDeadLine(scheduler.getClock().getTime() + 1);
            scheduler.submit(mixer);
            mixCount = 0;
            for (Input input: inputs) {
                input.start();
            }
        }
    }
    
    public synchronized void stop() {
//        started = false;
//        mixer.cancel();
//        output.stop();
//        for (Input input: inputs) input.stop();
        stopMixer(true);
    }

    public void stopMixer(boolean stopOutput) {
        started = false;
        mixer.cancel();
        if (stopOutput) {
            output.stop();
        }
        for (Input input: inputs) input.stop();
    }
    
    public String report() {
        StringBuilder builder = new StringBuilder();
        builder.append("Mixer{\n");
        
        builder.append("     output:  (");
        builder.append(output.report());
        builder.append(")\n");
        
        for (Input input : inputs) {
            builder.append("     input: (");
            builder.append(input.report());
            builder.append(")\n");
        }
        
        builder.append("}");
        return builder.toString();
    }
    
    /**
     * Input stream
     */
    private class Input extends AbstractSink {
        private final static int limit = 50;
        
        private ConcurrentLinkedQueue<Frame> buffer = new ConcurrentLinkedQueue();
        //private ElasticBuffer buffer = new ElasticBuffer(3, 10);

        /**
         * Creates new stream
         */
        public Input(Scheduler scheduler) {
            super("mixer.input", scheduler);
        }
        
        @Override
        public void onMediaTransfer(Frame frame) throws IOException {
            if (buffer.size() < limit) {
                buffer.offer(frame);
            }
//            buffer.write(frame);
        }

        /**
         * (Non Java-doc.)
         *
         * @see org.mobicents.media.server.impl.AbstractSink#getNativeFormats() 
         */
        public Formats getNativeFormats() {
            return formats;
        }

        /**
         * Indicates the state of the input buffer.
         *
         * @return true if input buffer has no frames.
         */
        protected boolean isEmpty() {
            return buffer.isEmpty();
        }

        /**
         * Retrieves frame from the input buffer.
         *
         * @return the media frame.
         */
        protected Frame poll() {
//            return buffer.read();
            return buffer.poll();
        }

        /**
         * Recycles output stream
         */
        protected void recycle() {
            buffer.clear();
            inputs.remove(this);
            pool.add(this);
        }

    }

    /**
     * Output stream
     */
    private class Output extends AbstractSource {
        private ConcurrentLinkedQueue<Frame> buffer = new ConcurrentLinkedQueue();

        /**
         * Creates new instance with default name.
         */
        public Output(Scheduler scheduler) {
            super("mixer.output", scheduler);
        }

        @Override
        public Frame evolve(long timestamp) {
            return buffer.poll();
        }

        /**
         * (Non Java-doc.)
         *
         * @see org.mobicents.media.server.impl.AbstractSource#getNativeFormats()
         */
        public Formats getNativeFormats() {
            return formats;
        }

        @Override
        public void start() {
            startMixer();
            super.start();            
        }
        
        @Override
        public void stop() {
            stopMixer(false);
            super.stop();
        }
    }

    private class MixTask extends Task {

        private volatile long priority;
        private volatile long duration = -100;

        public MixTask(Scheduler scheduler) {
            super(scheduler);
        }
        
        public long getPriority() {
            return priority;
        }

        public void setPriority(long priority) {
            this.priority = priority;
        }

        public long getDuration() {
            return duration;
        }

        public long perform() {
            try {
            //allocate new frame
            Frame frame = Memory.allocate(packetSize);
            byte[] data = frame.getData();

            //poll inputs
            frames = new Frame[inputs.size()];
            for (int i = 0; i < frames.length; i++) {
                frames[i] = inputs.get(i).poll();
            }

            //do mixing
            int k = 0;
            for (int j = 0; j < packetSize; j += 2) {
                short s = 0;
                for (int i = 0; i < frames.length; i++) {
                    if (frames[i] != null) {
                        s += (short) (((frames[i].getData()[j + 1]) << 8) | (frames[i].getData()[j] & 0xff));
                    } 
                }
                data[k++] = (byte) (s);
                data[k++] = (byte) (s >> 8);
            }
            //recycle received frames
            for (int i = 0; i < frames.length; i++) {
                if (frames[i] != null)  {
                    if (frames[i].getHeader() != null) {
                        frame.setHeader(frames[i].getHeader());
                    }
                    frames[i].recycle();
                }
                
            }

            //assign attributes to the new frame
            frame.setOffset(0);
            frame.setLength(packetSize);
            frame.setDuration(period);
            frame.setFormat(format);

            //finita la comedia
            output.buffer.offer(frame);
            setDeadLine(getDeadLine() + period);

            scheduler.submit(this);
            output.wakeup();

            mixCount++;

            return 0;
            } catch (Exception e) {
                e.printStackTrace();
            }
            return 0;
        }
    }

}
