/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xcap.diff.component;

import java.util.Map;

import org.mobicents.protocols.xcap.diff.BuildPatchException;

/**
 * Builds element patch components. Generated content is valid for xcap-diff
 * document, ie, xcap-diff document could look like follows(bold content is
 * generated by instance of this interface):
 * 
 * <pre>
 * &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *    &lt;xcap-diff xmlns="urn:ietf:params:xml:ns:xcap-diff"
 *               xcap-root="http://xcap.example.com/"&gt;
 * 
 *     <b>&lt;element sel="rls-services/users/sip:joe@example.com/index/~~/&#002A;/service%5b@uri='sip:marketing@example.com'%5d" xmlns:rl="urn:ietf:params:xml:ns:resource-lists"&gt;
 * 		&lt;s:service uri="sip:marketing@example.com"&gt;
 *        		&lt;s:list name="marketing"&gt;
 *        			&lt;rl:entry uri="sip:joe@example.com"/&gt;
 *        			&lt;rl:entry uri="sip:sudhir@example.com"/&gt;
 *        		&lt;/s:list&gt;
 *        		&lt;s:packages&gt;
 *        			&lt;s:package&gt;presence&lt;/s:package&gt;
 *        		&lt;/s:packages&gt;
 *        	&lt;/s:service&gt;
 *       &lt;/element&gt;</b>
 * 
 *    &lt;/xcap-diff&gt;
 * </pre>
 * 
 * @author baranowb
 * @author martins
 * 
 * @param <C>
 *            the component type, defines what is the concrete type of each
 *            patch component, to be aggregated in a XCAP DIFF patch
 * 
 * @param <E>
 *            the element type, defines what is the concrete type of XML
 *            elements used
 * 
 */
public interface ElementPatchComponentBuilder<C, E> {

	/**
	 * Builds a patch component which indicates an update of an element.
	 * 
	 * @param sel
	 *            the selector which points to the element in the owner
	 *            document, relative to the final patch xcap root.
	 * @param element
	 *            the updated element
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in the element
	 *            and in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public C buildPatchComponent(String sel, E element,
			Map<String, String> namespaceBindings) throws BuildPatchException;

	/**
	 * Builds a patch component which indicates the existence an element.
	 * 
	 * @param sel
	 *            the selector which points to the element in the owner
	 *            document, relative to the final patch xcap root.
	 * @param exists
	 *            indicates if the element exists
	 * @param namespaceBindings
	 *            namespace bindings for undeclared namespaces in sel
	 * @return
	 * @throws BuildPatchException
	 *             if an exception occurred while building the patch, the
	 *             concrete exception is the exception's cause.
	 */
	public C buildPatchComponent(String sel, boolean exists,
			Map<String, String> namespaceBindings) throws BuildPatchException;

}
