/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xcap.diff.dom.utils;

import javax.xml.namespace.NamespaceContext;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class DOMNodeComparator {

	public static class Result {

		boolean created;
		boolean removed;
		boolean different;
		boolean attribute;
		
		Node newNode;
		Node oldNode;

		public Node getOldNode() {
			return oldNode;
		}

		public Node getNewNode() {
			return newNode;
		}

		public boolean isDifferent() {
			return different;
		}

		public boolean isCreated() {
			return created;
		}

		public boolean isRemoved() {
			return removed;
		}
	}

	public Result compare(Document oldDoc, Document newDoc,
			String nodeSelector, NamespaceContext namespaceContext) {

		Node oldNode = getNode(oldDoc, nodeSelector, namespaceContext);
		Node newNode = getNode(newDoc, nodeSelector, namespaceContext);

		return this.compare(oldNode, newNode);
	}
	/**
	 * Performs full node comparison - attributes, children and node type.
	 * @param oldNode
	 * @param newNode
	 * @return
	 */
	public Result compare(Node oldNode, Node newNode) {

		Result result = new Result();
		result.oldNode = oldNode;
		result.newNode = newNode;

		if (oldNode == null) {
			if (newNode != null) {
				result.created = true;
				result.different = true;
			}
		} else {
			if (newNode == null) {
				result.removed = true;
				result.different = true;
			} else {
				result.different = !oldNode.isEqualNode(newNode);
			}
		}

		return result;
	}

	
	/**
	 * Performs only node comparison, that is - it compares only type, tag, prefix, and similar.
	 * @param n1
	 * @param n2
	 * @return
	 */
	public boolean compareNode(Node n1, Node n2) {
		//seriously, WHY those are not part of DOM ? NodeList, NodeAttributeMap etc?
		if (n1 == n2) {
			return true;
		}
	
		if( (n1!=null && n2 ==null) || (n1==null && n2 !=null))
		{
			return false;
		}
		
		if (n1.getNodeType() != n2.getNodeType()) {
			return false;
		}
		// in theory nodeName can't be null but better be careful
		// who knows what other implementations may be doing?...
		if (n2.getNodeName() == null) {
			if (n1.getNodeName() != null) {
				return false;
			}
		} else if (!n2.getNodeName().equals(n1.getNodeName())) {
			return false;
		}

		if (n2.getLocalName() == null) {
			if (n1.getLocalName() != null) {
				return false;
			}
		} else if (!n2.getLocalName().equals(n1.getLocalName())) {
			return false;
		}

		if (n2.getNamespaceURI() == null) {
			if (n1.getNamespaceURI() != null) {
				return false;
			}
		} else if (!n2.getNamespaceURI().equals(n1.getNamespaceURI())) {
			return false;
		}

		if (n2.getPrefix() == null) {
			if (n1.getPrefix() != null) {
				return false;
			}
		} else if (!n2.getPrefix().equals(n1.getPrefix())) {
			return false;
		}

		if (n2.getNodeValue() == null) {
			if (n1.getNodeValue() != null) {
				return false;
			}
		} else if (!n2.getNodeValue().equals(n1.getNodeValue())) {
			return false;
		}
		return true;

	}
	
	/**
	 * Performs
	 * @param nodes1
	 * @param nodes2
	 * @return
	 */
	public boolean compareChildren(NodeList nodes1, NodeList nodes2) {
		if(nodes1 == null && nodes2 == null)
		{
			return true;
		}
		if( (nodes1!=null && nodes2 ==null) || (nodes1==null && nodes2 !=null))
		{
			return false;
		}
		if (nodes1.getLength() != nodes2.getLength()) {
			return false;
		}
		for (int index = 0; index < nodes1.getLength(); index++) {
			Node child1 = nodes1.item(index);
			Node child2 = nodes2.item(index);
			if (child1 != null && child2 != null) {
				if (!(child1).isEqualNode(child2)) {
					return false;
				}

			}
		}
		return true;
	}
	
	public boolean compareAttributes(NamedNodeMap map1, NamedNodeMap map2) {
		if(map1 == null && map2 == null)
		{
			return true;
		}
		if( (map1!=null && map2 ==null) || (map1==null && map2 !=null))
		{
			return false;
		}
		
		int len = map1.getLength();
		if (len != map2.getLength()) {
			return false;
		}
		for (int i = 0; i < len; i++) {
			Node n1 = map1.item(i);
			if (n1.getLocalName() == null) { // DOM Level 1 Node
				Node n2 = map2.getNamedItem(n1.getNodeName());
				if (n2 == null || !(n1).isEqualNode(n2)) {
					return false;
				}
			} else {
				Node n2 = map2.getNamedItemNS(n1.getNamespaceURI(), n1.getLocalName());
				if (n2 == null || !(n1).isEqualNode(n2)) {
					return false;
				}
			}
		}
		return true;
	}
	
	private final XPathFactory xPathFactory = XPathFactory.newInstance();

	private Node getNode(Document doc, String nodeSelector,
			NamespaceContext namespaceContext) {

		if (doc == null) {
			return null;
		}

		final XPath xpath = xPathFactory.newXPath();

		if (namespaceContext != null) {
			xpath.setNamespaceContext(namespaceContext);
		}

		try {
			// exec query to get element
			final NodeList nodeList = (NodeList) xpath.evaluate(nodeSelector,
					doc, XPathConstants.NODESET);
			if (nodeList.getLength() == 1) {
				return nodeList.item(0);
			} else if (nodeList.getLength() == 0) {
				return null;
			} else {
				throw new IllegalArgumentException("Multiple nodes match "
						+ nodeSelector);
			}
		} catch (XPathExpressionException e) {
			throw new IllegalArgumentException(e.getMessage(), e);
		}
	}
}
