/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat, Inc. and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.mobicents.protocols.xml.patch.dom;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;


import org.mobicents.protocols.xcap.diff.BuildPatchException;
import org.mobicents.protocols.xcap.diff.dom.utils.DOMNodeComparator;
import org.mobicents.protocols.xcap.diff.dom.utils.DOMXmlUtils;
import org.mobicents.protocols.xml.patch.XmlPatchOperationsBuilder;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * 
 * DOM Impl for {@link XmlPatchOperationsBuilder}.
 * 
 * @author baranowb
 * @author martins
 * 
 */
public class DOMXmlPatchOperationsBuilder implements
		XmlPatchOperationsBuilder<Document, Element, Node, Node> {

	private static final String ADD_ELEMENT_NAME = "add";
	private static final String REPLACE_ELEMENT_NAME = "replace";
	private static final String REMOVE_ELEMENT_NAME = "remove";

	private static final String SEL_ATTR_NAME = "sel";
	private static final String TYPE_ATTR_NAME = "type";
	private static final String POS_ATTR_NAME = "pos";
	private static final String WS_ATTR_NAME = "ws";

	private static final String DEFAULT_ELEM_PREFIX = "xpo";

	@Override
	public Element addAttribute(String sel, String attrName, String attrValue,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return addText(sel, '@' + attrName, attrValue, namespaceBindings);
	}

	@Override
	public Element addElement(String sel, Node element,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return addOrReplaceNode(ADD_ELEMENT_NAME, sel, element,
				namespaceBindings);
	}

	@Override
	public Element addNode(
			String sel,
			org.mobicents.protocols.xml.patch.XmlPatchOperationsBuilder.Pos pos,
			Node node, Map<String, String> namespaceBindings)
			throws BuildPatchException {
		final Element patchInstruction = addOrReplaceNode(ADD_ELEMENT_NAME,
				sel, node, namespaceBindings);
		patchInstruction.setAttribute(POS_ATTR_NAME, pos.toString());
		return patchInstruction;
	}

	@Override
	public Element addPrefixNamespaceDeclaration(String sel,
			String namespacePrefix, String namespaceValue,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return addText(sel, "namespace::" + namespacePrefix, namespaceValue,
				namespaceBindings);
	}

	@Override
	public Element replaceAttribute(String sel, String attributeValue,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return replaceText(sel, attributeValue, namespaceBindings);
	}

	@Override
	public Element replaceElement(String sel, Node element,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return addOrReplaceNode(REPLACE_ELEMENT_NAME, sel, element,
				namespaceBindings);
	}

	@Override
	public Element replaceNode(String sel, Node node,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return addOrReplaceNode(REPLACE_ELEMENT_NAME, sel, node,
				namespaceBindings);
	}

	@Override
	public Element replacePrefixNamespaceDeclaration(String sel,
			String namespaceValue, Map<String, String> namespaceBindings)
			throws BuildPatchException {
		return replaceText(sel, namespaceValue, namespaceBindings);
	}

	@Override
	public Element removeAttribute(String sel,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return remove(sel, null, namespaceBindings);
	}

	@Override
	public Element removeElement(String sel,
			org.mobicents.protocols.xml.patch.XmlPatchOperationsBuilder.Ws ws,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return remove(sel, ws, namespaceBindings);
	}

	@Override
	public Element removeNode(String sel, Map<String, String> namespaceBindings)
			throws BuildPatchException {
		return remove(sel, null, namespaceBindings);
	}

	@Override
	public Element removePrefixNamespaceDeclaration(String sel,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		return remove(sel, null, namespaceBindings);
	}

	private Element addText(String sel, String type, String textContent,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		Element patchInstruction = DOMXmlUtils.createElement(ADD_ELEMENT_NAME,
				DEFAULT_ELEM_PREFIX,
				XmlPatchOperationsBuilder.XML_PATCH_OPS_NAMESPACE,
				namespaceBindings);
		patchInstruction.setAttribute(SEL_ATTR_NAME, sel);
		patchInstruction.setAttribute(TYPE_ATTR_NAME, type);
		patchInstruction.setTextContent(textContent);
		return patchInstruction;
	}

	private Element addOrReplaceNode(String elementName, String sel, Node node,
			Map<String, String> namespaceBindings) throws BuildPatchException {

		Element patchInstruction = DOMXmlUtils.createElement(elementName,
				DEFAULT_ELEM_PREFIX,
				XmlPatchOperationsBuilder.XML_PATCH_OPS_NAMESPACE,
				namespaceBindings);
		patchInstruction.setAttribute(SEL_ATTR_NAME, sel);
		// import element
		Node importedNode = patchInstruction.getOwnerDocument().importNode(
				node, true);
		patchInstruction.appendChild(importedNode);
		return patchInstruction;
	}

	private Element replaceText(String sel, String textContent,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		Element patchInstruction = DOMXmlUtils.createElement(
				REPLACE_ELEMENT_NAME, DEFAULT_ELEM_PREFIX,
				XmlPatchOperationsBuilder.XML_PATCH_OPS_NAMESPACE,
				namespaceBindings);
		patchInstruction.setAttribute(SEL_ATTR_NAME, sel);
		patchInstruction.setTextContent(textContent);
		return patchInstruction;
	}

	private Element remove(String sel, Ws ws,
			Map<String, String> namespaceBindings) throws BuildPatchException {
		Element patchInstruction = DOMXmlUtils.createElement(
				REMOVE_ELEMENT_NAME, DEFAULT_ELEM_PREFIX,
				XmlPatchOperationsBuilder.XML_PATCH_OPS_NAMESPACE,
				namespaceBindings);
		patchInstruction.setAttribute(SEL_ATTR_NAME, sel);
		if (ws != null) {
			patchInstruction.setAttribute(WS_ATTR_NAME, ws.toString());
		}
		return patchInstruction;
	}

	/* (non-Javadoc)
	 * @see org.mobicents.protocols.xml.patch.XmlPatchOperationsBuilder#buildPatchOperations(java.lang.Object, java.lang.Object)
	 */
	@Override
	public Element[] buildPatchInstructions(Document originalDocument, Document patchedDocument) throws BuildPatchException {
		Collection<Element> instructions = buildPatchOperationsForNode(WILD_CARD_SELECTOR,originalDocument.getDocumentElement(),patchedDocument.getDocumentElement());
		return instructions.toArray(new Element[instructions.size()]);
	}
	
	// --------------- helper methods
	protected DOMNodeComparator domNodeComparator = new DOMNodeComparator();
	protected static final String WILD_CARD_SELECTOR = "*";
	protected static final String NAMESPACE_PREFIX = "namespace::";
	protected static final String XMLNS_PREFIX = "xmlns:";
	protected static final int OP_THRESHOLD = 3;
	
	protected String createPositionalSelector(String parentSelector, int index)
	{
		//NOTE: passed index is DOM, starts from '0', XML Diff index constraint starts from '1'.
		return new StringBuilder(parentSelector).append('/').append(WILD_CARD_SELECTOR).append('[').append(index+1).append(']').toString();  
	}

	protected String createAttributeSelector(String parentSelector, String attributeName)
	{
		return new StringBuilder(parentSelector).append("/@").append(attributeName).toString();
	}
	
	protected String createNamespaceSelector(String selector, String name) {
		return new StringBuilder(selector).append('/').append(NAMESPACE_PREFIX).append(name.replace(XMLNS_PREFIX, "")).toString();
	}
	/**
	 * 
	 * @param selector
	 *            - selector pointing to node
	 * @param originalDocument
	 *            - node in original
	 * @param patchedDocument
	 * @return
	 * @throws BuildPatchException
	 */
	protected Collection<Element> buildPatchOperationsForNode(String selector, Node oldElement, Node newElement) throws BuildPatchException {
		if (oldElement == null && newElement == null) {
			throw new BuildPatchException("oldElement and newElement must not be null!");
		}

		if (selector == null) {
			throw new BuildPatchException("Selector must not be null.");
		}

		// FIXME: add check if node is a text node ?
		if (oldElement == null && newElement != null) {
			// add
			//NOTE: default, impl, supports add only at the end. 
			//pushing parent - cause without pos, we add at targetNode children, so current sel, must point ot parent.
			Element addElement = this.addElement(selector.substring(0,selector.lastIndexOf("/")), newElement, null);
			List<Element> resultInstructions = new ArrayList<Element>();
			resultInstructions.add(addElement);
			return resultInstructions;
		} else if (oldElement != null && newElement == null) {
			// remove
			Element removeElement = this.removeNode(selector, null);
			List<Element> resultInstructions = new ArrayList<Element>();
			resultInstructions.add(removeElement);
			return resultInstructions;
		} else if (!domNodeComparator.compareNode(oldElement, newElement)) {
			// shallow compare node says 'no' - that means node type/tag/content
			// does not match.
			// ergo: replace
			Element replaceElement = this.replaceElement(selector, newElement, null);
			List<Element> resultInstructions = new ArrayList<Element>();
			resultInstructions.add(replaceElement);
			return resultInstructions;
		} else {
			// NOTE: there actually should be a way to compare attribs in the
			// same way nodes are compared.
			// not sure if its a good thing here :)

			// NOTE: type/tag/etc is the same. We have to compare attributes and
			// children.
			List<Element> nodePatchOpInstruction = new ArrayList<Element>();
			// lets do children first. if there is enough of them, we will skip
			// attribs and provide full patch over current element.
			NodeList oldChildNodeList = oldElement.getChildNodes();
			NodeList newChildNodeList = newElement.getChildNodes();
			if (!domNodeComparator.compareChildren(oldChildNodeList, newChildNodeList)) {
				// we have different children
				int index = 0;
				for (; index < newChildNodeList.getLength(); index++) {
					Node oldChildNode = null;
					Node newChildNode = newChildNodeList.item(index);
					if (index < oldChildNodeList.getLength()) {
						oldChildNode = oldChildNodeList.item(index);
					}

					// DOM - contrary to all programming, indexes from 1...
					Collection<Element> diff = buildPatchOperationsForNode(createPositionalSelector(selector, index), oldChildNode, newChildNode);
					nodePatchOpInstruction.addAll(diff);
				}

				// now, if there is more nodes in old list, we need to check.
				for (; index < oldChildNodeList.getLength(); index++) {

					Node oldChildNode = oldChildNodeList.item(index);
					Node newChildNode = null;
					// DOM - contrary to all programming, indexes from 1...
					Collection<Element> diff = buildPatchOperationsForNode(createPositionalSelector(selector, index), oldChildNode, newChildNode);
					nodePatchOpInstruction.addAll(diff);
				}

			}

			// if there is more than 3 present, replace with whole node
			// if its / than we cant go up...
			if ((nodePatchOpInstruction.size() > OP_THRESHOLD) && !selector.equals(WILD_CARD_SELECTOR)) 
			{
				// return replace element patch
				Element replaceElement = this.replaceElement(selector, newElement, null);
				List<Element> resultInstructions = new ArrayList<Element>();
				resultInstructions.add(replaceElement);
				return resultInstructions;
			} else if (!domNodeComparator.compareAttributes(oldElement.getAttributes(), newElement.getAttributes())) {
				// This is required step, if one attribute changes its better to
				// provide patch for that
				// rather than pushing whole element 'replace';
				// NOTE: this is tricky, since it includes namespaces ;[
				NamedNodeMap oldNodeAttributeMap = oldElement.getAttributes();
				NamedNodeMap newNodeAttributeMap = newElement.getAttributes();

				
				List<Element> resultInstructions = new ArrayList<Element>();
				for (int index = 0; index < newNodeAttributeMap.getLength(); index++) {

					Attr newAttr = (Attr) newNodeAttributeMap.item(index);

					Attr oldAttr = null;
					if (newAttr.getNamespaceURI() == null) {
						oldAttr = (Attr) oldNodeAttributeMap.getNamedItem(newAttr.getName());
					} else {
						oldAttr = (Attr) oldNodeAttributeMap.getNamedItemNS(newAttr.getNamespaceURI(), newAttr.getName());
					}
					if (oldAttr == null) {
						// ADD
						String name = newAttr.getName();
						Element addAttributeElement = null;
						if (name.startsWith(XMLNS_PREFIX)) {
							addAttributeElement = this.addPrefixNamespaceDeclaration(selector, name.replace(XMLNS_PREFIX, ""), newAttr.getValue(), null);
						} else {
							addAttributeElement = this.addAttribute(selector, newAttr.getName(), newAttr.getValue(), null);
						}

						resultInstructions.add(addAttributeElement);
					} else if (!newAttr.getValue().equals(oldAttr.getValue())) {
						// replace value

						String name = newAttr.getName();
						Element replaceAttributeElement = null;
						if (name.startsWith(XMLNS_PREFIX)) {
							replaceAttributeElement = this.replacePrefixNamespaceDeclaration(createNamespaceSelector(selector, name), newAttr.getValue(), null);
						} else {
							replaceAttributeElement = this.replaceAttribute(createAttributeSelector(selector,newAttr.getName()), newAttr.getValue(), null);
						}

						resultInstructions.add(replaceAttributeElement);
					}
				}

				// inverse, this seems the easiest way to check for remove...
				for (int index = 0; index < oldNodeAttributeMap.getLength(); index++) {

					Attr oldAttr = (Attr) oldNodeAttributeMap.item(index);

					Attr newAttr = null;
					if (oldAttr.getNamespaceURI() == null) {
						newAttr = (Attr) newNodeAttributeMap.getNamedItem(oldAttr.getName());
					} else {
						newAttr = (Attr) newNodeAttributeMap.getNamedItemNS(oldAttr.getNamespaceURI(), oldAttr.getName());
					}

					if (newAttr == null) {
						// REMOVE
						String name = oldAttr.getName();
						Element removeAttributeElement = null;
						if (name.startsWith(XMLNS_PREFIX)) {
							removeAttributeElement = this.removeNode(createNamespaceSelector(selector, name), null);
						} else {
							removeAttributeElement = this.removeAttribute(createAttributeSelector(selector,oldAttr.getName()) , null);
						}

						resultInstructions.add(removeAttributeElement);
					}
				}

				nodePatchOpInstruction.addAll(resultInstructions);
			}
			return nodePatchOpInstruction;
		}

	}

	
// -------------- Bit more sophisticated implementation, needs more CPU, but patch creation is more refined.	
//	/*
//	 * (non-Javadoc)
//	 * 
//	 * @see org.mobicents.protocols.xml.patch.XmlPatchOperationsBuilder#
//	 * buildPatchOperations(java.lang.Object, java.lang.Object)
//	 */
//	@Override
//	public Element[] buildPatchOperations(Document originalDocument, Document patchedDocument) throws BuildPatchException {
//		Collection<XMLChange> instructions = buildPatchOperationsForNode(WILD_CARD_SELECTOR, 0, originalDocument.getDocumentElement(),
//				patchedDocument.getDocumentElement());
//		ArrayList<Element> changes = new ArrayList<Element>();
//		for (XMLChange xmlChange : instructions) {
//			changes.add(xmlChange.element);
//		}
//		return changes.toArray(new Element[changes.size()]);
//	}
//
//	/**
//	 * 
//	 * @param selector
//	 *            - selector pointing to node
//	 * @param originalDocument
//	 *            - node in original
//	 * @param patchedDocument
//	 * @return
//	 * @throws BuildPatchException
//	 */
//	protected Collection<XMLChange> buildPatchOperationsForNode(String selector, int level, Node oldElement, Node newElement) throws BuildPatchException {
//		if (oldElement == null && newElement == null) {
//			throw new BuildPatchException("oldElement and newElement must not be null!");
//		}
//
//		if (selector == null) {
//			throw new BuildPatchException("Selector must not be null.");
//		}
//		if (oldElement == null && newElement != null) {
//			// add
//			Element addElement = this.addElement(selector, newElement, null);
//			List<XMLChange> resultInstructions = new ArrayList<XMLChange>();
//			XMLChange change = createXMLChange(XMLChangeType.add, level, true, addElement);
//			resultInstructions.add(change);
//			return resultInstructions;
//		} else if (oldElement != null && newElement == null) {
//			// remove
//			Element removeElement = this.removeNode(selector, null);
//			XMLChange change = createXMLChange(XMLChangeType.remove, level, true, removeElement);
//			List<XMLChange> resultInstructions = new ArrayList<XMLChange>();
//			resultInstructions.add(change);
//			return resultInstructions;
//		} else if (!domNodeComparator.compareNode(oldElement, newElement)) {
//			// shallow compare node says 'no' - that means node type/tag/content
//			// does not match.
//			// ergo: replace
//			Element replaceElement = this.replaceElement(selector, newElement, null);
//
//			XMLChange change = createXMLChange(XMLChangeType.replace, level, true, replaceElement);
//			List<XMLChange> resultInstructions = new ArrayList<XMLChange>();
//			resultInstructions.add(change);
//			return resultInstructions;
//		} else {
//			// NOTE: there actually should be a way to compare attribs in the
//			// same way nodes are compared.
//			// not sure if its a good thing here :)
//
//			// NOTE: type/tag/etc is the same. We have to compare attributes and
//			// children.
//			List<XMLChange> nodePatchOpInstruction = new ArrayList<XMLChange>();
//			// lets do children first. if there is enough of them, we will skip
//			// attribs and provide full patch over current element.
//			NodeList oldChildNodeList = oldElement.getChildNodes();
//			NodeList newChildNodeList = newElement.getChildNodes();
//			if (!domNodeComparator.compareChildren(oldChildNodeList, newChildNodeList)) {
//				// we have different children
//				int index = 0;
//				for (; index < newChildNodeList.getLength(); index++) {
//					Node oldChildNode = null;
//					Node newChildNode = newChildNodeList.item(index);
//					if (index < oldChildNodeList.getLength()) {
//						oldChildNode = oldChildNodeList.item(index);
//					}
//
//					// DOM - contrary to all programming, indexes from 1...
//					Collection<XMLChange> diff = buildPatchOperationsForNode(createPositionalSelector(selector, index), level + 1, oldChildNode, newChildNode);
//					nodePatchOpInstruction.addAll(diff);
//				}
//
//				// now, if there is more nodes in old list, we need to check.
//				for (; index < oldChildNodeList.getLength(); index++) {
//
//					Node oldChildNode = oldChildNodeList.item(index);
//					Node newChildNode = null;
//					// DOM - contrary to all programming, indexes from 1...
//					Collection<XMLChange> diff = buildPatchOperationsForNode(createPositionalSelector(selector, index), level + 1, oldChildNode, newChildNode);
//					nodePatchOpInstruction.addAll(diff);
//				}
//
//			}
//
//			// if there is more than 3 present, replace with whole node
//			if ((nodePatchOpInstruction.size() > OP_THRESHOLD) && checkReplaceStorm(nodePatchOpInstruction, level + 1)) {
//				// check, if there are OP_THRESHOLD consecutive ops for replace
//				// on elements we execute code below;
//
//				// return replace element patch
//				Element replaceElement = this.replaceElement(selector, newElement, null);
//
//				XMLChange change = createXMLChange(XMLChangeType.replace, level, true, replaceElement);
//				List<XMLChange> resultInstructions = new ArrayList<XMLChange>();
//				resultInstructions.add(change);
//				return resultInstructions;
//			} else if (!domNodeComparator.compareAttributes(oldElement.getAttributes(), newElement.getAttributes())) {
//				// This is required step, if one attribute changes its better to
//				// provide patch for that
//				// rather than pushing whole element 'replace';
//				// NOTE: this is tricky, since it includes namespaces ;[
//				NamedNodeMap oldNodeAttributeMap = oldElement.getAttributes();
//				NamedNodeMap newNodeAttributeMap = newElement.getAttributes();
//
//				int index = 0;
//				List<XMLChange> resultInstructions = new ArrayList<XMLChange>();
//				for (; index < newNodeAttributeMap.getLength(); index++) {
//
//					Attr newAttr = (Attr) newNodeAttributeMap.item(index);
//
//					Attr oldAttr = null;
//					if (newAttr.getNamespaceURI() == null) {
//						oldAttr = (Attr) oldNodeAttributeMap.getNamedItem(newAttr.getName());
//					} else {
//						oldAttr = (Attr) oldNodeAttributeMap.getNamedItemNS(newAttr.getNamespaceURI(), newAttr.getName());
//					}
//
//					XMLChange change = null;
//					if (oldAttr == null) {
//						// ADD
//						String name = newAttr.getName();
//						Element addAttributeElement = null;
//
//						if (name.startsWith(XMLNS_PREFIX)) {
//							// TODO: ??
//							continue;
//						} else {
//							addAttributeElement = this.addAttribute(selector, newAttr.getName(), newAttr.getValue(), null);
//							change = createXMLChange(XMLChangeType.add, level, false, addAttributeElement);
//						}
//
//						resultInstructions.add(change);
//					} else if (!newAttr.getValue().equals(oldAttr.getValue())) {
//						// replace value
//
//						String name = newAttr.getName();
//						Element replaceAttributeElement = null;
//						if (name.startsWith(XMLNS_PREFIX)) {
//							// TODO: ??
//							continue;
//						} else {
//							replaceAttributeElement = this.replaceAttribute(selector + "/@" + newAttr.getName(), newAttr.getValue(), null);
//							createXMLChange(XMLChangeType.replace, level, false, replaceAttributeElement);
//						}
//
//						resultInstructions.add(change);
//					}
//				}
//
//				// inverse, this seems the easiest way to check for remove...
//				for (; index < oldNodeAttributeMap.getLength(); index++) {
//
//					Attr oldAttr = (Attr) oldNodeAttributeMap.item(index);
//
//					Attr newAttr = null;
//					if (oldAttr.getNamespaceURI() == null) {
//						newAttr = (Attr) newNodeAttributeMap.getNamedItem(oldAttr.getName());
//					} else {
//						newAttr = (Attr) newNodeAttributeMap.getNamedItemNS(oldAttr.getNamespaceURI(), oldAttr.getName());
//					}
//
//					if (newAttr == null) {
//						// REMOVE
//						String name = oldAttr.getName();
//						XMLChange change = null;
//						Element removeAttributeElement = null;
//						if (name.startsWith(XMLNS_PREFIX)) {
//							// TODO: ??
//							continue;
//						} else {
//							removeAttributeElement = this.removeAttribute(selector + "/@" + oldAttr.getName(), null);
//							change = createXMLChange(XMLChangeType.remove, level, false, removeAttributeElement);
//						}
//
//						resultInstructions.add(change);
//					}
//				}
//
//				nodePatchOpInstruction.addAll(resultInstructions);
//			}
//			return nodePatchOpInstruction;
//		}
//
//	}
//	
//	/**
//	 * @param nodePatchOpInstruction
//	 * @return
//	 */
//	private boolean checkReplaceStorm(List<XMLChange> nodePatchOpInstruction, int level) {
//		int consecutive = 0;
//		
//		for(XMLChange change:nodePatchOpInstruction)
//		{
//			if(change.node && change.level == level && change.type == XMLChangeType.replace)
//			{
//				
//				consecutive ++;
//				if(consecutive>OP_THRESHOLD)
//					return true;
//			}else
//			{
//				consecutive = 0;
//			}
//		}
//		
//		return false;
//	}
//
//	private static XMLChange createXMLChange(XMLChangeType type, int level,boolean isNode, Element element)
//	{
//		XMLChange change = new XMLChange();
//		change.level = level;
//		change.element = element;
//		change.node = isNode;
//		change.type = type;
//		return change;
//	}
//	private static enum XMLChangeType{ add,replace,remove};
//	private static class XMLChange
//	{
//		XMLChangeType type;
//		int level;
//		boolean node; //if false, it means attribute has changed. BAM
//		Element element; //diff element/instruction.
//
//		
//		public static XMLChange createXMLChange(XMLChangeType type, int level,boolean isNode, Element element)
//		{
//			XMLChange change = new XMLChange();
//			change.level = level;
//			change.element = element;
//			change.node = isNode;
//			change.type = type;
//			return change;
//		}
//	}


}
