package org.mobicents.media.server.impl.events.announcement;

import java.io.Serializable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.log4j.Logger;

/**
 * This is a Singleton class that acts as Manager for File caching. The
 * FileCacheManager.startCleanerProcess() will initiate back ground thread that
 * scans the Files. Files that are not used for period greater than scanDelay
 * are removed from cache and if these files are remote files for which
 * AudioPlayer would have created local File, these local files are deleted
 * 
 * @author amit bhayani
 * 
 */
public class FileCacheManager implements Serializable {

    private static final Logger logger = Logger.getLogger(FileCacheManager.class);
    private static final AtomicLong fileName = new AtomicLong(1);
    public static final FileCacheManager fileCacheManager = new FileCacheManager();
    private transient Future worker;
    private static transient ScheduledExecutorService cleanerThread = Executors.newSingleThreadScheduledExecutor(new FileCacheManagerThreadFactory());
    private boolean scannerStarted = false;    // Let the default cleaning be kicked in after every 5 mins
    private long scanDelay = 1000 * 60 * 5;
    private transient ConcurrentHashMap<String, FileCache> fileCache = new ConcurrentHashMap<String, FileCache>();

    private FileCacheManager() {
    }

    public static FileCacheManager getInstance() {
        return fileCacheManager;
    }

    public void addFileCache(String key, FileCache file) {
        this.fileCache.put(key, file);
    }

    public void removeFileCache(String key) {
        this.fileCache.remove(key);
    }

    public FileCache getFileCache(String key) {
        return fileCache.get(key);
    }

    public void setScanDelay(long scanDelay) {
        this.scanDelay = scanDelay;
    }

    public long getScanDelay() {
        return this.scanDelay;
    }

    public void startCleanerProcess() throws Exception {
        if (!this.scannerStarted) {
            this.scannerStarted = true;
            worker = cleanerThread.scheduleAtFixedRate(new Cleaner(), this.scanDelay, this.scanDelay,
                    TimeUnit.MILLISECONDS);
            if (logger.isDebugEnabled()) {
                logger.debug("Background cleaning thread started and will be kicked after every " + this.scanDelay + " millisecons");
            }
        } else {
            throw new Exception("Cleaner Thread already started");
        }
    }

    public long getUniqueFileName() {
        long current = -1;
        boolean b = true;
        while (b) {
            current = fileName.get();
            if (current == Long.MAX_VALUE) {
                b = !fileName.compareAndSet(current, 1);
            } else {
                b = !fileName.compareAndSet(current, current + 1);
            }
        }
        return current;
    }

    private class Cleaner implements Runnable {

        public void run() {
            long currentTime = System.currentTimeMillis();

            for (String key : fileCache.keySet()) {
                FileCache b = fileCache.get(key);
                if (((currentTime - b.getTimestamp()) > scanDelay)) {
                    b = fileCache.remove(key);
                    if (b.isDelete()) {
                        b.getFile().delete();
                    }
                }
            }
        }
    }

    private static class FileCacheManagerThreadFactory implements ThreadFactory {

        public Thread newThread(Runnable r) {
            Thread thread = new Thread(r, "BufferCacheManager cleanerThread");
            thread.setPriority(Thread.MIN_PRIORITY);
            return thread;
        }
    }
}
