/*
 * Mobicents Media Gateway
 *
 * The source code contained in this file is in in the public domain.
 * It can be used in any project or product without prior permission,
 * license or royalty payments. There is  NO WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR STATUTORY, INCLUDING, WITHOUT LIMITATION,
 * THE IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * AND DATA ACCURACY.  We do not warrant or make any representations
 * regarding the use of the software or the  results thereof, including
 * but not limited to the correctness, accuracy, reliability or
 * usefulness of the software.
 */
package org.mobicents.media.server.impl.rtp;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.SocketTimeoutException;
import java.util.concurrent.ScheduledFuture;
import org.mobicents.media.Buffer;
import org.mobicents.media.BufferFactory;
import org.mobicents.media.Format;
import org.mobicents.media.server.impl.AbstractSource;
import org.mobicents.media.server.impl.clock.Timer;

/**
 * 
 * @author Oleg Kulikov
 */
public class ReceiveStream extends AbstractSource implements Runnable {

    /**
     * 
     */
    private static final long serialVersionUID = -2277812497480986797L;
    private int period;
    private JitterBuffer jitterBuffer;
    
    private transient Timer timer = new Timer();
    private transient ScheduledFuture worker;
    
    private volatile boolean started = false;
    private Buffer frame;
    protected Format[] formats;
    private RtpSocketImpl rtpSocket;
    private BufferFactory bufferFactory = null;
    private RtpHeader header = new RtpHeader();
    
    private int pt = -1;
    private Format fmt = null;

    private transient Receiver receiver;
    private Thread receiverThread;
    
    //private Receiver 
    /** Creates a new instance of ReceiveStream */
    public ReceiveStream(RtpSocket rtpSocket, int period, int jitter) {
        super("ReceiveStream");
        bufferFactory = new BufferFactory(10, "ReceiveStream");
        this.rtpSocket = (RtpSocketImpl) rtpSocket;
        this.period = period;
        jitterBuffer = new JitterBuffer(jitter, period);
        receiver = new Receiver();
    }

    private void push(Buffer buffer) {
        jitterBuffer.write(buffer);
    }

    public synchronized void stop() {
        if (started) {
            started = false;
            worker.cancel(true);
        }
    }

    public synchronized void start() {
        if (!started) {
            started = true;
            receiverThread = new Thread(this);
            receiverThread.setName("RTP Receiver Stream");
            receiverThread.start();
            jitterBuffer.reset();
            worker = timer.synchronize(receiver);
        }
    }

    public Format[] getFormats() {
        Format[] fmts = new Format[rtpSocket.getRtpMap().size()];
        rtpSocket.getRtpMap().values().toArray(fmts);
        return fmts;
    }


    private class Receiver implements Runnable {

        public void run() {
            frame = jitterBuffer.read();

            if (frame == null) {
                return;
            }
            if (sink == null) {
                return;
            }

            // The sink for ReceiveStream is Processor.Input
            try {
                sink.receive(frame);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public void run() {
        byte[] buff = new byte[172];
        DatagramPacket udpPacket = new DatagramPacket(buff, buff.length);
        while (started) {
            try {
                rtpSocket.receivePacket(udpPacket);
            } catch (SocketTimeoutException e) {
                continue;
            } catch (IOException e) {
                continue;
            }


            header.init(buff);

            // change format if payload type is changed
            if (pt != header.getPayloadType()) {
                pt = header.getPayloadType();
                fmt = rtpSocket.getRtpMap().get(header.getPayloadType());
            }
            if (fmt == null) {
                continue;
            }
            Buffer buffer = bufferFactory.allocate();
            buffer.setLength(udpPacket.getLength() - 12);
            buffer.setFormat(fmt);

            // buffer.setFormat(rtpSocket.getRtpMap().get(header.getPayloadType()));
            System.arraycopy(buff, 12, (byte[]) buffer.getData(), 0, buffer.getLength());

            try {
                push(buffer);
            } catch (Exception e) {
                e.printStackTrace();
                continue;
            }
        }
    }
        
}
