/*
 * Decompiled with CFR 0.152.
 */
package org.monarchinitiative.phenol.annotations.scoredist;

import java.io.IOException;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import org.monarchinitiative.phenol.annotations.scoredist.ScoreDistributionWriter;
import org.monarchinitiative.phenol.base.PhenolException;
import org.monarchinitiative.phenol.io.utils.ObjHexStringConverter;
import org.monarchinitiative.phenol.ontology.data.TermId;
import org.monarchinitiative.phenol.ontology.scoredist.ObjectScoreDistribution;
import org.monarchinitiative.phenol.ontology.scoredist.ScoreDistribution;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class H2ScoreDistributionWriter
implements ScoreDistributionWriter {
    private static final Logger LOGGER = LoggerFactory.getLogger(H2ScoreDistributionWriter.class);
    private final String pathDb;
    private final String tableName;
    private final Connection conn;
    private static final String H2_DROP_TABLE_STATEMENT = "DROP TABLE %s";
    private static final String[] H2_CREATE_TABLE_STATEMENTS = new String[]{"CREATE TABLE %s (num_terms INT, object_id VARCHAR, sample_size INT, scores OTHER, p_values OTHER)", "CREATE INDEX ON %s (num_terms)", "CREATE UNIQUE INDEX ON %s (num_terms, object_id)"};
    private static final String H2_INSERT_STATEMENT = "INSERT INTO %s (num_terms, object_id, sample_size, scores, p_values) VALUES (?, ?, ?, ?, ?)";

    public H2ScoreDistributionWriter(String pathDb, String dataTableName, boolean resetTableIfExists) throws PhenolException {
        this.pathDb = pathDb;
        this.tableName = dataTableName;
        this.conn = this.openConnection(resetTableIfExists);
    }

    private Connection openConnection(boolean resetTableIfExists) throws PhenolException {
        boolean tableExists;
        Connection resultConn;
        LOGGER.info("Opening connection to H2 database file at {} and configuring...", (Object)this.pathDb);
        try {
            Class.forName("org.h2.Driver");
            resultConn = DriverManager.getConnection("jdbc:h2:" + this.pathDb, "", "");
        }
        catch (ClassNotFoundException e) {
            throw new PhenolException("H2 driver class could not be found", (Throwable)e);
        }
        catch (SQLException e) {
            throw new PhenolException("Could not open database at " + this.pathDb, (Throwable)e);
        }
        try (ResultSet rs = resultConn.getMetaData().getTables(null, null, this.tableName.toUpperCase(), new String[]{"TABLE"});){
            tableExists = rs.next();
        }
        catch (SQLException e) {
            throw new PhenolException("Checking for table of name " + this.tableName + " failed", (Throwable)e);
        }
        if (tableExists) {
            if (!resetTableIfExists) {
                throw new PhenolException("Table exists but not allowed to reset!");
            }
            LOGGER.info("Table {} exists, dropping.", (Object)this.tableName);
            String sqlStmt = String.format(H2_DROP_TABLE_STATEMENT, this.tableName);
            LOGGER.info("Executing SQL statement: {}", (Object)sqlStmt);
            try (PreparedStatement stmt = resultConn.prepareStatement(sqlStmt);){
                stmt.executeUpdate();
            }
            catch (SQLException e) {
                throw new PhenolException("Could not drop table with statement " + sqlStmt, (Throwable)e);
            }
        }
        LOGGER.info("Creating table {} and indices...", (Object)this.tableName);
        for (String sql : H2_CREATE_TABLE_STATEMENTS) {
            String sqlStmt = String.format(sql, this.tableName);
            LOGGER.info("Executing SQL statement: {}", (Object)sqlStmt);
            try (PreparedStatement stmt = resultConn.prepareStatement(sqlStmt);){
                stmt.executeUpdate();
            }
            catch (SQLException e) {
                throw new PhenolException("Could execute table creation/index statement: " + sqlStmt, (Throwable)e);
            }
        }
        LOGGER.info("Successfully connected and configured H2 database file.");
        return resultConn;
    }

    @Override
    public void close() throws IOException {
        try {
            this.conn.close();
        }
        catch (SQLException e) {
            throw new IOException("Problem closing connection to database", e);
        }
    }

    @Override
    public void write(int numTerms, ScoreDistribution scoreDistribution, int resolution) throws PhenolException {
        for (TermId objectId : scoreDistribution.getObjectIds()) {
            ObjectScoreDistribution dist = scoreDistribution.getObjectScoreDistribution(objectId);
            this.writeObjectScoreDistribution(numTerms, dist, resolution);
        }
    }

    private <T extends Serializable> void writeObjectScoreDistribution(int numTerms, ObjectScoreDistribution dist, int resolution) throws PhenolException {
        int i;
        double[] pValues;
        double[] scores;
        List observedScores = dist.observedScores();
        if (resolution != 0) {
            scores = new double[resolution + 1];
            pValues = new double[resolution + 1];
            for (i = 0; i <= resolution; ++i) {
                double score;
                double pos = ((double)observedScores.size() - 1.0) / (double)resolution * (double)i;
                int left = Math.max(0, (int)Math.floor(pos));
                int right = Math.min(observedScores.size() - 1, (int)Math.ceil(pos));
                double dx = (double)right - pos;
                scores[i] = score = (Double)observedScores.get(left) + (1.0 - dx) * ((Double)observedScores.get(right) - (Double)observedScores.get(left));
                pValues[i] = dist.estimatePValue(score);
            }
        } else {
            scores = new double[dist.getCumulativeFrequencies().size()];
            pValues = new double[dist.getCumulativeFrequencies().size()];
            i = 0;
            for (Map.Entry e : dist.getCumulativeFrequencies().entrySet()) {
                scores[i] = (Double)e.getKey();
                pValues[i] = (Double)e.getValue();
            }
        }
        String sqlStmt = String.format(H2_INSERT_STATEMENT, this.tableName);
        try (PreparedStatement stmt = this.conn.prepareStatement(sqlStmt);){
            stmt.setInt(1, numTerms);
            try {
                stmt.setString(2, ObjHexStringConverter.object2hex((Serializable)dist.getObjectId()));
            }
            catch (IOException e) {
                throw new SQLException();
            }
            stmt.setInt(3, dist.getSampleSize());
            stmt.setObject(4, scores);
            stmt.setObject(5, pValues);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            throw new PhenolException("Problem with inserting into score distribution table", (Throwable)e);
        }
    }
}

