/*
 * Decompiled with CFR 0.152.
 */
package org.monarchinitiative.sgenes.jannovar;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSortedMap;
import de.charite.compbio.jannovar.data.JannovarData;
import de.charite.compbio.jannovar.data.ReferenceDictionary;
import de.charite.compbio.jannovar.reference.GenomeInterval;
import de.charite.compbio.jannovar.reference.Strand;
import de.charite.compbio.jannovar.reference.TranscriptModel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.monarchinitiative.sgenes.model.Gene;
import org.monarchinitiative.sgenes.model.GeneIdentifier;
import org.monarchinitiative.sgenes.model.Transcript;
import org.monarchinitiative.sgenes.model.TranscriptEvidence;
import org.monarchinitiative.sgenes.model.TranscriptIdentifier;
import org.monarchinitiative.sgenes.model.TranscriptMetadata;
import org.monarchinitiative.svart.Contig;
import org.monarchinitiative.svart.CoordinateSystem;
import org.monarchinitiative.svart.Coordinates;
import org.monarchinitiative.svart.GenomicRegion;
import org.monarchinitiative.svart.assembly.GenomicAssembly;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class JannovarIterator
implements Iterator<Gene> {
    private static final Logger LOGGER = LoggerFactory.getLogger(JannovarIterator.class);
    private static final CoordinateSystem COORDINATE_SYSTEM = CoordinateSystem.zeroBased();
    private static final Pattern REFSEQ_TX = Pattern.compile("(?<prefix>[NX][MR])_(?<value>\\d+)\\.(?<version>\\d+)");
    private static final Pattern ENSEMBL_TX = Pattern.compile("(?<prefix>ENST)(?<value>\\d{11})\\.(?<version>\\d+)");
    private static final Pattern UCSC_TX = Pattern.compile("(?<prefix>uc)(?<value>\\d{3})(?<suffix>\\w{3})\\.(?<version>\\d+)");
    private final ReferenceDictionary rd;
    private final Iterator<Map.Entry<String, Collection<TranscriptModel>>> iterator;
    private final GenomicAssembly assembly;
    private final Queue<Gene> genes = new LinkedList<Gene>();

    JannovarIterator(JannovarData jannovarData, GenomicAssembly assembly) {
        this.assembly = assembly;
        this.rd = jannovarData.getRefDict();
        ImmutableMap geneMap = jannovarData.getTmByGeneSymbol().asMap();
        this.iterator = geneMap.entrySet().iterator();
        this.readNextGenes();
    }

    private static org.monarchinitiative.svart.Strand parseStrand(Strand strand) {
        switch (strand) {
            case FWD: {
                return org.monarchinitiative.svart.Strand.POSITIVE;
            }
            case REV: {
                return org.monarchinitiative.svart.Strand.NEGATIVE;
            }
        }
        throw new IllegalArgumentException(String.format("Unknown strand %s", strand));
    }

    private static Optional<GenomicRegion> parseLocation(Contig contig, Collection<TranscriptModel> transcripts) {
        int start = contig.length();
        int end = 0;
        if (transcripts.isEmpty()) {
            return Optional.empty();
        }
        org.monarchinitiative.svart.Strand st = null;
        for (TranscriptModel tx : transcripts) {
            GenomeInterval txRegion = tx.getTXRegion();
            start = Math.min(start, txRegion.getBeginPos());
            end = Math.max(end, txRegion.getEndPos());
            org.monarchinitiative.svart.Strand currentStrand = JannovarIterator.parseStrand(tx.getStrand());
            if (st == null) {
                st = currentStrand;
                continue;
            }
            if (st.equals((Object)currentStrand)) continue;
            return Optional.empty();
        }
        return Optional.of(GenomicRegion.of((Contig)contig, st, (CoordinateSystem)COORDINATE_SYSTEM, (int)start, (int)end));
    }

    private static Collection<Gene> parseNextGene(GenomicAssembly assembly, ReferenceDictionary rd, String hgvsGeneSymbol, Collection<TranscriptModel> transcripts) {
        Map<Integer, List<TranscriptModel>> txsByContig = transcripts.stream().collect(Collectors.groupingBy(TranscriptModel::getChr));
        ArrayList<Gene> genes = new ArrayList<Gene>(txsByContig.size());
        for (Map.Entry<Integer, List<TranscriptModel>> entry : txsByContig.entrySet()) {
            Integer contigId = entry.getKey();
            List<TranscriptModel> txs = entry.getValue();
            Optional<GeneIdentifier> id = JannovarIterator.createGeneIdentifier(hgvsGeneSymbol, txs);
            if (id.isEmpty()) {
                LOGGER.debug("No proper gene accession is available for gene `{}`", (Object)hgvsGeneSymbol);
                continue;
            }
            String contigName = (String)rd.getContigIDToName().get((Object)contigId);
            Contig contig = assembly.contigByName(contigName);
            if (contig.isUnknown()) {
                LOGGER.debug("Unknown contig {}({}) for gene `{}`", new Object[]{contigName, contigId, hgvsGeneSymbol});
                continue;
            }
            Optional<GenomicRegion> location = JannovarIterator.parseLocation(contig, txs);
            if (location.isEmpty()) {
                LOGGER.debug("Transcripts of {} are located on multiple strands", (Object)hgvsGeneSymbol);
                continue;
            }
            List<? extends Transcript> parsedTxs = JannovarIterator.parseTranscripts(contig, txs);
            Gene gene = Gene.of((GeneIdentifier)id.get(), (GenomicRegion)location.get(), parsedTxs);
            genes.add(gene);
        }
        return genes;
    }

    private static Optional<GeneIdentifier> createGeneIdentifier(String symbol, Collection<TranscriptModel> transcripts) {
        String geneId = null;
        String hgncId = null;
        String ncbiGeneId = null;
        for (TranscriptModel tx : transcripts) {
            ImmutableSortedMap altGeneIDs = tx.getAltGeneIDs();
            if (geneId == null && (geneId = (String)altGeneIDs.get("ENSEMBL_GENE_ID")) == null) {
                geneId = (String)altGeneIDs.get("HGNC_ID");
            }
            if (hgncId == null) {
                hgncId = (String)altGeneIDs.get("HGNC_ID");
            }
            if (ncbiGeneId == null) {
                String entrezId = (String)altGeneIDs.get("ENTREZ_ID");
                String string = ncbiGeneId = entrezId == null ? null : "NCBIGene:" + entrezId;
            }
            if (geneId == null || hgncId == null || ncbiGeneId == null) continue;
            return Optional.of(GeneIdentifier.of((String)geneId, (String)symbol, (String)hgncId, (String)ncbiGeneId));
        }
        return geneId == null ? Optional.empty() : Optional.of(GeneIdentifier.of(geneId, (String)symbol, hgncId, ncbiGeneId));
    }

    private static List<? extends Transcript> parseTranscripts(Contig contig, Collection<TranscriptModel> transcripts) {
        ArrayList<Transcript> txs = new ArrayList<Transcript>(transcripts.size());
        for (TranscriptModel tx : transcripts) {
            org.monarchinitiative.svart.Strand strand = JannovarIterator.parseStrand(tx.getStrand());
            TranscriptIdentifier txId = TranscriptIdentifier.of((String)tx.getAccession(), (String)tx.getAccession(), null);
            TranscriptMetadata metadata = TranscriptMetadata.of((TranscriptEvidence)JannovarIterator.parseTranscriptEvidence(tx.getAccession(), tx.getTranscriptSupportLevel()));
            if (tx.isCoding()) {
                txs.add(JannovarIterator.parseCodingTranscript(contig, strand, txId, metadata, tx));
                continue;
            }
            txs.add(JannovarIterator.parseNoncodingTranscript(contig, strand, txId, metadata, tx));
        }
        return Collections.unmodifiableList(txs);
    }

    private static TranscriptEvidence parseTranscriptEvidence(String accession, int transcriptSupportLevel) {
        Matcher ensemblMatcher;
        Matcher refseqMatcher = REFSEQ_TX.matcher(accession);
        if (refseqMatcher.matches()) {
            String prefix = refseqMatcher.group("prefix");
            switch (prefix.toUpperCase()) {
                case "NM": 
                case "NR": {
                    return TranscriptEvidence.KNOWN;
                }
                case "XM": 
                case "XR": {
                    return TranscriptEvidence.MODEL;
                }
            }
        }
        if ((ensemblMatcher = ENSEMBL_TX.matcher(accession)).matches()) {
            return JannovarIterator.parseTranscriptSupportLevel(transcriptSupportLevel);
        }
        Matcher ucscMatcher = UCSC_TX.matcher(accession);
        if (ucscMatcher.matches()) {
            return JannovarIterator.parseTranscriptSupportLevel(transcriptSupportLevel);
        }
        return null;
    }

    private static TranscriptEvidence parseTranscriptSupportLevel(int transcriptSupportLevel) {
        switch (transcriptSupportLevel) {
            case -1: {
                return null;
            }
            case 6: {
                return TranscriptEvidence.CANONICAL;
            }
            case 8: {
                return null;
            }
        }
        LOGGER.warn("Unexpected transcript support level {}", (Object)transcriptSupportLevel);
        return null;
    }

    private static Transcript parseCodingTranscript(Contig contig, org.monarchinitiative.svart.Strand strand, TranscriptIdentifier txId, TranscriptMetadata metadata, TranscriptModel tx) {
        GenomeInterval txRegion = tx.getTXRegion();
        GenomicRegion location = GenomicRegion.of((Contig)contig, (org.monarchinitiative.svart.Strand)strand, (CoordinateSystem)COORDINATE_SYSTEM, (int)txRegion.getBeginPos(), (int)txRegion.getEndPos());
        List<Coordinates> exons = JannovarIterator.remapExons((List<GenomeInterval>)tx.getExonRegions());
        GenomeInterval cds = tx.getCDSRegion();
        Coordinates cdsCoordinates = Coordinates.of((CoordinateSystem)COORDINATE_SYSTEM, (int)cds.getBeginPos(), (int)cds.getEndPos());
        return Transcript.of((TranscriptIdentifier)txId, (GenomicRegion)location, exons, (Coordinates)cdsCoordinates, (TranscriptMetadata)metadata);
    }

    private static Transcript parseNoncodingTranscript(Contig contig, org.monarchinitiative.svart.Strand strand, TranscriptIdentifier txId, TranscriptMetadata metadata, TranscriptModel tx) {
        GenomeInterval txRegion = tx.getTXRegion();
        GenomicRegion location = GenomicRegion.of((Contig)contig, (org.monarchinitiative.svart.Strand)strand, (CoordinateSystem)COORDINATE_SYSTEM, (int)txRegion.getBeginPos(), (int)txRegion.getEndPos());
        List<Coordinates> exons = JannovarIterator.remapExons((List<GenomeInterval>)tx.getExonRegions());
        return Transcript.of((TranscriptIdentifier)txId, (GenomicRegion)location, exons, null, (TranscriptMetadata)metadata);
    }

    private static List<Coordinates> remapExons(List<GenomeInterval> exonRegions) {
        ArrayList<Coordinates> exons = new ArrayList<Coordinates>(exonRegions.size());
        for (GenomeInterval exon : exonRegions) {
            exons.add(Coordinates.of((CoordinateSystem)COORDINATE_SYSTEM, (int)exon.getBeginPos(), (int)exon.getEndPos()));
        }
        return exons;
    }

    private void readNextGenes() {
        while (this.iterator.hasNext() && this.genes.isEmpty()) {
            Map.Entry<String, Collection<TranscriptModel>> entry = this.iterator.next();
            this.genes.addAll(JannovarIterator.parseNextGene(this.assembly, this.rd, entry.getKey(), entry.getValue()));
        }
    }

    @Override
    public boolean hasNext() {
        return !this.genes.isEmpty();
    }

    @Override
    public Gene next() {
        Gene current = this.genes.poll();
        if (this.genes.isEmpty()) {
            this.readNextGenes();
        }
        return current;
    }
}

