/*
 * Copyright 2008-present MongoDB, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mongodb.kafka.connect.util;

import static java.lang.String.format;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;

import org.apache.kafka.common.config.AbstractConfig;
import org.apache.kafka.common.config.Config;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.ConfigValue;

import org.bson.Document;

import com.mongodb.MongoDriverInformation;
import com.mongodb.client.model.Collation;
import com.mongodb.client.model.CollationAlternate;
import com.mongodb.client.model.CollationCaseFirst;
import com.mongodb.client.model.CollationMaxVariable;
import com.mongodb.client.model.CollationStrength;
import com.mongodb.client.model.changestream.FullDocument;
import com.mongodb.client.model.changestream.FullDocumentBeforeChange;

import com.mongodb.kafka.connect.Versions;

public final class ConfigHelper {

  private ConfigHelper() {}

  public static Optional<List<Document>> jsonArrayFromString(final String jsonArray) {
    return jsonArrayFromString(jsonArray, null);
  }

  public static Optional<Document> documentFromString(final String jsonDocument) {
    return documentFromString(jsonDocument, null);
  }

  public static Optional<Document> documentFromString(
      final String jsonDocument, final ConfigException originalError) {
    if (jsonDocument.isEmpty()) {
      return Optional.empty();
    } else {
      try {
        return Optional.of(Document.parse(jsonDocument));
      } catch (Exception e) {
        if (originalError != null) {
          throw originalError;
        } else {
          return documentFromString(
              jsonDocument.replace("\\", "\\\\"),
              new ConfigException("Not a valid JSON document", e));
        }
      }
    }
  }

  private static Optional<List<Document>> jsonArrayFromString(
      final String jsonArray, final ConfigException originalError) {
    if (jsonArray.isEmpty()) {
      return Optional.empty();
    } else {
      try {
        List<Document> s =
            Document.parse(format("{s: %s}", jsonArray)).getList("s", Document.class);
        return s.isEmpty() ? Optional.empty() : Optional.of(s);
      } catch (Exception e) {
        if (originalError != null) {
          throw originalError;
        } else {
          return jsonArrayFromString(
              jsonArray.replace("\\", "\\\\"), new ConfigException("Not a valid JSON array", e));
        }
      }
    }
  }

  public static Optional<FullDocumentBeforeChange> fullDocumentBeforeChangeFromString(
      final String s) {
    if (s.isEmpty()) {
      return Optional.empty();
    } else {
      return Optional.of(FullDocumentBeforeChange.fromString(s));
    }
  }

  public static Optional<FullDocument> fullDocumentFromString(final String fullDocument) {
    if (fullDocument.isEmpty()) {
      return Optional.empty();
    } else {
      return Optional.of(FullDocument.fromString(fullDocument));
    }
  }

  public static Optional<Collation> collationFromJson(final String collationString) {
    if (collationString.isEmpty()) {
      return Optional.empty();
    }
    Collation.Builder builder = Collation.builder();
    Document collationDoc = Document.parse(collationString);
    if (collationDoc.containsKey("locale")) {
      builder.locale(collationDoc.getString("locale"));
    }
    if (collationDoc.containsKey("caseLevel")) {
      builder.caseLevel(collationDoc.getBoolean("caseLevel"));
    }
    if (collationDoc.containsKey("caseFirst")) {
      builder.collationCaseFirst(
          CollationCaseFirst.fromString(collationDoc.getString("caseFirst")));
    }
    if (collationDoc.containsKey("strength")) {
      builder.collationStrength(CollationStrength.fromInt(collationDoc.getInteger("strength")));
    }
    if (collationDoc.containsKey("numericOrdering")) {
      builder.numericOrdering(collationDoc.getBoolean("numericOrdering"));
    }
    if (collationDoc.containsKey("alternate")) {
      builder.collationAlternate(
          CollationAlternate.fromString(collationDoc.getString("alternate")));
    }
    if (collationDoc.containsKey("maxVariable")) {
      builder.collationMaxVariable(
          CollationMaxVariable.fromString(collationDoc.getString("maxVariable")));
    }
    if (collationDoc.containsKey("normalization")) {
      builder.normalization(collationDoc.getBoolean("normalization"));
    }
    if (collationDoc.containsKey("backwards")) {
      builder.backwards(collationDoc.getBoolean("backwards"));
    }
    return Optional.of(builder.build());
  }

  public static MongoDriverInformation getMongoDriverInformation(
      final String type, final String provider) {
    String name = Versions.NAME + "|" + type;
    if (!provider.isEmpty()) {
      name = name + "|" + provider;
    }
    return MongoDriverInformation.builder()
        .driverName(name)
        .driverVersion(Versions.VERSION)
        .build();
  }

  public static String getOverrideOrDefault(
      final AbstractConfig config, final String overrideConfig, final String defaultConfig) {
    String stringConfig = config.getString(overrideConfig);
    if (stringConfig.isEmpty()) {
      stringConfig = config.getString(defaultConfig);
    }
    return stringConfig;
  }

  public static <C extends AbstractConfig, T> T getOverrideOrFallback(
      final C config,
      final BiFunction<? super C, String, T> getter,
      final String overrideProperty,
      final String defaultProperty) {
    String propertyToRead =
        config.originals().containsKey(overrideProperty) ? overrideProperty : defaultProperty;
    return getter.apply(config, propertyToRead);
  }

  public static Optional<ConfigValue> getConfigByName(final Config config, final String name) {
    return config.configValues().stream().filter(cv -> cv.name().equals(name)).findFirst();
  }

  public static Optional<ConfigValue> getConfigByNameWithoutErrors(
      final Config config, final String name) {
    Optional<ConfigValue> configByName = getConfigByName(config, name);
    if (configByName.isPresent() && configByName.get().errorMessages().isEmpty()) {
      return configByName;
    }
    return Optional.empty();
  }

  public static Config evaluateConfigValues(
      final Config rawConfig, final AbstractConfig resolvedConfig) {
    final Map<String, ?> evalValues = resolvedConfig.values();
    rawConfig
        .configValues()
        .forEach(
            configValue -> {
              Object ev = evalValues.get(configValue.name());
              if (ev != null) {
                configValue.value(ev);
              }
            });
    return rawConfig;
  }
}
