/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.kafka.connect.source;

import com.mongodb.MongoNamespace;
import com.mongodb.client.MongoClient;
import com.mongodb.kafka.connect.source.MongoSourceConfig;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.kafka.connect.errors.ConnectException;
import org.bson.BsonBinaryReader;
import org.bson.BsonDocument;
import org.bson.BsonType;
import org.bson.RawBsonDocument;
import org.bson.conversions.Bson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class MongoCopyDataManager
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(MongoCopyDataManager.class);
    private static final String NAMESPACE_FIELD = "ns";
    static final String ALT_NAMESPACE_FIELD = "__";
    private static final byte[] NAMESPACE_BYTES = "ns".getBytes(StandardCharsets.UTF_8);
    private static final String PIPELINE_TEMPLATE = String.format("{$replaceRoot: {newRoot: {_id: {_id: '$_id', copyingData: true}, operationType: 'insert', %s: {db: '%%s', coll: '%%s'}documentKey: {_id: '$_id'}, fullDocument: '$$ROOT'}}}", "ns");
    private static final BsonDocument ADD_ALT_NAMESPACE_STAGE = BsonDocument.parse((String)String.format("{'$addFields': {'%s': '$%s'}}", "__", "ns"));
    private static final BsonDocument UNSET_ORIGINAL_NAMESPACE_STAGE = BsonDocument.parse((String)String.format("{'$project': {'%s': 0}}", "ns"));
    private volatile boolean closed;
    private volatile Exception errorException;
    private final AtomicInteger namespacesToCopy;
    private final MongoSourceConfig sourceConfig;
    private final MongoClient mongoClient;
    private final ExecutorService executor;
    private final ArrayBlockingQueue<BsonDocument> queue;

    MongoCopyDataManager(MongoSourceConfig sourceConfig, MongoClient mongoClient) {
        this.sourceConfig = sourceConfig;
        this.mongoClient = mongoClient;
        List<MongoNamespace> namespaces = MongoCopyDataManager.selectNamespaces(sourceConfig, mongoClient);
        LOGGER.info("Copying existing data on the following namespaces: {}", namespaces);
        this.namespacesToCopy = new AtomicInteger(namespaces.size());
        this.queue = new ArrayBlockingQueue(sourceConfig.getInt("copy.existing.queue.size"));
        this.executor = Executors.newFixedThreadPool(Math.max(1, Math.min(namespaces.size(), sourceConfig.getInt("copy.existing.max.threads"))));
        namespaces.forEach(n -> this.executor.submit(() -> this.copyDataFrom((MongoNamespace)n)));
    }

    Optional<BsonDocument> poll() {
        if (this.errorException != null) {
            if (!this.closed) {
                this.close();
            }
            throw new ConnectException((Throwable)this.errorException);
        }
        if (this.namespacesToCopy.get() == 0) {
            this.close();
        }
        return Optional.ofNullable(this.queue.poll());
    }

    boolean isCopying() {
        return this.namespacesToCopy.get() > 0 || !this.queue.isEmpty();
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closed = true;
            LOGGER.debug("Shutting down executors");
            this.executor.shutdownNow();
        }
    }

    private void copyDataFrom(MongoNamespace namespace) {
        LOGGER.debug("Copying existing data from: {}", (Object)namespace.getFullName());
        try {
            this.mongoClient.getDatabase(namespace.getDatabaseName()).getCollection(namespace.getCollectionName(), RawBsonDocument.class).aggregate(MongoCopyDataManager.createPipeline(this.sourceConfig, namespace)).allowDiskUse(this.sourceConfig.getBoolean("copy.existing.allow.disk.use")).forEach(this::putToQueue);
            this.namespacesToCopy.decrementAndGet();
        }
        catch (Exception e) {
            this.errorException = e;
        }
    }

    private void putToQueue(RawBsonDocument bsonDocument) {
        try {
            this.queue.put((BsonDocument)MongoCopyDataManager.convertDocument(bsonDocument));
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    static List<MongoNamespace> selectNamespaces(MongoSourceConfig sourceConfig, MongoClient mongoClient) {
        String database = sourceConfig.getString("database");
        String collection = sourceConfig.getString("collection");
        String namespacesRegex = sourceConfig.getString("copy.existing.namespace.regex");
        List<Object> namespaces = database.isEmpty() ? MongoCopyDataManager.getCollections(mongoClient) : (collection.isEmpty() ? MongoCopyDataManager.getCollections(mongoClient, database) : Collections.singletonList(MongoCopyDataManager.createNamespace(database, collection)));
        if (!namespacesRegex.isEmpty()) {
            Predicate<String> predicate = Pattern.compile(namespacesRegex).asPredicate();
            namespaces = namespaces.stream().filter(n -> predicate.test(n.getFullName())).collect(Collectors.toList());
        }
        return namespaces;
    }

    static List<Bson> createPipeline(MongoSourceConfig cfg, MongoNamespace namespace) {
        ArrayList<Bson> pipeline = new ArrayList<Bson>();
        cfg.getPipeline("copy.existing.pipeline").map(pipeline::addAll);
        pipeline.add((Bson)BsonDocument.parse((String)String.format(PIPELINE_TEMPLATE, namespace.getDatabaseName(), namespace.getCollectionName())));
        cfg.getPipeline().map(pipeline::addAll);
        pipeline.add((Bson)ADD_ALT_NAMESPACE_STAGE);
        pipeline.add((Bson)UNSET_ORIGINAL_NAMESPACE_STAGE);
        return pipeline;
    }

    static RawBsonDocument convertDocument(RawBsonDocument original) {
        ByteBuffer sourceBuffer = original.getByteBuffer().asNIO();
        BsonBinaryReader reader = new BsonBinaryReader(sourceBuffer);
        int currentPosition = 0;
        reader.readStartDocument();
        while (reader.readBsonType() != BsonType.END_OF_DOCUMENT) {
            if (reader.readName().equals(ALT_NAMESPACE_FIELD)) {
                ++currentPosition;
                byte[] sourceBytes = sourceBuffer.array();
                for (byte namespaceByte : NAMESPACE_BYTES) {
                    sourceBytes[currentPosition++] = namespaceByte;
                }
                return original;
            }
            reader.skipValue();
            currentPosition = reader.getBsonInput().getPosition();
        }
        return original;
    }

    private static List<MongoNamespace> getCollections(MongoClient mongoClient) {
        return ((ArrayList)mongoClient.listDatabaseNames().into(new ArrayList())).stream().filter(s -> !s.startsWith("admin") && !s.startsWith("config") && !s.startsWith("local")).map(d -> MongoCopyDataManager.getCollections(mongoClient, d)).flatMap(Collection::stream).collect(Collectors.toList());
    }

    private static List<MongoNamespace> getCollections(MongoClient mongoClient, String database) {
        return ((ArrayList)mongoClient.getDatabase(database).listCollectionNames().into(new ArrayList())).stream().filter(s -> !s.startsWith("system.")).map(c -> MongoCopyDataManager.createNamespace(database, c)).collect(Collectors.toList());
    }

    private static MongoNamespace createNamespace(String database, String collection) {
        return new MongoNamespace(database, collection);
    }
}

