/*
 * Copyright 2008-present MongoDB, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.mongodb.scala

import java.util.concurrent.TimeUnit

import scala.concurrent.duration.Duration
import com.mongodb.async.client.ListDatabasesIterable
import org.mongodb.scala.bson.conversions.Bson
import org.mongodb.scala.internal.ObservableHelper._

/**
 * Observable interface for ListDatabases.
 *
 * @param wrapped the underlying java ListDatabasesObservable
 * @tparam TResult The type of the result.
 * @since 1.0
 */
case class ListDatabasesObservable[TResult](wrapped: ListDatabasesIterable[TResult]) extends Observable[TResult] {

  /**
   * Sets the maximum execution time on the server for this operation.
   *
   * [[http://docs.mongodb.org/manual/reference/operator/meta/maxTimeMS/ Max Time]]
   * @param duration the duration
   * @return this
   */
  def maxTime(duration: Duration): ListDatabasesObservable[TResult] = {
    wrapped.maxTime(duration.toMillis, TimeUnit.MILLISECONDS)
    this
  }

  /**
   * Sets the query filter to apply to the returned database names.
   *
   * @param filter the filter, which may be null.
   * @return this
   * @since 2.2
   * @note Requires MongoDB 3.4.2 or greater
   */
  def filter(filter: Bson): ListDatabasesObservable[TResult] = {
    wrapped.filter(filter)
    this
  }

  /**
   * Sets the nameOnly flag that indicates whether the command should return just the database names or return the database names and
   * size information.
   *
   * @param nameOnly the nameOnly flag, which may be null
   * @return this
   * @since 2.2
   * @note Requires MongoDB 3.4.3 or greater
   */
  def nameOnly(nameOnly: Boolean): ListDatabasesObservable[TResult] = {
    wrapped.nameOnly(nameOnly)
    this
  }

  /**
   * Sets the number of documents to return per batch.
   *
   * @param batchSize the batch size
   * @return this
   * @since 2.7
   */
  def batchSize(batchSize: Int): ListDatabasesObservable[TResult] = {
    wrapped.batchSize(batchSize)
    this
  }

  override def subscribe(observer: Observer[_ >: TResult]): Unit = observe(wrapped).subscribe(observer)
}
