/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.appserv.ha.spi;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Registry for storing BackingStore implementation. This is a singleton
 *  and contains a mapping between persistence-type and BackingStoreFactory.
 *  An instance of this class can be obtained using getInstance() method.
 *  
 * Typically, a store implementation for a particular persistencee-type
 *  will have a LifeCycleModule that creates and registers a factory 
 *  with BackingStoreRegistry using registerFactory()
 * 
 */
public class BackingStoreRegistry {
    
    private static BackingStoreRegistry _instance
        = new BackingStoreRegistry();
    
    private Map<String, String> factories
        = new ConcurrentHashMap<String, String>();
    
    //Prvate constructor
    private BackingStoreRegistry() {
    }
    
    /**
     * Return the (singleton) instance of BackingStoreRegistry
     * @return BackingStoreRegistry
     */
    public static BackingStoreRegistry getInstance() {
        return _instance;
    }
    
    /**
     * Register a BackingStoreFactory class name for the given persistence type. 
     *  This method will be called from the LifeCycleModule implementation
     *  that wishes to register a BackingStoreFactory for a given persistence-type.
     *  
     * @param type The persistence-type
     * @param factory The factory for this type
     * @throws DuplicateFactoryRegistrationException If this type is already registered
     */
    public void registerFactory(String type, String factoryClassName)
        throws DuplicateFactoryRegistrationException {
        if (factories.get(type) != null) {
            throw new DuplicateFactoryRegistrationException(
                    "Duplicate factory class (" + factoryClassName
                    + ") for type: " + type);
        } else {
            factories.put(type, factoryClassName);
        }
    }
    
    /**
     * Unregister the factory
     * @param type the type
     */
    public void removeFactory(String type) {
        factories.remove(type);
    }
    
    /**
     * Get the registered factory for the type specified in the parameter
     * @param type the type
     * @return The BackingStoreFactory class name or null
     */
    public String getFactoryClassName(String type) {
        return factories.get(type);
    }
    
    /**
     * Get a list of all registered store type
     * @return Collection<String> where each entry in the collection is the type
     */
    public Collection<String> getRegisteredTypes() {
        return factories.keySet();
    }
    
}
