/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.appserv.ha.spi;

/**
 * An object that stores the given data against the key.
 * 
 * <p>This interface defines the set of operation that a container could perform on
 * this store.</p>
 * 
 * <p>The store implementation must be thread safe</p>
 */
public interface BackingStore {

    /**
     * Load and return the MetaData for the given id
     * 
     * @param id the id whose value must be returned
     * @return the value if this store contains it or null. The implementation 
     *  must return the exact same type as that was passed to it in the save method.
     *
     * @throws NullPointerException if the key is null
     * @throws BackingStoreException if the underlying store 
     *              implementation encounters any exception
     */
    public MetaData load(String id)
        throws BackingStoreException;

    /**
     * Save the value whose Key is id. The value could be an instance of 
     *  WebMetaData OR SimpleMetaData OR CompositeMetaData
     * 
     * @param id the id
     * @param value The MetaData
     * @throws BackingStoreException if the underlying store 
     *              implementation encounters any exception
     */
    public void save(String id, MetaData value)
        throws BackingStoreException;

    /**
     * Update the last access time for this id.
     * 
     * @param id the id for the MetaData
     * @param time the time at which this data was last accessed
     * @throws BackingStoreException if the underlying store 
     *              implementation encounters any exception
     */
    public void updateLastAccessTime(String id, long time)
            throws BackingStoreException;

    /**
     * Remove the MetaData for the id. After this call, any call to
     *  load(id) <b>must</b> return null
     * 
     * @param id the id
     * @throws BackingStoreException if the underlying store 
     *              implementation encounters any exception
     */
    public void remove(String id)
        throws BackingStoreException;

    /**
     * Remove all instances that are idle. The implementation of this 
     * method must use MetaData.getLastAccessTime() and
     * MetaData.getMaxInactiveInterval() to determine if a particular 
     * MetaData is idle or not. A MetaData is considered idle if the 
     * difference between System.currentTimeMillis() and 
     * MetaData.getLastAccessTime() is greater than or equal to 
     * MetaData.getMaxInactiveInterval
     * 
     * @return The number of (idle) sessions removed.
     * @throws BackingStoreException if the underlying store 
     *              implementation encounters any exception
     */
    public int removeExpired()
        throws BackingStoreException;

    /**
     * Get the current size of the store
     * 
     * @return the number of entries in the store
     * @throws BackingStoreException if the underlying store 
     *              implementation encounters any exception
     */
    public int size()
        throws BackingStoreException;

    /**
     * Called when the store is no longer needed. Must clean up and 
     * close any opened resources.
     */
    public void destroy()
        throws BackingStoreException;

}
