package org.mulesoft.als.actions.hover

import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.{AmfElement, AmfObject, DataNode}
import amf.core.client.scala.vocabulary.ValueType
import amf.core.internal.metamodel.{Field, Obj}
import amf.core.internal.parser.domain.FieldEntry
import amf.shapes.client.scala.model.document.JsonLDInstanceDocument
import org.mulesoft.als.common.ASTElementWrapper.AlsPositionRange
import org.mulesoft.als.common.ObjectInTree
import org.mulesoft.als.common.cache.{ASTPartBranchCached, ObjectInTreeCached}
import org.mulesoft.als.common.dtoTypes.{Position, PositionRange}
import org.mulesoft.als.common.finder.StrictObjectStack
import org.mulesoft.als.common.finder.jsonschema.JsonSchemaDefinitionGuesser
import org.mulesoft.als.convert.LspRangeConverter
import org.mulesoft.amfintegration.AmfImplicits._
import org.mulesoft.amfintegration.amfconfiguration.DocumentDefinitionImplicits.DocumentDefinitionImplicits
import org.mulesoft.amfintegration.amfconfiguration.{DocumentDefinition, JsonSchemaDocumentDefinition}
import org.mulesoft.amfintegration.vocabularies.AmlCoreVocabulary
import org.mulesoft.amfintegration.vocabularies.integration.VocabularyProvider
import org.mulesoft.amfintegration.vocabularies.propertyterms.NamePropertyTerm
import org.mulesoft.common.client.lexical.{PositionRange => AmfPositionRange}
import org.mulesoft.lsp.feature.hover.Hover
import org.yaml.model.YMapEntry

case class HoverAction(
                        bu: BaseUnit,
                        tree: ObjectInTreeCached,
                        astPartBranchCached: ASTPartBranchCached,
                        dtoPosition: Position,
                        location: String,
                        provider: VocabularyProvider,
                        documentDefinition: DocumentDefinition
) {

  private val objectInTree: ObjectInTree = tree.getCachedOrNew(dtoPosition, location)

  private val yPartBranch = astPartBranchCached.getCachedOrNew(dtoPosition, location)

  def getHover: Hover =
    bu match {
      case _: JsonLDInstanceDocument =>
        getHoverForJsonSchemaDefinedDocument
      case _ =>
          getSemantic
          .map(s =>
            Hover(s._1, s._2.map(r => LspRangeConverter.toLspRange(PositionRange(r))))
          ) // if in a sequence, we could show all the semantic hierarchy?
          .getOrElse(Hover.empty)
      }


  private def getHoverForJsonSchemaDefinedDocument: Hover =
    JsonSchemaDefinitionGuesser.guessingGame(objectInTree.obj +: objectInTree.stack, dtoPosition, yPartBranch).filter(_.description.option().isDefined) match {
      case Nil => // fallback if I don't have a definition with description, I look for the closest definition with description I can find
        StrictObjectStack(bu, dtoPosition.toAmfPosition).branch.flatMap(element => element.annotations.sourceSchemaDef().map(schemaDef => (schemaDef.definition, element))).collectFirst {
          case (shape, element) if shape.description.option().isDefined =>
            Hover(Seq(shape.description.value()), element.annotations.range().map(_.toPositionRange).map(LspRangeConverter.toLspRange))
        }.getOrElse(Hover.empty)
      case entries =>
        Hover(entries.flatMap(_.description.option()).distinct, objectInTree.fieldEntry.flatMap(_.value.annotations.range()).map(_.toPositionRange).map(LspRangeConverter.toLspRange))
    }

  /** if obj is in the correct location and has a range defined, return it, if not, return more specific node from AST
    * todo: should links be already filtered out in AmfSonElementFinder?
    */
  private def mostSpecificRangeInFile(obj: AmfElement): Option[AmfPositionRange] =
    if (isInFileAndHasRange(obj))
      obj.annotations.range()
    else Option(yPartBranch.node.location.range)

  private def isInFileAndHasRange(obj: AmfElement) =
    obj.location().contains(location) && obj.annotations.range().exists(r => PositionRange(r).contains(dtoPosition))

  private def getSemantic: Option[(Seq[String], Option[AmfPositionRange])] =
    if (objectInTree.obj.isInstanceOf[DataNode]) hackFromNonDynamic()
    else if (isInDeclarationKey) fromDeclarationKey()
    else getPatchedHover.orElse(fromTree())

  private def hackFromNonDynamic(): Option[(Seq[String], Option[AmfPositionRange])] =
    objectInTree.stack.find(obj => !obj.isInstanceOf[DataNode]).flatMap(classTerm)

  def isLocal(f: FieldEntry): Boolean =
    f.value.annotations.trueLocation().contains(location) &&
      f.value.annotations.lexicalInformation().exists(_.contains(dtoPosition.toAmfPosition))

  lazy val localFieldEntry: Option[FieldEntry] =
    objectInTree.fieldEntry
      .filter(isLocal)
      .orElse(objectInTree.obj.fields.fields().find(isLocal))

  private def fromTree(): Option[(Seq[String], Option[AmfPositionRange])] =
    localFieldEntry
      .filterNot(isDeclaredName)
      .flatMap(fieldEntry)
      .orElse(objectInTree.nonVirtualObj.flatMap(classTerm))
      .orElse(classTerm(objectInTree.obj))

  private def isDeclaredName(fe: FieldEntry) =
    objectInTree.obj.annotations.isDeclared &&
      fe.field.value.iri() == AmlCoreVocabulary().base.value() + NamePropertyTerm.name

  private def fieldEntry(f: FieldEntry): Option[(Seq[String], Option[AmfPositionRange])] =
    propertyTerm(f.field).map(s =>
      (
        Seq(s),
        f.value.annotations
          .range()
          .orElse(
            mostSpecificRangeInFile(f.value.value)
          )
      )
    )

  private def propertyTerm(field: Field): Option[String] =
    provider
      .getDescription(field.value)
      .orElse({
        if (field.doc.description.nonEmpty) Some(field.doc.description) else None
      })
  // TODO: inherits from another???

  private def getSemanticForMeta(meta: Obj): Seq[String] = {
    val classSemantic = meta.`type`.flatMap(vt => provider.getDescription(vt))

    if (classSemantic.isEmpty && meta.doc.description.nonEmpty) Seq(meta.doc.description)
    else classSemantic
  }

  private def classTerm(obj: AmfObject): Option[(Seq[String], Option[AmfPositionRange])] = {
    val finalSemantics = getSemanticForMeta(obj.meta)
    if (finalSemantics.nonEmpty) Some((finalSemantics, obj.annotations.range()))
    else None
  }

  private def getDeclarationValueType(entry: YMapEntry): Option[ValueType] =
    documentDefinition.declarationsMapTerms
      .find(_._2 == entry.key.value.toString)
      .map(a => {
        ValueType(a._1)
      })

  def isInDeclarationKey: Boolean =
    bu.declarationKeys.exists(k => k.entry.key.range.contains(dtoPosition.toAmfPosition))

  private def buildDeclarationKeyUri(name: String): ValueType =
    ValueType(s"http://als.declarationKeys/#${name}DeclarationKey")

  def fromDeclarationKey(): Option[(Seq[String], Option[AmfPositionRange])] =
    bu.declarationKeys
      .find(k => k.entry.key.range.contains(dtoPosition.toAmfPosition))
      .map(key => {
        val valueType = getDeclarationValueType(key.entry)
        val description = valueType
          .map(v =>
            provider
              .getDescription(buildDeclarationKeyUri(v.name))
              .getOrElse(s"Contains declarations of reusable ${v.name} objects")
          ) // todo: extract messages to a common place (in order to easy parametrize)
          .getOrElse({
            s"Contains declarations for ${key.entry.key.value.toString}"
          })

        (Seq(description), Some(key.entry.range))
      })

  def getPatchedHover: Option[(Seq[String], Option[AmfPositionRange])] =
    patchedHover.getHover(objectInTree.obj, yPartBranch, documentDefinition)

  private lazy val patchedHover =
    PatchedHover(provider, Seq(DocumentTerms(bu, documentDefinition)))
}

sealed case class DocumentTerms(bu: BaseUnit, documentDefinition: DocumentDefinition)
