package org.mulesoft.als.actions.codeactions.plugins.declarations

import amf.core.annotations.DeclaredElement
import amf.core.errorhandling.UnhandledErrorHandler
import amf.core.model.document.Document
import amf.core.model.domain.{AmfObject, DomainElement, Linkable}
import amf.core.remote.{Mimes, Vendor}
import amf.plugins.document.vocabularies.model.document.Dialect
import amf.plugins.document.webapi.annotations.ForceEntry
import amf.plugins.document.webapi.parser.spec.common.emitters.WebApiDomainElementEmitter
import org.mulesoft.als.actions.codeactions.plugins.base.CodeActionRequestParams
import org.mulesoft.als.common.YamlUtils.isJson
import org.mulesoft.als.common.YamlWrapper.YNodeImplicits
import org.mulesoft.als.common.cache.Location
import org.mulesoft.als.common.dtoTypes.{Position, PositionRange}
import org.mulesoft.als.common.{ObjectInTree, YPartBranch}
import org.mulesoft.als.convert.LspRangeConverter
import org.mulesoft.amfintegration.AmfImplicits.{AmfAnnotationsImp, AmfObjectImp, BaseUnitImp, DialectImplicits}
import org.mulesoft.lsp.edit.TextEdit
import org.mulesoft.lsp.feature.common
import org.yaml.model.{YDocument, YMap, YMapEntry, YNode, YPart}
import org.yaml.render.{JsonRender, YamlRender}

import scala.annotation.tailrec

trait BaseDeclarableExtractors {

  protected val params: CodeActionRequestParams

  /**
    * Placeholder for the new name (key and reference)
    */
  protected val newName = "$1"

  /**
    * check if the range matches with one specific element,
    * choose the position and element which matches it
    */
  protected lazy val tree: Option[ObjectInTree] = {
    val start = params.tree.getCachedOrNew(params.range.start, params.uri)
    val end   = params.tree.getCachedOrNew(params.range.end, params.uri)
    if (start.obj == end.obj) Some(start)
    else if (start.stack.contains(end.obj)) Some(end)
    else if (end.stack.contains(start.obj)) Some(start)
    else None
  }

  /**
    *  Based on the chosen position from the range
    */
  private lazy val position: Option[Position] =
    tree.map(_.amfPosition).map(Position(_))

  /**
    * Information about the AST for the chosen position
    */
  protected lazy val yPartBranch: Option[YPartBranch] =
    position.map(params.yPartBranch.getCachedOrNew(_, params.uri))

  /**
    * If the dialect groups all declarations in a specific node (like OAS 3 `components`)
    */
  protected lazy val declarationsPath: Option[String] =
    params.dialect.documents().declarationsPath().option()

  /**
    * Selected object if there is a clean match in the range and it is a declarable
    */
  protected lazy val amfObject: Option[AmfObject] =
    //    extractable(tree.flatMap(_.realMatchedObj)) orElse {
    extractable(tree.map(_.obj)).orElse {
      extractable(tree.flatMap(_.stack.headOption))
    }

  private def extractable(maybeObject: Option[AmfObject]) =
    maybeObject
//      .filterNot(_.isAbstract)
      .filterNot(_.isInstanceOf[Document])
      .find(o => o.declarableKey(params.dialect).isDefined)

  /**
    * The original node with lexical info for the declared node
    */
  protected lazy val entryAst: Option[YPart] =
    amfObject.flatMap(_.annotations.ast()) match {
      case Some(entry: YMapEntry) => Some(entry.value)
      case c                      => c
    }

  /**
    * The original range info for the declared node
    */
  protected lazy val entryRange: Option[common.Range] =
    entryAst
      .map(_.range)
      .map(PositionRange(_))
      .map(LspRangeConverter.toLspRange)

  /**
    * The indentation for the existing node, as we already ensured it is a key, the first position gives de current indentation
    */
  protected lazy val entryIndentation: Int =
    yPartBranch.map(_.node.range.columnFrom).getOrElse(0)

  protected def positionIsExtracted: Boolean =
    entryRange
      .map(n => PositionRange(n))
      .exists(r => position.exists(r.contains))

  protected lazy val sourceName: String =
    entryAst.map(_.sourceName).getOrElse(params.uri)

  /**
    * Fallback entry, should not be necessary as the link should be rendered
    */
  protected lazy val jsonRefEntry: YNode =
    YNode(
      YMap(
        IndexedSeq(YMapEntry(YNode("$ref"), YNode(s"$newName\n"))),
        sourceName
      ))

  /**
    * Render of the link generated by the new object
    */
  protected lazy val renderLink: Option[YNode] =
    amfObject
      .collect {
        case l: Linkable =>
          l.annotations += DeclaredElement()
          val linkDe: DomainElement = l.link(newName)
          linkDe.annotations += ForceEntry() // raml explicit types
          WebApiDomainElementEmitter
            .emit(linkDe, vendor, UnhandledErrorHandler)
      }

  protected lazy val vendor: Vendor =
    params.bu.sourceVendor.getOrElse(Vendor.AML)

  /**
    * Emit a new domain element
    * @param e DomainElement to be emited
    * @return
    */
  protected def emitElement(e: DomainElement): YNode = {
    WebApiDomainElementEmitter
      .emit(e, vendor, UnhandledErrorHandler)
  } // todo: check if it is not renderable, what happens?

  protected lazy val declaredElementNode: Option[YNode] =
    amfObject
      .collect {
        case e: DomainElement => emitElement(e)
      }

  /**
    * The entry which holds the reference for the new declaration (`{"$ref": "declaration/$1"}`)
    */
  protected lazy val linkEntry: Option[TextEdit] =
    if (isJson(params.bu)) {
      entryRange.map(
        TextEdit(
          _,
          JsonRender.render(renderLink
                              .getOrElse(jsonRefEntry),
                            entryIndentation)
        ))
    } else if (params.dialect.isRamlStyle)
      entryRange.map(TextEdit(_, s" ${renderLink.map(YamlRender.render(_, 0)).getOrElse(newName)}\n"))
    else if (params.dialect.isJsonStyle)
      entryRange.map(
        TextEdit(
          _,
          s"\n${YamlRender.render(renderLink.getOrElse(jsonRefEntry),
                                  entryIndentation +
                                    params.configuration.getFormatOptionForMime(Mimes.`APPLICATION/YAML`).indentationSize)}"
        ))
    else None

  /**
    * The complete node and the entry where it belongs, contemplating the path for the declaration and existing AST
    */
  protected lazy val wrappedDeclaredEntry: Option[(YNode, Option[YMapEntry])] =
    (declaredElementNode, amfObject, params.dialect) match {
      case (Some(den), Some(fdp), dialect) =>
        val keyPath  = declarationPath(fdp, dialect)
        var fullPath = den.withKey(newName)
        val maybePart = params.bu.references
          .find(_.location().contains(params.uri))
          .getOrElse(params.bu)
          .objWithAST
          .flatMap(_.annotations.ast())
        val entries = getExistingParts(maybePart, keyPath)
        keyPath
          .dropRight(entries.size)
          .foreach(k => fullPath = fullPath.withKey(k))
        Some(fullPath, entries.lastOption)
      case _ => None
    }

  protected def declarationPath(fdp: AmfObject, dialect: Dialect): Seq[String] =
    Seq(fdp.declarableKey(dialect), declarationsPath).flatten

  /**
    * Secuential list for each node in the AST that already exists for the destiny
    *
    * @param maybePart
    * @param keys
    * @return
    */
  private def getExistingParts(maybePart: Option[YPart], keys: Seq[String]): Seq[YMapEntry] =
    maybePart match {
      case Some(n: YMap) => getExistingParts(YNode(n), keys.reverse, Seq.empty)
      case Some(d: YDocument) =>
        getExistingParts(d.node, keys.reverse, Seq.empty)
      case _ => Seq.empty
    }

  @tailrec
  private def getExistingParts(node: YNode, keys: Seq[String], acc: Seq[YMapEntry] = Seq.empty): Seq[YMapEntry] =
    keys match {
      case head :: _ =>
        node.value match {
          case m: YMap =>
            val maybeEntry = m.entries
              .find(_.key.asScalar.exists(_.text == head))
            maybeEntry match { // with match instead of map for tailrec optimization
              case Some(v) => getExistingParts(v.value, keys.tail, acc :+ v)
              case None    => acc
            }
          case _ => acc
        }
      case _ => acc
    }

  /**
    * Render for the new declaration, and the top entry on which it should be nested
    */
  protected lazy val declaredEntry: Option[(String, Option[YMapEntry])] = {
    val maybeParent: Option[YMapEntry] = wrappedDeclaredEntry.flatMap(_._2)
    wrappedDeclaredEntry
      .map(_._1)
      .flatMap { node =>
        params.dialect match {
          case _ if isJson(params.bu) =>
            Some(JsonRender.render(node, getIndentation(Mimes.`APPLICATION/JSON`, maybeParent)))
          case _ =>
            Some(YamlRender.render(node, getIndentation(Mimes.`APPLICATION/YAML`, maybeParent)))
        }
      }
      .map((_, maybeParent))
  }

  /**
    * Current indentation
    * @param mime
    * @param maybeParent
    * @return
    */
  private def getIndentation(mime: String, maybeParent: Option[YMapEntry]): Int =
    maybeParent
      .map(
        _.key.range.columnFrom + params.configuration
          .getFormatOptionForMime(mime)
          .indentationSize)
      .getOrElse(0)
}
