package org.mulesoft.als.common.finder.jsonschema

import amf.core.client.scala.model.domain.extensions.PropertyShape
import amf.core.client.scala.model.domain.{AmfElement, AmfObject, ScalarNode, Shape}
import amf.core.internal.parser.domain.FieldEntry
import amf.shapes.client.scala.model.domain.NodeShape
import org.mulesoft.als.common.ASTPartBranch
import org.mulesoft.als.common.dtoTypes.Position
import org.mulesoft.amfintegration.AmfImplicits.{AmfAnnotationsImp, ShapeImp}
import org.mulesoft.common.client.lexical.ASTElement
import org.yaml.model.{YMap, YScalar}

import scala.annotation.tailrec
import scala.util.matching.Regex

object JsonSchemaDefinitionGuesser {

  // based on the stack, the most specific definition we can find, and the position, we will try and guess possible
  // candidates for where we are. This is used because when getting to a `oneOf` or `anyOf`, AMF stops providing definitions.
  // in the future we might encounter more complex cases with arrays and such, but for now I'm only tackling the specific
  // cases we are encountering for agent-networks. Ideally we shouldn't be far off the last known definition.
  // If this gets highly nested or more complex we should try to enhance by applying memoization and/or eliminating
  // candidates earlier (for example excluding elements which contain non-allowed facets)
  def guessingGame(stack: Seq[AmfObject], position: Position, astPartBranch: ASTPartBranch): Seq[Shape] = {
    stack.find(_.annotations.sourceSchemaDef().isDefined)  match {
      case Some(obj) =>
        val definition = obj.annotations.sourceSchemaDef().get.definition // can do `get` because I checked `isDefined` above
        findInner(obj, definition, position, astPartBranch).map {
          case propertyShape: PropertyShape =>
            propertyShape.range
          case candidate =>
            candidate
        }
      case None => Seq.empty
    }
  }

  // iterate on par both the fields of the obj and the shapes of the definition in hopes to find the most specific one
  private def findInnerObject(obj: AmfObject, definition: Shape, position: Position, astPartBranch: ASTPartBranch, candidates: Seq[Shape] = Seq.empty): Seq[Shape] = {
    val maybeEntry = obj.fields.fields().find(_.value.annotations.astElement().exists(contains(_, astPartBranch)))
    val shapes = maybeEntry
      .toSeq
      .flatMap(findViableDefinitions(_, definition))
    if(shapes.isEmpty || maybeEntry.isEmpty)
      findBestSelection(candidates, obj, astPartBranch.siblingKeys)
    else
      shapes.flatMap(findInner(maybeEntry.get.value.value, _, position, astPartBranch, shapes))
  }

  private def findBestSelection(candidates: Seq[Shape], obj: AmfObject, siblingKeys: Set[String]): Seq[Shape] =
    candidates.flatMap(findMatches(_, siblingKeys, obj))

  private def contains(element: ASTElement, astPartBranch: ASTPartBranch) =
    (astPartBranch.node +: astPartBranch.stack).contains(element)

  // todo: could use memoization for performance, but mostly makes sense if using in other features (such as hover),
  //    in that case we should move this logic to a common place such as the document definition object
  private def findInner(element: AmfElement, definition: Shape, position: Position, astPartBranch: ASTPartBranch, candidates: Seq[Shape] = Seq.empty): Seq[Shape] = {
    element match {
      case amfObject: AmfObject =>
        findInnerObject(amfObject, definition, position, astPartBranch, candidates)
      case _ =>
        candidates
    }
  }

  private def findViableDefinitions(entry: FieldEntry, definition: Shape): Seq[Shape] = {
    definition match {
      case shape: PropertyShape =>
        findViableDefinitions(entry, shape.range)
      case shape: NodeShape if shape.properties.exists(matchesPatternName(_, entry.field.value.name)) =>
        shape.properties.find(matchesPatternName(_, entry.field.value.name)).toSeq
      case shape: NodeShape if shape.properties.flatMap(_.name.option()).contains(entry.field.value.name) =>
        shape.properties.find(p => p.name.value() == entry.field.value.name).toSeq
      case shape: NodeShape if Option(shape.additionalPropertiesSchema).isDefined =>
        Option(shape.additionalPropertiesSchema).toSeq
      case shape =>
        shape.mergedAndOrXone.flatMap(findViableDefinitions(entry, _))
    }
  }

  private def matchesPatternName(shape: PropertyShape, name: String): Boolean =
    shape.patternName.option().exists(matches(_, name))

  private def matches(pattern: String, name: String): Boolean = {
    // the key is a name (regex) and we are inside
    val regex = new Regex(pattern)
    regex.pattern.matcher(name).matches()
  }

  // this is currently only used on the last check of the node, if there were more complex structures we could go checking in every step of the way
  @tailrec
  private def findMatches(definition: Shape, siblings: Set[String], obj: AmfObject): Seq[Shape] =
    definition match {
      case shape: PropertyShape => findMatches(shape.range, siblings, obj)
      case shape if shape.mergedAndOrXone.nonEmpty =>
        shape.mergedAndOrXone.filter{
          case node: NodeShape =>
            containsSiblings(siblings, node) && matchesEnums(node, obj)
          case _ => false
        }
      case _ => Seq(definition)
    }

  private def matchesEnums(node: NodeShape, obj: AmfObject) =
    node.properties.forall { ps =>
      val enumValues = ps.range.values.collect {
        case scalar: ScalarNode if !scalar.value.isNullOrEmpty =>
          scalar.value.value()
      }
      enumValues.isEmpty || ps.name.option().forall(matchesNode(obj, _, enumValues))
    }

  private def matchesNode(obj: AmfObject, entry: String, enumValues: Seq[String]) =
    obj.annotations.astElement().exists {
      case part: YMap =>
        part.entries.find(_.key.value match {
          case scalar: YScalar => scalar.text == entry
          case _ => false
        }).forall{ entry =>
          entry.value.value match {
            case scalar: YScalar => enumValues.contains(scalar.text)
            case _ => false
          }
        }
      case _ => false
    }

  private def containsSiblings(siblings: Set[String], node: NodeShape) =
    siblings.subsetOf(node.properties.flatMap(_.name.option()).toSet)
}
