package org.mulesoft.amfintegration.visitors.agentnetwork

import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.AmfElement
import amf.core.internal.parser.domain.FieldEntry
import amf.core.internal.remote.Spec
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.amfintegration.AmfImplicits.AmfAnnotationsImp
import org.mulesoft.amfintegration.relationships.RelationshipLink
import org.mulesoft.amfintegration.visitors.{AmfElementVisitor, AmfElementVisitorFactory}
import org.mulesoft.amfintegration.vocabularies.jsonschema.agentnetwork.AgentNetworkVocabulary
import org.yaml.model.{YMap, YMapEntry, YPart, YScalar}

object AgentNetworkRelationshipLinks {
  private case class Declaration(kind: String, name: String, yMapEntry: YMapEntry)
  private case class Reference(kind: String, name: String, yScalar: YScalar)

  private def toDeclaration(jsonLDObject: JsonLDObject): Seq[Declaration] =(
    for {
      kind <- jsonLDObject.typeIris.find(_.startsWith(AgentNetworkVocabulary.KIND_BASE))
      yPart <- jsonLDObject.annotations.yPart()
    } yield
      getDeclaredNames(jsonLDObject).flatMap(name => findEntry(yPart, name).map(Declaration(kind, name, _)))
    ).getOrElse(Seq.empty)

  private def toReference(jsonLDObject: JsonLDObject): Option[Reference] =
    for {
      kind <- jsonLDObject.typeIris.find(_.startsWith(AgentNetworkVocabulary.KIND_BASE))
      yMap <- jsonLDObject.annotations.yPart().collectFirst {case map: YMap => map}
      entry <- yMap.entries.find(_.key.asScalar.map(_.text).contains("name"))
      name <- entry.value.asScalar
    } yield
      Reference(kind, name.text, name)

  private def getDeclaredNames(jsonLDObject: JsonLDObject): Seq[String] =
    jsonLDObject.fields.fields().map {
      case FieldEntry(field, _) => field.value.name
    }.toSeq

  private def findEntry(yPart: YPart, name: String): Option[YMapEntry] = yPart match {
    case map: YMap => map.entries.find(_.key.asScalar.map(_.text).contains(name))
    case _ => None
  }


  def collectAgentNetworkVisitors(allVisitors: Seq[AmfElementVisitor[_]]): Seq[RelationshipLink] = (
    for {
      declarations <- allVisitors.collectFirst {case v: AgentNetworkDeclarationsVisitor => v}
        .map(_.report)
        .map(_.flatMap(toDeclaration))
      references <- allVisitors.collectFirst{case v: AgentNetworkReferencesVisitor => v}.map(_.report).map(_.flatMap(toReference))
    } yield {
      references
        .flatMap{reference => findDeclarationFor(reference, declarations).map(declaration => (declaration, reference))}
        .map{case (target, source) => toRelationship(target, source)}
    }).getOrElse(Seq.empty)

  private def toRelationship(target: Declaration, source: Reference): RelationshipLink =
    RelationshipLink(sourceEntry = source.yScalar, targetEntry = target.yMapEntry, targetName = target.yMapEntry.key.asScalar, sourceName = Some(source.yScalar))

  private def findDeclarationFor(reference: Reference, declarations: List[Declaration]): Option[Declaration] =
    declarations.find(declaration => declaration.kind == reference.kind && declaration.name == reference.name)
}

class AgentNetworkReferencesVisitor extends AmfElementVisitor[JsonLDObject] {
  override protected def innerVisit(element: AmfElement): Seq[JsonLDObject] =
    element match {
      case element: JsonLDObject if element.typeIris.contains(AgentNetworkVocabulary.REFERENCE) && element.typeIris.exists(_.startsWith(AgentNetworkVocabulary.KIND_BASE)) =>
        Seq(element)
      case _ => Seq.empty
    }
}

object AgentNetworkReferencesVisitor extends AmfElementVisitorFactory {
  override def apply(bu: BaseUnit): Option[AmfElementVisitor[_]] = if(bu.sourceSpec.contains(Spec.AGENT_NETWORK)) Some(new AgentNetworkReferencesVisitor()) else None
}

class AgentNetworkDeclarationsVisitor extends AmfElementVisitor[JsonLDObject] {
  override protected def innerVisit(element: AmfElement): Seq[JsonLDObject] =
    element match {
      case element: JsonLDObject if element.typeIris.contains(AgentNetworkVocabulary.DECLARATIONS) && element.typeIris.exists(_.startsWith(AgentNetworkVocabulary.KIND_BASE)) =>
        Seq(element)
      // todo: add `connections` case, where the `kind` is set separately
      case _ => Seq.empty
    }
}

object AgentNetworkDeclarationsVisitor extends AmfElementVisitorFactory {
  override def apply(bu: BaseUnit): Option[AmfElementVisitor[_]] = if(bu.sourceSpec.contains(Spec.AGENT_NETWORK)) Some(new AgentNetworkDeclarationsVisitor()) else None
}
