package org.mulesoft.als.common.finder.jsonschema

import amf.core.client.scala.model.domain.extensions.PropertyShape
import amf.core.client.scala.model.domain.{AmfElement, AmfObject, Shape}
import amf.core.internal.parser.domain.FieldEntry
import amf.shapes.client.scala.model.domain.{AnyShape, NodeShape}
import org.mulesoft.als.common.dtoTypes.{Position, PositionRange}
import org.mulesoft.amfintegration.AmfImplicits.{AmfAnnotationsImp, ShapeImp}

object JsonSchemaDefinitionGuesser {

  // based on the stack, the most specific definition we can find, and the position, we will try and guess possible
  // candidates for where we are. This is used because when getting to a `oneOf` or `anyOf`, AMF stops providing definitions.
  // in the future we might encounter more complex cases with arrays and such, but for now I'm only tackling the specific
  // cases we are encountering for agent-domains. Ideally we shouldn't be far off the last known definition.
  // If this gets highly nested or more complex we should try to enhance by applying memoization and/or eliminating
  // candidates earlier (for example excluding elements which contain non-allowed facets)
  def guessingGame(stack: Seq[AmfObject], position: Position): Seq[Shape] = {
    stack.find(_.annotations.sourceSchemaDef().isDefined)  match {
      case Some(obj) =>
        val definition = obj.annotations.sourceSchemaDef().get.definition // can do `get` because I checked `isDefined` above
        findInner(obj, definition, position).map {
          case propertyShape: PropertyShape => propertyShape.range
          case candidate => candidate
        }
      case None => Seq.empty
    }
  }

  // iterate on par both the fields of the obj and the shapes of the definition in hopes to find the most specific one
  private def findInnerObject(obj: AmfObject, definition: Shape, position: Position, candidates: Seq[Shape] = Seq.empty): Seq[Shape] = {
    val maybeEntry = obj.fields.fields().find(_.value.annotations.range().map(PositionRange(_)).exists(_.contains(position)))
    val shapes = maybeEntry
      .toSeq
      .flatMap(findViableDefinitions(_, definition))
    if(shapes.isEmpty || maybeEntry.isEmpty)
      candidates
    else
      shapes.flatMap(findInner(maybeEntry.get.value.value, _, position, shapes))
  }

  // todo: could use memoization for performance, but mostly makes sense if using in other features (such as hover),
  //    in that case we should move this logic to a common place such as the document definition object
  private def findInner(element: AmfElement, definition: Shape, position: Position, candidates: Seq[Shape] = Seq.empty): Seq[Shape] = {
    element match {
      case amfObject: AmfObject =>
        findInnerObject(amfObject, definition, position, candidates)
      case _ =>
        candidates
    }
  }

  private def findViableDefinitions(entry: FieldEntry, definition: Shape): Seq[Shape] = {
    definition match {
      case shape: PropertyShape => findViableDefinitions(entry, shape.range)
      case shape: NodeShape if shape.properties.flatMap(_.name.option()).contains(entry.field.value.name) =>
        shape.properties.find(p => p.name.value() == entry.field.value.name).toSeq
      case shape: NodeShape => Option(shape.additionalPropertiesSchema).toSeq
      case shape =>
        shape.mergedAndOrXone.flatMap(findViableDefinitions(entry, _))
      case _ => Seq.empty
    }
  }
}
