package org.mulesoft.amfintegration.amfconfiguration

import amf.aml.client.scala.model.document.Dialect
import amf.aml.client.scala.model.domain.{DocumentsModel, NodeMapping}
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.{AmfScalar, DomainElement}
import amf.core.client.scala.vocabulary.Namespace
import amf.core.internal.parser.domain.Value
import amf.shapes.client.scala.model.document.JsonSchemaDocument
import org.mulesoft.amfintegration.AmfImplicits.DialectImplicits

trait DocumentDefinition {
  def nameAndVersion(): String
  def name(): Option[String]
  def version(): Option[String]
  def isJsonStyle: Boolean
  val baseUnit: BaseUnit
}

// plenty of info "implemented" for json documents in case we need them to add more custom functionality to stuff like
//   hover, structure or something like that. If we decide to go another direction now that we can use the
//   `SourceSchemaDef` annotation, we might not need to implement most of this methods
object DocumentDefinitionImplicits {
  implicit class DocumentDefinitionImplicits(documentDefinition: DocumentDefinition) {
    def documents(): Option[DocumentsModel] = documentDefinition match {
      case DialectDocumentDefinition(baseUnit) => Option(baseUnit.documents())
      case _ => None
    }
    def declarationsMapTerms: Map[String, String] = documentDefinition match {
      case DialectDocumentDefinition(baseUnit) => baseUnit.declarationsMapTerms
      case _ => Map.empty
    }

    def findNodeMappingByTerm(term: String): Option[NodeMapping] = documentDefinition match {
      case DialectDocumentDefinition(baseUnit) => baseUnit.findNodeMappingByTerm(term)
      case _ => None
    }

    def findNodeMapping(mappingId: String): Option[NodeMapping] = documentDefinition match {
      case DialectDocumentDefinition(baseUnit) => baseUnit.findNodeMapping(mappingId)
      case _ => None
    }

    def declares: Seq[DomainElement] = documentDefinition match {
      case DialectDocumentDefinition(baseUnit) => baseUnit.declares
      case JsonSchemaDocumentDefinition(baseUnit) => baseUnit.declares
      case _ => Seq.empty
    }

    def termsForId: Map[String, String] = documentDefinition match {
      case DialectDocumentDefinition(baseUnit) => baseUnit.termsForId
      case _ => Map.empty
    }
  }
}

sealed case class DialectDocumentDefinition(override val baseUnit: Dialect) extends DocumentDefinition {
  override def nameAndVersion(): String = baseUnit.nameAndVersion()

  override def name(): Option[String] = baseUnit.name().option()

  override def version(): Option[String] = baseUnit.version().option()

  override def isJsonStyle: Boolean = baseUnit.isJsonStyle
}

sealed case class JsonSchemaDocumentDefinition(override val baseUnit: JsonSchemaDocument) extends DocumentDefinition {
  override def nameAndVersion(): String = baseUnit.schemaVersion.option().getOrElse(baseUnit.id)

  override def name(): Option[String] = baseUnit.encodes.fields.getValueAsOption((Namespace.Core + "name").iri())
    .collect({ case Value(scalar: AmfScalar, _) =>
      scalar
    }).map(_.toString())

  override def version(): Option[String] = baseUnit.schemaVersion.option()

  override def isJsonStyle: Boolean = true
}

object DocumentDefinition {
  def apply(document: Dialect): DocumentDefinition = DialectDocumentDefinition(document)
  def apply(document: JsonSchemaDocument): DocumentDefinition = JsonSchemaDocumentDefinition(document)
}