package org.mulesoft.als.common.finder

import amf.core.client.scala.model.domain.{AmfArray, AmfElement, AmfObject}
import amf.core.internal.annotations.LexicalInformation
import org.mulesoft.als.common.finder.StrictObjectStack.{Branch, getChildren, getRangeForObject}
import org.mulesoft.amfintegration.AmfImplicits.{AlsLexicalInformation, AmfAnnotationsImp}
import org.mulesoft.common.client.lexical.{Position => AmfPosition, PositionRange => AmfPositionRange}

import scala.collection.mutable

class StrictObjectStack private(val root: AmfElement, val position: AmfPosition, val branch: Branch) {

  val element: Option[AmfElement] = branch.headOption

  val stack: Seq[AmfElement] = branch.drop(1)

  val siblings: Seq[AmfElement] = stack.headOption match {
    case Some(parent) =>
      getChildren(parent)
        .filterNot(getRangeForObject(_).isEmpty)
        .sortWith(sortByRange)
    case None => Seq.empty
  }

  val (preSiblings: Seq[AmfElement], postSiblings: Seq[AmfElement]) = element match {
    case Some(el) if siblings.contains(el) =>
      val (pre, post) = siblings.splitAt(siblings.indexOf(el))
      (pre, post.filterNot(_ == el))
    case _ => (Seq.empty, Seq.empty)
  }

  private def sortByRange(a: AmfElement, b: AmfElement): Boolean = {
    (getRangeForObject(a), getRangeForObject(b)) match {
      case (Some(ra), Some(rb)) =>
        val startA = ra.start
        val startB = rb.start
        if (startA != startB) startA < startB
        else ra.end < rb.end
      case _ => true // should not be possible, filtered before
    }
  }
}

object StrictObjectStack {
  private type Branch = Seq[AmfElement]
  private type Tree = Seq[Branch]
  private def getRangeForObject(amfElement: AmfElement): Option[AmfPositionRange] = amfElement.annotations.range()
  private def rangeContains(amfPositionRange: AmfPositionRange, amfPosition: AmfPosition): Boolean =
    LexicalInformation(amfPositionRange).contains(amfPosition)
  private def getChildren(amfElement: AmfElement): Seq[AmfElement] = amfElement match {
    case AmfArray(values, _) => values
    case amfObject: AmfObject => amfObject.fields.fields().map(_.value.value).toSeq
    case _ => Nil
  }

  private def chooseMoreSpecific(l: Branch, r: Branch) = {
    def getRange(stack: Branch): Option[AmfPositionRange] =
      stack.headOption.flatMap(getRangeForObject)
    (getRange(l), getRange(r)) match {
      case (Some(lRange), Some(rRange)) if lRange != rRange && lRange.contains(rRange) => r
      case (Some(lRange), Some(rRange)) if lRange != rRange && rRange.contains(lRange) => l
      case (Some(_), Some(_)) if r.size > l.size => r
      case (Some(_), Some(_)) => l
      case (None, _) => r
      case _ => l
    }
  }

  def apply(amfElement: AmfElement, position: AmfPosition): StrictObjectStack = {
    val rootLocation = amfElement.location()
    val memo = mutable.Map.empty[AmfElement, Tree]
    def changedLocation(element: AmfElement): Boolean =
      !element.location().forall(elementLocation => rootLocation.forall(_ == elementLocation))
    def outsideRange(element: AmfElement) =
      getRangeForObject(element).exists(!rangeContains(_, position))
    def cleanRangeless(tree: Tree) =
      tree.map(_.dropWhile(getRangeForObject(_).isEmpty))
    def selectBranch(tree: Tree): Branch =
      cleanRangeless(tree).distinct.reduceOption(chooseMoreSpecific).getOrElse(Seq.empty)
    def getBranches(element: AmfElement, path: Branch): Tree =
      if (changedLocation(element)) Seq(path)
      else if (outsideRange(element)) Seq(path)
      else getChildren(element).flatMap(getTree(_, element +: path))
    def memoizeBranches(element: AmfElement, path: Branch): Tree = {
      val result =
        getBranches(element, path)
      memo(element) = result
      result
    }
    def getTree(element: AmfElement, path: Branch): Tree =
      if (memo.contains(element)) memo(element)
      else memoizeBranches(element, path)

    val branch = selectBranch(getTree(amfElement, Seq.empty))
    new StrictObjectStack(amfElement, position, branch)
  }
}
