package org.mulesoft.als.server.modules.actions

import org.mulesoft.als.actions.definition.FindDefinition
import org.mulesoft.als.common.dtoTypes.Position
import org.mulesoft.als.convert.LspRangeConverter
import org.mulesoft.als.server.RequestModule
import org.mulesoft.als.server.modules.actions.links.{LinkKnowledge, UnitWithLinks}
import org.mulesoft.als.server.workspace.WorkspaceManager
import org.mulesoft.amfintegration.relationships.LinkTypes
import org.mulesoft.lsp.ConfigType
import org.mulesoft.lsp.configuration.WorkDoneProgressOptions
import org.mulesoft.lsp.feature.TelemeteredRequestHandler
import org.mulesoft.lsp.feature.common.{Location, LocationLink}
import org.mulesoft.lsp.feature.telemetry.MessageTypes
import org.mulesoft.lsp.feature.telemetry.MessageTypes.MessageTypes
import org.mulesoft.lsp.feature.typedefinition.{TypeDefinitionClientCapabilities, TypeDefinitionConfigType, TypeDefinitionParams, TypeDefinitionRequestType}

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class GoToTypeDefinitionManager(
    val workspace: WorkspaceManager
) extends RequestModule[TypeDefinitionClientCapabilities, Either[Boolean, WorkDoneProgressOptions]] {

  private var conf: Option[TypeDefinitionClientCapabilities] = None

  override val `type`: ConfigType[TypeDefinitionClientCapabilities, Either[Boolean, WorkDoneProgressOptions]] =
    TypeDefinitionConfigType

  override val getRequestHandlers: Seq[TelemeteredRequestHandler[_, _]] = Seq(
    new TelemeteredRequestHandler[TypeDefinitionParams, Either[Seq[Location], Seq[LocationLink]]] {
      override def `type`: TypeDefinitionRequestType.type =
        TypeDefinitionRequestType

      override def task(params: TypeDefinitionParams): Future[Either[Seq[Location], Seq[LocationLink]]] =
        goToTypeDefinition(params.textDocument.uri, LspRangeConverter.toPosition(params.position), uuid(params))

      override protected def code(params: TypeDefinitionParams): String =
        "GotoTypeDefinitionManager"

      override protected def beginType(params: TypeDefinitionParams): MessageTypes =
        MessageTypes.BEGIN_GOTO_T_DEF

      override protected def endType(params: TypeDefinitionParams): MessageTypes =
        MessageTypes.END_GOTO_T_DEF

      override protected def msg(params: TypeDefinitionParams): String =
        s"request for go to type definition on ${params.textDocument.uri}"

      override protected def uri(params: TypeDefinitionParams): String =
        params.textDocument.uri

      /** If Some(_), this will be sent as a response as a default for a managed exception
        */
      override protected val empty: Option[Either[Seq[Location], Seq[LocationLink]]] = Some(Right(Seq()))
    }
  )

  override def applyConfig(
      config: Option[TypeDefinitionClientCapabilities]
  ): Either[Boolean, WorkDoneProgressOptions] = {
    conf = config
    Left(true)
  }

  private def goToTypeDefinition(
      uri: String,
      position: Position,
      uuid: String
  ): Future[Either[Seq[Location], Seq[LocationLink]]] =
    for {
      UnitWithLinks(cu, aliases, relationships) <- LinkKnowledge.getUnitWithLinks(workspace, uri, uuid)
    } yield
        Right(FindDefinition
          .getDefinition(
            uri,
            position,
            relationships.filter(_.linkType == LinkTypes.TRAITRESOURCES),
            aliases,
            cu.astPartBranch
          ))

  override def initialize(): Future[Unit] = Future.successful()
}
