package org.mulesoft.als.server.modules.actions.rename

import amf.core.internal.remote.Platform
import org.mulesoft.als.actions.definition.FindDefinition
import org.mulesoft.als.actions.rename.FindRenameLocations
import org.mulesoft.als.actions.renamefile.RenameFileAction
import org.mulesoft.als.common.dtoTypes.{Position, PositionRange}
import org.mulesoft.als.configuration.AlsConfigurationReader
import org.mulesoft.als.logger.Logger
import org.mulesoft.als.server.modules.actions.links.{LinkKnowledge, UnitWithLinks}
import org.mulesoft.als.server.modules.workspace.CompilableUnit
import org.mulesoft.als.server.workspace.WorkspaceManager
import org.mulesoft.amfintegration.relationships.{AliasInfo, AliasRelationships, RelationshipLink}
import org.mulesoft.lsp.edit.WorkspaceEdit
import org.mulesoft.lsp.feature.TelemeteredRequestHandler
import org.mulesoft.lsp.feature.common.TextDocumentIdentifier
import org.mulesoft.lsp.feature.link.DocumentLink
import org.mulesoft.lsp.feature.rename.{RenameParams, RenameRequestType}
import org.mulesoft.lsp.feature.telemetry.MessageTypes
import org.mulesoft.lsp.feature.telemetry.MessageTypes.MessageTypes

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class RenameHandler(
    workspace: WorkspaceManager,
    configurationReader: AlsConfigurationReader,
    platform: Platform
) extends TelemeteredRequestHandler[RenameParams, WorkspaceEdit]
    with RenameTools {
  override def `type`: RenameRequestType.type = RenameRequestType

  override def task(params: RenameParams): Future[WorkspaceEdit] =
    rename(
      params.textDocument.uri,
      Position(params.position.line, params.position.character),
      params.newName,
      uuid(params)
    )
  override protected def code(params: RenameParams): String = "RenameManager"

  override protected def beginType(params: RenameParams): MessageTypes = MessageTypes.BEGIN_RENAME

  override protected def endType(params: RenameParams): MessageTypes = MessageTypes.END_RENAME

  override protected def msg(params: RenameParams): String = s"Request for renaming on ${params.textDocument.uri}"

  override protected def uri(params: RenameParams): String = params.textDocument.uri

  def rename(uri: String, position: Position, newName: String, uuid: String): Future[WorkspaceEdit] = {
    for {
      UnitWithLinks(cu, aliases, relationships) <- LinkKnowledge.getUnitWithLinks(workspace, uri, uuid)
      links <- workspace.getDocumentLinks(uri, uuid)
      allLinks <- workspace.getAllDocumentLinks(uri, uuid)
    } yield {
      val isAliasDeclaration = AliasRelationships.isAliasDeclaration(aliases, position, cu.astPartBranch)
      if (isAliasDeclaration || isDeclarableKey(cu, position, uri, relationships)) {
        FindRenameLocations
          .changeDeclaredName(
            uri,
            position,
            newName,
            aliases,
            relationships,
            cu.astPartBranch,
            cu.unit
          ).toWorkspaceEdit(configurationReader.supportsDocumentChanges)
      } else if (renameThroughReferenceEnabled) {// enable when polished, add logic to prepare rename
        renameFromLink(
            position,
            newName,
            links,
            allLinks
          ) // if Some() then it's a link to a file
          .orElse(renameFromDefinition(
            uri,
            position,
            newName,
            cu,
            aliases,
            relationships
          ) // if Some() then it's a reference to a declaration
        ).getOrElse(WorkspaceEdit.empty) // if none of the above, return empty
      } else WorkspaceEdit.empty
    }
  }

  private def renameFromLink(
      position: Position,
      newName: String,
      links: Seq[DocumentLink],
      allLinks: Map[String, Seq[DocumentLink]]
  ): Option[WorkspaceEdit] = {
    if (configurationReader.supportsDocumentChanges) {
        Logger.debug("Got the following document links", "RenameFileActionManager", "rename")
        links.foreach { l =>
          Logger.debug(s"-> link: ${l.target}", "RenameFileActionManager", "rename")
        }
        links
          .find(l => PositionRange(l.range).contains(position))
          .map(l =>
            RenameFileAction.renameFileEdits(
              TextDocumentIdentifier(l.target),
              TextDocumentIdentifier(getUriWithNewName(l.target, newName)),
              allLinks,
              platform
            )
          )
          .map(_.toWorkspaceEdit(configurationReader.supportsDocumentChanges))
      }
    else None
  }

  // todo: re-check when moving paths is available
  private def getUriWithNewName(target: String, newName: String): String =
    s"${splitUriName(target)._1}${splitUriName(newName)._2}"

  private def splitUriName(target: String) =
    target.splitAt(target.lastIndexOf('/') + 1)

  private def renameFromDefinition(
      uri: String,
      position: Position,
      newName: String,
      bu: CompilableUnit,
      aliases: Seq[AliasInfo],
      relationships: Seq[RelationshipLink]
  ): Option[WorkspaceEdit] =
    FindDefinition
      .getDefinition(
        uri,
        position,
        relationships,
        aliases,
        bu.astPartBranch
      )
      .headOption
      .map { definition =>
        FindRenameLocations
          .changeDeclaredName(
            definition.targetUri,
            Position(definition.targetRange.start),
            newName,
            aliases,
            relationships,
            bu.astPartBranch,
            bu.unit
            ).toWorkspaceEdit(configurationReader.supportsDocumentChanges)
      }

  /** If Some(_), this will be sent as a response as a default for a managed exception
    */
  override protected val empty: Option[WorkspaceEdit] = Some(WorkspaceEdit(None, None))
}
