package org.mulesoft.als.server.modules.rename.bydirectory

import amf.core.client.scala.AMFGraphConfiguration
import amf.core.internal.remote.Hint
import org.mulesoft.als.common.dtoTypes.{Position => DtoPosition}
import org.mulesoft.als.common.{ByDirectoryTest, MarkerFinderTest}
import org.mulesoft.als.convert.LspRangeConverter
import org.mulesoft.als.server.MockDiagnosticClientNotifier
import org.mulesoft.als.server.client.scala.LanguageServerBuilder
import org.mulesoft.als.server.modules.WorkspaceManagerFactoryBuilder
import org.mulesoft.als.server.protocol.LanguageServer
import org.mulesoft.als.server.protocol.configuration.AlsInitializeParams
import org.mulesoft.common.io.{Fs, SyncFile}
import org.mulesoft.lsp.converter
import org.mulesoft.lsp.converter.SEither3
import org.mulesoft.lsp.edit.TextEdit
import org.mulesoft.lsp.feature.common.{Position, Range, TextDocumentIdentifier, TextDocumentItem}
import org.mulesoft.lsp.feature.rename._
import org.mulesoft.lsp.textsync.DidOpenTextDocumentParams
import org.scalatest.OptionValues.convertOptionToValuable
import org.scalatest.Succeeded
import upickle.default.{macroRW, write, ReadWriter => RW}

import scala.concurrent.{ExecutionContext, Future}

trait ServerRenameByDirectoryTest extends ByDirectoryTest  with MarkerFinderTest{
  override implicit val executionContext: ExecutionContext =
    ExecutionContext.Implicits.global

  def basePath: String

  def dir: SyncFile = Fs.syncFile(basePath)

  def origin: Hint

  s"Suggestions test for vendor $origin by directory" - {
    forDirectory(dir, "")
  }

  override def testFile(content: String, f: SyncFile, parent: String): Unit = {
    s"Suggest over ${f.name} at dir ${cleanDirectory(f)}" in {
      val expected = expectedFile(f)
      val expectedPrepare = expectedPrepareFile(f)
      val uri = toUrl(f)
      for {
        server <- initializedServer
        markerInfo <- platform.fetchContent(uri, AMFGraphConfiguration.predefined()).map{ content =>
          val fileContentsStr = content.stream.toString
          this.findMarker(fileContentsStr, "*")
        }
        _ <- openFile(server)(uri, markerInfo.content)
        prepareRename <- prepareRename(server, markerInfo.position, uri)
        rename <- doRename("TEST-RENAME", server, markerInfo.position, uri)
        tmpPrepare <- writeTemporaryFile(expectedPrepare)(
          writePrepareToString(prepareRename)
        )
        tmp <- writeTemporaryFile(expected)(
          writeDataToString(rename.changes)
        )
        rPrepare <- assertDifferences(tmpPrepare, expectedPrepare)
        r <- assertDifferences(tmp, expected)
      } yield {
        assert(rPrepare == Succeeded)
        assert(r == Succeeded)
      }
    }
  }

  private def initializedServer = {
    val server = buildServer()
    server
      .testInitialize(AlsInitializeParams.default)
      .map(_ => {
        server.initialized()
        server
      })
  }

  private def toUrl(f: SyncFile) =
    "file://" + f.path.replaceAllLiterally(platform.fs.separatorChar.toString, "/")

  def expectedFile(f: SyncFile): String =
    s"${f.parent}${platform.fs.separatorChar}expected${platform.fs.separatorChar}${f.name}.json"

  def expectedPrepareFile(f: SyncFile): String =
    s"${f.parent}${platform.fs.separatorChar}expected${platform.fs.separatorChar}${f.name}.prepare.json"

  private def cleanDirectory(f: SyncFile) =
    f.path.stripPrefix("als-server/shared/src/test/resources/").stripSuffix(f.name)


  def buildServer(): LanguageServer = {
    val factory =
      new WorkspaceManagerFactoryBuilder(new MockDiagnosticClientNotifier).buildWorkspaceManagerFactory()
    new LanguageServerBuilder(
      factory.documentManager,
      factory.workspaceManager,
      factory.configurationManager,
      factory.resolutionTaskManager
    )
      .addInitializable(factory.documentManager)
      .addRequestModule(factory.renameManager)
      .build()
  }

  def openFile(server: LanguageServer)(uri: String, text: String): Future[Unit] =
    server.textDocumentSyncConsumer.didOpen(DidOpenTextDocumentParams(TextDocumentItem(uri, "", 0, text)))

  private def prepareRename(server: LanguageServer, position: DtoPosition, filePath: String) = {
    val prepareRenameHandler = server.resolveHandler(PrepareRenameRequestType).value
    prepareRenameHandler(
      PrepareRenameParams(TextDocumentIdentifier(filePath), LspRangeConverter.toLspPosition(position))
    )
  }

  private def doRename(
                        newName: String,
                        server: LanguageServer,
                        position: DtoPosition,
                        filePath: String
                      ) = {
    val renameHandler = server.resolveHandler(RenameRequestType).value
    renameHandler(RenameParams(TextDocumentIdentifier(filePath), LspRangeConverter.toLspPosition(position), newName))
  }

  private def writeDataToString(changes: Option[Map[String, Seq[TextEdit]]]) =
    write[List[ChangesNode]](changes.get.toList.map{case (uri, edits) => ChangesNode.sharedToTransport(uri, edits)}, 2)

  private def writePrepareToString(prepareRename: Option[SEither3[Range, PrepareRenameResult, PrepareRenameDefaultBehavior]]) =
    prepareRename.get match {
      case converter.Left(value) =>  write[RangeNode](value)
    }

}


case class PositionNode(line: Int, character: Int)

object PositionNode {

  implicit def rw: RW[PositionNode] = macroRW

  implicit def sharedToTransport(from: Position): PositionNode = {
    PositionNode(from.line, from.character)
  }
}

case class RangeNode(start: PositionNode, end: PositionNode)

object RangeNode {

  implicit def rw: RW[RangeNode] = macroRW

  implicit def sharedToTransport(from: Range): RangeNode =
    RangeNode(PositionNode.sharedToTransport(from.start), PositionNode.sharedToTransport(from.end))
}

case class TextEditNode(range: RangeNode, newText: String)

object TextEditNode {

  implicit def rw: RW[TextEditNode] = macroRW

  implicit def sharedToTransport(from: TextEdit): TextEditNode =
    TextEditNode(RangeNode.sharedToTransport(from.range), from.newText)
}

case class ChangesNode(
                              uri: String,
                              changes: Seq[TextEditNode]
                            )
object ChangesNode {

  implicit def rw: RW[ChangesNode] = macroRW

  implicit def sharedToTransport(uri: String, changes: Seq[TextEdit]): ChangesNode =
    ChangesNode(uri, changes.map(TextEditNode.sharedToTransport))
}
