package org.mulesoft.als.suggestions.antlr.grpc

import amf.core.internal.annotations.LexicalInformation
import org.mulesoft.als.common.finder.StrictObjectStack
import org.mulesoft.amfintegration.AmfImplicits.{AlsLexicalInformation, AmfAnnotationsImp}
import org.mulesoft.antlrast.ast
import org.mulesoft.antlrast.ast.{ASTNode, Node, Terminal}
import org.mulesoft.common.client.lexical.{ASTElement, Position}

import scala.annotation.tailrec


object AstNodeKnowledge {
  /**
   * Left: matches a strict name
   * Right: matches between two names
   */
  private type NodeMatchingType = Either[String, (String, String)]
  def isInside(stack: StrictObjectStack, nodeStack: Seq[NodeMatchingType]): Boolean = {
    @tailrec
    def inner(nodes: Seq[ASTNode], position: Position, parentStack: Seq[NodeMatchingType]): Boolean =
      parentStack match {
        case head :: tail =>
          matches(nodes, head, position) match {
            case (true, Some(element)) =>
              element match {
                case Node(_, _, _, children) if tail.nonEmpty =>
                  inner(children, position, tail)
                case _ => tail.isEmpty
              }
            case (value, _) => value && tail.isEmpty
          }
        case _ => false
      }

    stack.element
      .flatMap(_.annotations.astNode())
      .exists(node => inner(Seq(node), stack.position, nodeStack))
  }

  def mostSpecificAstNode(node: ASTNode, position: Position): Option[ASTNode] = {
    @tailrec
    def innerFind(current: Node): ASTNode = {
      current.children.find(element => nodeContainsPosition(element, position)) match {
        case Some(node: Node) => innerFind(node)
        case Some(element) => element // terminal/error
        case _ => current
      }
    }

    node match {
      case current: Node if nodeContainsPosition(node, position) =>
        Some(innerFind(current))
      case _ if nodeContainsPosition(node, position)=> Some(node)
      case _ => None
    }
  }

  private def nodeContainsPosition(node: ASTNode, position: Position) =
    LexicalInformation(node.location.range).contains(position)

  /**
   * @return if `n` is a string, find an element with that name and return the node
   *         if `n` is a tuple, find both elements and return true if the position is between, return no node
   *         because there is no specific one
   */
  private def matches(nodes: Seq[ASTNode], n: NodeMatchingType, position: Position): (Boolean, Option[ASTNode]) =
    n match {
      case Left(name) => // matches a specific node
        val maybeNode = nodes.find(node => node.name == name && nodeContainsPosition(node, position))
        (maybeNode.nonEmpty, maybeNode)
      case Right((leftName, rightName)) => // is between two nodes
        val maybeBoolean = for {
          leftNode <- nodes.find(_.name == leftName).map(_.location.range)
          rightNode <- nodes.find(_.name == rightName).map(_.location.range)
        } yield {
          LexicalInformation(leftNode.end, rightNode.start).contains(position)
        }
        maybeBoolean.map(b => (b, None)).getOrElse((false, None))
    }
}
