package org.mulesoft.als.suggestions.plugins.antlr.grpc.snippets

import amf.apicontract.client.scala.model.domain.api.WebApi
import amf.core.client.scala.model.document.Document
import org.mulesoft.als.common.finder.StrictObjectStack
import org.mulesoft.als.suggestions.plugins.antlr.CompletionPluginParams
import org.mulesoft.als.suggestions.plugins.antlr.grpc.declarations.GRPCDeclarationsScopeUtils.getScope
import org.mulesoft.als.suggestions.plugins.antlr.grpc.snippets.ReusableSnippets.{enumStructureSnippet, messageStructureSnippet}
import org.mulesoft.als.suggestions.plugins.antlr.snippets.{SnippetPlugin, SnippetStructure}

object RootSnippets extends SnippetPlugin {
  override def getAll(params: CompletionPluginParams): Seq[SnippetStructure] =
    if(!isAtRoot(params.stack)) Seq.empty
    else
      allRootSnippets(params, getFields(params)) ++ maybeSyntaxSnippet(params)

  private def getFields(params: CompletionPluginParams) =
    getScope(params).map(_.value()).map(params.declarations.getDeclaredFor).getOrElse(Seq.empty)

  private def maybeSyntaxSnippet(params: CompletionPluginParams): Seq[SnippetStructure] =
    if(hasDefinedSyntax(params)) Seq.empty else Seq(syntaxSnippet)


  // todo: ask AMF if we can have a field, token or lexical for this, so we don't need to check if the document starts with the keyword
  private def hasDefinedSyntax(params: CompletionPluginParams) =
    params.stack.root match {
      case document: Document =>
        document.raw match {
          case Some(raw) => raw.startsWith("syntax")
          case None => false
        }
      case _ => false
    }

  private def allRootSnippets(params: CompletionPluginParams, fields: Seq[String]): Seq[SnippetStructure] = {
    Seq(
      SnippetStructure(
        "import",
        "import ${1|weak,public|} \"$2\";\n$0",
        Some("Imports another proto file. Qualifier can be omitted, 'weak', or 'public'.")
      ),
      SnippetStructure(
        "option",
        "option ${1:OPTION_NAME} = ${2:OPTION};\n$0",
        Some("Sets a file-level option.")
      ),
      SnippetStructure( // todo: add names to variables
        "extend",
        """extend ${1:NAME} {
          |  optional $2 $3 = $4;
          |}$0""".stripMargin,
        Some("Defines extend from command in a proto file.")
      ),
      // todo: why are there two `service`? which are different from each other
      SnippetStructure(
        "service",
        // todo: instead of completing with the rpc in the body, put the `service` block empty and create a snippet for the body
        // consider `options` and also inside the `rpc` snippet consider `stream` optional keyword (and probably should filter for only declared `Message`s
        s"service $${1:SERVICE_NAME} {\n  rpc $${2:METHOD_NAME} ($${3${makeFieldsEnum(fields)}}) returns ($${4${makeFieldsEnum(fields)}});\n$$0\n}",
        Some("Defines a gRPC service with an example rpc method.")
      ),
      SnippetStructure(
        "package",
        "package ${1:PACKAGE_NAME};\n$0",
        Some("Declares the package for the proto file.")
      ),
      messageStructureSnippet,
      enumStructureSnippet,
    )
  }

  private def makeFieldsEnum(fields: Seq[String]): String =
    if(fields.isEmpty) ":FIELD_NAME"
    else s"|${fields.mkString(",")}|"

  private val syntaxSnippet: SnippetStructure = SnippetStructure(
    "syntax",
    "syntax = \"proto3\";\n$0",
    Some("Defines a protobuf 3 document.")
  )

  private def isAtRoot(stack: StrictObjectStack): Boolean =
    stack.element.forall(_.isInstanceOf[WebApi]) &&
      stack.position.column == 0 // todo: check for `prefix` when in the middle of a sentence?
}