package org.mulesoft.als.suggestions.jsonschema.declarations

import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.model.domain.{AmfElement, AmfObject}
import amf.core.client.scala.traversal.iterator.AmfElementIterator
import amf.core.internal.remote.Spec
import amf.shapes.client.scala.model.document.JsonLDInstanceDocument
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.als.suggestions.jsonschema.declarations.InferredDeclarationProvider.DeclarationType
import org.mulesoft.als.suggestions.jsonschema.declarations.agentnetwork.AgentNetworkDeclarationProviderFactory
import org.mulesoft.amfintegration.visitors.Visitor

// this is used to extract interpreted declarations from documents that don't have AMF conventional declarations,
//   such as json schema definition instances
trait InferredDeclarationProviderFactory {
  protected val types: Seq[(DeclarationType, AmfObject => String)]
  final def apply(baseUnit: BaseUnit): InferredDeclarationProvider = {
    def applyVisitors(elements: Seq[AmfElement], allVisitors: Seq[TypeVisitor]): Unit = {
      val iterator = AmfElementIterator(elements.toList)
      while (iterator.hasNext) {
        val element = iterator.next()
        allVisitors.foreach(_.visit(element))
      }
    }

    baseUnit match {
      case bu: JsonLDInstanceDocument =>
        val visitors = types.map(t => new TypeVisitor(t._1, t._2))
        applyVisitors(bu.encodes, visitors)
        new InferredDeclarationProvider(
          visitors.map{ tv =>
            tv.declarationType -> tv.report
          }.flatMap(_._2)
            .groupBy(_._1)
            .map(t => t._1 -> t._2.map(_._2))
        )
      case _ =>
        InferredDeclarationProvider.empty
    }
  }

  private class TypeVisitor(val declarationType: DeclarationType, getType: AmfObject => String) extends Visitor[AmfElement, (String, AmfElement)] {
    override protected def innerVisit(element: AmfElement): Seq[(String, AmfElement)] =
      element match {
        case e: JsonLDObject if e.typeIris.exists(_.contains(declarationType)) =>
          Seq((getType(e), element))
        case _ => Seq.empty
      }
  }
}

class InferredDeclarationProvider(private val declarations: Map[DeclarationType, Seq[AmfElement]]) {
  final def getDeclarationFor(declarationType: DeclarationType): Seq[AmfElement] = declarations.getOrElse(declarationType, Seq.empty)
}

object InferredDeclarationProvider {
  type DeclarationType = String
  val empty = new InferredDeclarationProvider(Map.empty)

  def apply(baseUnit: BaseUnit): InferredDeclarationProvider = {
    baseUnit.sourceSpec match {
      case Some(Spec.AGENT_NETWORK) =>
        AgentNetworkDeclarationProviderFactory(baseUnit)
      case _ => empty
    }
  }
}
