package org.mulesoft.als.suggestions.plugins.jsonschema

import amf.core.client.scala.model.domain.Shape
import amf.core.client.scala.model.domain.extensions.PropertyShape
import amf.core.client.scala.model.{DataType, StrField}
import amf.core.client.scala.vocabulary.Namespace.Core
import amf.core.internal.parser.domain.FieldEntry
import amf.shapes.client.scala.model.domain.{ArrayShape, NodeShape, ScalarShape}
import org.mulesoft.als.suggestions._
import org.mulesoft.als.suggestions.aml.AmlCompletionRequest
import org.mulesoft.amfintegration.AmfImplicits.{AmfAnnotationsImp, ShapeImp}
import org.mulesoft.amfintegration.amfconfiguration.executioncontext.Implicits.global
import org.mulesoft.lexer.AstToken
import org.yaml.lexer.YamlToken.Indent
import org.yaml.model.YNonContent

import scala.annotation.tailrec
import scala.concurrent.Future

object JsonSchemaStructureCompletionPlugin extends JsonSchemaBaseCompletionPlugin {
  private val IGNORED_FIELD = s"${Core.base}${Core.base.substring(0, Core.base.length-1)}" // check why this comes from AMF when we have an empty field

  override def id: String = "JsonSchemaStructureCompletionPlugin"

  override def innerResolve(request: AmlCompletionRequest): Future[Seq[RawSuggestion]] =
    if(!applies(request)) emptySuggestion
    else Future {
      getSourceDef(request)
        .flatMap(extractSuggestions(_, isJson(request)))
    }

  private def isJson(request: AmlCompletionRequest) =
    request.baseUnit.location().forall(_.endsWith(".json"))

  private def getSourceDef(request: AmlCompletionRequest): Seq[Shape] = {
    val defaultValue = request.amfObject.annotations.sourceSchemaDef().map(_.definition)
    request.strictObjectStack.element match {
      case _ if isFieldEntryEmpty(request) && correctlyNested(request) && defaultValue.nonEmpty =>
        defaultValue.toSeq
      case _ =>
        request.guessedShapes.collect{
          case array: ArrayShape => array.items
          case o => o
        }
    }
  }

  // maps or sequences which are still being built may match with the lexical of the closest valid shape,
  // which incorrectly suggests parents facets
  private def correctlyNested(request: AmlCompletionRequest) = isJson(request) ||
    (request.amfObject.annotations.yPart() match {
    case Some(part) =>
      val maybeLocation = part.children.collect {
        case nonContent: YNonContent => nonContent.tokens
      }.flatMap(_.collectFirst {
        case AstToken(Indent, _, location) => location
      })
      maybeLocation.forall(location => request.position.toAmfPosition.column - request.prefix.length == location.columnTo)
    case _ => true
  })

  private def extractSuggestions(definition: Shape, isInflow: Boolean): Seq[RawSuggestion] = {
    definition match {
      case shape if shape.mergedAndOrXone.nonEmpty =>
        shape.mergedAndOrXone.flatMap(extractSuggestions(_, isInflow)).distinct
      case shape: NodeShape =>
        // todo: could add snippet here, but I think Styler currently does not support this
        // this cases are for the fields that are defined by a patternProperty in the schema (is not a token literal)
        val properties = shape.properties.filter(_.patternName.isNullOrEmpty)
        properties.flatMap(toRaw(_, isInflow)) ++
          properties.filter(containsMandatory).flatMap(buildTemplate(_, isInflow))
      case _ =>
        Nil // check if we need to support something here (arrays?) if not, return Nil
    }
  }

  private def containsMandatory(shape: PropertyShape) =
    shape.range match {
      case nodeShape: NodeShape => nodeShape.properties.exists(isMandatory)
      case arrayShape: ArrayShape => arrayShape.items.isInstanceOf[NodeShape]
      case _ => false
    }

  private def toRaw(property: PropertyShape, isInflow: Boolean, name: Option[String] = None): Option[RawSuggestion] = {
    getRangeKindForProperty(property) match {
      case ArrayRange =>
        property.name.option().map(RawSuggestion.forKey(_,
          "structure",
          isMandatory(property),
          rangeKind = ArrayRange,
          displayText = name,
          children = childFromArrayItem(property, isInflow)
        ))
      case ObjectRange =>
        property.name.option().map(RawSuggestion.forObject(_,
          "structure",
          isMandatory(property),
          displayText = name
        ))
      case kind =>
        property.name.option().map(RawSuggestion.forKey(_,
          "structure",
          isMandatory(property),
          rangeKind = kind,
          displayText = name
        ))
    }
  }

  private def itemsToRaw(property: Shape, isInflow: Boolean): Seq[RawSuggestion] = property match {
    case shape: NodeShape =>
      shape.properties.filter(isMandatory).flatMap(toRaw(_, isInflow))
    case _ => Seq.empty
  }

  private def childFromArrayItem(property: PropertyShape, isInflow: Boolean) = property.range match {
    case array: ArrayShape if isInflow && array.items.isInstanceOf[NodeShape] => // only for json-like, add the `{}` inside of the array
      Seq(RawSuggestion.emptyObject())
    case _ => Seq.empty
  }

  private def toRawTemplate(property: PropertyShape): Option[RawSuggestion] =
    property.name.option().map(v => RawSuggestion.forObject(v,
      "structure",
      isMandatory(property),
      displayText = Some(s"new $v"),
    ))

  private def buildTemplate(property: PropertyShape, isInflow: Boolean): Option[RawSuggestion] = {
    property.range match {
      case nodeShape: NodeShape =>
        val mandatoryProperties = nodeShape.properties.filter(isMandatory)
        val maybeSuggestion = toRawTemplate(property)
        maybeSuggestion.map(raw => raw.withChildren(mandatoryProperties.flatMap(buildTemplate(_, isInflow))))
      case arrayShape: ArrayShape =>
        toRaw(property, isInflow, property.name.option().map(name => s"new $name")).map(_.withChildren(itemsToRaw(arrayShape.items, isInflow)))
      case _ => toRaw(property, isInflow)
    }
  }

  @tailrec
  private def getRangeKindForProperty(shape: Shape): RangeKind = shape match {
    case propertyShape: PropertyShape => getRangeKindForProperty(propertyShape.range)
    case _: ArrayShape => ArrayRange
    case shape: ScalarShape => rangeFor(shape.dataType)
    case any if any.mergedAndOrXone.nonEmpty => getRangeKindForProperty(any.mergedAndOrXone.head)
    case _ => ObjectRange
  }

  private def rangeFor(dataType: StrField): RangeKind = dataType.option() match {
    case Some(value) if numberDataTypes.contains(value) => NumberScalarRange
    case Some(DataType.Boolean) => BoolScalarRange
    case Some(DataType.String) => StringScalarRange
    case _ => PlainText
  }


  private val numberDataTypes = Seq(DataType.Number, DataType.Long, DataType.Float, DataType.Double)

  private def isMandatory(shape: PropertyShape) =
    shape.minCount.option().exists(_ > 0)

  private def applies(request: AmlCompletionRequest): Boolean =
    request.astPartBranch.isKeyLike

  private def isFieldEntryEmpty(request: AmlCompletionRequest) =
    request.fieldEntry.forall(isIgnored)

  private def isIgnored(fe: FieldEntry) =
    fe.field.value.iri() == IGNORED_FIELD
}
